#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import argparse
import sys
import os
import time
import signal
import shutil
import MySQLdb

from subprocess import PIPE, Popen

MYSQL_SOCKET_PPCDICT = "/opt/root.ppcdict.1/run/mysqld.ppcdict/mysqld.sock"
MYSQL_SOCKET = "/var/run/mysqld.{0}/mysqld.sock"
FILE_XTRABACKUP_INFO = "/opt/mysql.{0}/data/xtrabackup_binlog_info"

USAGE = '''Программа для обновления gtid позиций в базах на основе файла xtrabackup_binlog_info.
Обновляются только gtid указанные в файле, другие позииции не трогаются. 
'''

#логирование осуществляется в STDOUT/STDERR
def startLogging(level='DEBUG'):
    logger = logging.getLogger('stream logs to console')
    logger.setLevel(level=getattr(logging, level))
    formatter = logging.Formatter('%(asctime)s %(message)s')
    logch = logging.StreamHandler()
    logch.setLevel(level=getattr(logging, level))
    logch.setFormatter(formatter)
    logger.addHandler(logch)
    return logger, logch

#Проверяем подключение к сокету mysqld. Возвращает стутус выполнения(true/false) и ошибку(error).
def mysqlConnect(instance, command):
    socket = MYSQL_SOCKET.format(instance) if instance.find('ppcdict') == -1 else MYSQL_SOCKET_PPCDICT
    logger.info("connect to socket {0}".format(socket))
    try:
        if not os.path.exists(socket):
            msg = "socket {0} dosnt exists".format(socket)
            return False, ValueError(msg)
        db = MySQLdb.connect(host="localhost", user="root", passwd="", db="mysql", unix_socket=socket)
        cur = db.cursor()
        cur.execute(command)
        data = cur.fetchall()
        cur.close()
        db.close()
        return data, None
    except Exception as err:
        return None, ValueError(err)

#Получить хэш с позициями инстансов из файла бекапов: {'ppcdata1': [gtid1:pos1, gtid2:pos2], 'ppcdata2': [gtid1:pos1, gtid2:pos2] }
def fileGtidInfo():
    result = dict()
    for instance in instances:
        result[instance] = list()
        xinfo = FILE_XTRABACKUP_INFO.format(instance)
        if os.path.exists(xinfo):
            logger.info("read xtrabackup file {0}".format(xinfo))
            line = open(xinfo, 'r').read()
            line = line.replace("\n", "").split("\t")
            if len(line) == 3:
                result[instance].extend(line[2].split(","))
    return result

#Получить хэш с позициями инстансов из mysql: {'ppcdata1': [gtid1:pos1, gtid2:pos2], 'ppcdata2': [gtid1:pos1, gtid2:pos2] }
def mysqlGtidInfo():
    result = dict()
    for instance in instances:
        uuid = mysqlConnect(instance, "SHOW VARIABLES LIKE 'server_uuid'")[0][0][-1]
        line = mysqlConnect(instance, "SHOW MASTER STATUS")[0][0]
        result[instance] = [ i for i in line[-1].replace("\n", "").split(",") if i.find(uuid) < 0 ]
    return result

#Принимает два хеша, в котором второй дополняет первый при условии, что gtid отсутствует в ней. Возвращает хеш вида:
# {'ppcdata1': [gtid1:pos1, gtid2:pos2], 'ppcdata2': [gtid1:pos1, gtid2:pos2] }
def mergeGtidInfo(mainGtids, secondGtids):
    result = dict()
    for instance in mainGtids:
        mains = mainGtids.get(instance, list())
        seconds = secondGtids.get(instance, list())
        str1 = "{0}".format(mains)
        for s in seconds:
            if str1.find(s.split(":")[0]) > -1:
                continue
            logger.info("add old gtid: {0}".format(s))
            mains.append(s)
        logger.info("instance {0} new position {1}".format(instance, mains))
        result[instance] = mains
    return result

#Принимает два хэша и выполняет reset master
def resetMasterPositions(newpositions):
    code = 0
    for instance in newpositions:
        if instance == "ppcdict":
            logger.info("ignore ppcdict pxc: RESET MASTER not allowed when node is in cluster")
            continue
        gtids = ','.join(newpositions.get(instance))
        cmd = "RESET MASTER"
        _, err = mysqlConnect(instance, cmd)
        if not err is None:
            logger.critical("mysql error {0}: {1}".format(cmd, err))
            code = 1 
        cmd = "SET GLOBAL GTID_PURGED='{0}'".format(gtids)
        _, err = mysqlConnect(instance, cmd)
        if not err is None:
            logger.critical("mysql error {0}: {1}".format(cmd, err))
            code = 1
    return code

def main():
    mainGtids = fileGtidInfo()
    secondsGtids = mysqlGtidInfo()
    mergedGtids = mergeGtidInfo(mainGtids, secondsGtids)
    if opts.doit:
        logger.info("start apply GTID positions")
        code = resetMasterPositions(mergedGtids)
        logger.info("finish apply GTID position")
        sys.exit(code)

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description=USAGE,
                    epilog='Пример: %(prog)s ppcdata8 ppcdata9')
    parser.add_argument("-d", "--debug", action="store_true",
                    dest='debug', help="включить debug режим")
    parser.add_argument("--doit", action="store_true",
                    dest='doit', help="применить gtid purged к базам")
    parser.add_argument("instances", nargs='*', type=str, action='store',
            help="список инстансов mysqld(например: ppcdata1, ppcdata2.new)")
    opts = parser.parse_args()

    if len(opts.instances) == 0:
        parser.print_help()
        sys.exit(1)

    log_level = "DEBUG" if opts.debug else "INFO"
    logger, logch = startLogging(level=log_level)
    instances = opts.instances
    main()

