#!/usr/bin/perl -w

=head1 NAME

=encoding utf8

sync-db.pl - перезаливка тестовых БД Директа

=head1 DESCRIPTION

    Скрипт для регулярной перезаливки тестовых БД Директа
    (переписанный на perl и снабженный настройками скрипт sync-db)

    Какие базы на какой машине надо обрабатывать читает из конфига /etc/sync-db/sync-db.conf

    --copy-db
        действие: привезти данные (в каталоги .new)

    --start-db
        действие: запустить mysql'и на новых данных

    --place-data
        действие: переместить привезенные данные на правильное место (каталоги без суффикса)
        mysql не останавливается и не стартуется
        ВНИМАНИЕ! Использовать с осторожностью, заботиться об остановке и старте mysql самостоятельно.

    -d, --date <str>
        для ключа --copy-db: за какую дату брать данные, (YYYYMMDD|today|yesterday)
        по умолчанию yesterday

    -db, --database ppcdata1
        базы данных, с котрыми нужно работать
        по умолчанию - все, указанные в конфиге
        можно указывать несколько раз

    --log-dir <dir>
        в каком каталоге писать логи
        по умолчанию /var/log/sync-db

    --log-file <file>
        префикс файла с логами
        по умолчанию sync-db.log

    --date-suf <datefmt>
        суффикс даты для логов, по умолчанию '%Y%m%d'

    --conf <file>
        из какого файла брать конфиг
        по умолчанию /etc/sync-db/sync-db.conf

    --verbose
        дублировать лог на STDOUT

    --rsync-opts <opt>=<val>
        переопределение умолчательных опций создания Fils::Rsync,
	например bwlimit=60000

    --rsync-exclude <pattern>
        исключить <pattern> при копировании, например,
        --rsync-exclude='*_arc' --rsync-exclude='*_bin' ...
        По умолчанию, '*-bin*' 'mysqld*' 'relay*' 'master*'

    --chown <group.user>
        кому должны принадлежать данные, по умолчанию mysql.mysql

    --base-data-dir <path>
        базовое имя для каталогов с данными, по умолчанию /opt/mysql
        база $db будет располагаться в каталоге <path>.$db
        т.е. по умолчанию /opt/mysql.ppcdata1, /opt/mysql.ppcdict и т.п.

    --no-remove-new-set
        не удалять недокачанные директории с базами, и не удалять
        (перемещать) предыдущий рабочий набор баз. Полезно, когда хочется
        докачать набор баз после флапа сети. По умолчанию удаляет недокачанные,
        перемещает старые рабочие базы (только для --copy-db)

    --remove-ready-set
        удалить готовый к запуску набор баз, и перекачать их заново,
        по умолчанию - ничего не удаляет и падает при наличии готовых баз
        (только для --copy-db)

    --no-save-old-set
        удалить текущий набор баз, и заменить его новым-готовым. Полезно,
        когда места мало и бекап рабочих баз не нужен. По умолчанию -
        сохраняет текущий набор с суффиксом .old (только для --start-db
        и --place-data)

    -h, --help
        показать справку


    Примеры

    скопировать на машину новые данные, mysql'и не перезапускать
    данные сначала rsync'ются в каталоги .new, потом все переименовываются в .ready
    каталоги .old удаляются (на самом деле перед rsyn'ом переименовываютс в .new)
    если уже есть готовые к использованию данные (каталоги .ready) -- ничего не делает
    # sync-db.pl --copy-db

    запустить mysql'и на новых данных
    старые данные переименовываются в .old
    если нет ни одного готового каталога (.ready) -- ничего не делает
    # sync-db.pl --start-db

    сначала --copy-db, потом --start-db
    # sync-db.pl --copy-db --start-db

    Определить имя машины для включения в конфиг:
    perl -MSys::Hostname -le 'print hostname()'

    $Id$

=cut

use strict;
use warnings;

use YAML;
use Data::Dumper;
use Getopt::Long;
use File::Rsync;
use Sys::Hostname;
use Pid::File::Flock;

use Yandex::Shell;
use Yandex::Log;
use Yandex::HashUtils;
$Yandex::Log::LOG_ROOT = '/var/log/sync-db';

use Yandex::TimeCommon;

use utf8;
use open ':std' => ':utf8';

=head1 METHODS

=cut

# откуда брать данные по умолчанию. Может переопределяться в конфиге
our $DEFAULT_DBDIR="rsync://ppcbackup.yandex.ru:/mysqlbackup";

our $DBHOME="/root";

our $CONFFILE = "/etc/sync-db/sync-db.conf";
our $LOGFILE = "sync-db.log";
our $DATESUF = '%Y%m%d';
our $VERBOSE = 0;

our $BASE_DATA_DIR = '/opt/mysql';


run() unless caller();


sub run
{
    my $opt = parse_options();

    # different locks for different configs
    my ($name) = $CONFFILE =~ m!([^/]+).conf$!;
    $name ||= 'default';
    Pid::File::Flock->new(name => "sync-db-$name");

    write_log("started with config $CONFFILE");
    write_log({options => $opt});

    if ( $opt->{copy_db} ) {
        copy_db(%$opt);
    }
    if ( $opt->{start_db} ){
        start_db_new(%$opt);
    } elsif ( $opt->{place_data} ){
        place_new_data(%$opt);
    }
}


sub parse_options
{
    my %O = (
        date => ['yesterday'],
        copy_db => 0,
        start_db => 0,
        place_data => 0,
        rsync_opts => {},
        chown => 'mysql.mysql',
        remove_new_set => 1,
        remove_ready_set => 0,
        save_old_set => 1,
    );
    my @db_filter;
    GetOptions (
        "h|help"             => \&usage,
        "d|date=s@"          => \$O{date},
        "db|database=s@"     => \@db_filter,
        "copy-db"            => \$O{copy_db},
        "start-db"           => \$O{start_db},
        "place-data"         => \$O{place_data},
        "verbose"            => \$VERBOSE,
        "log-dir=s"          => \$Yandex::Log::LOG_ROOT,
        "log-file=s"         => \$LOGFILE,
        "date-suf=s"         => \$DATESUF,
        "conf=s"             => \$CONFFILE,
        "rsync-opts=s"       => $O{rsync_opts},
        "chown=s"            => \$O{chown},
        "base-data-dir=s"    => \$BASE_DATA_DIR,
        "remove-new-set!"    => \$O{remove_new_set},
        "remove-ready-set!"  => \$O{remove_ready_set},
        "save-old-set!"      => \$O{save_old_set},
        "rsync-exclude=s@"   => \$O{rsync_exclude},
    ) or die $@;

    $O{rsync_exclude} = $O{rsync_exclude} // [
            '*-bin*',
            'mysqld*',
            'relay*',
            'master*',
    ];

    # даты приводим к числовому виду и сортируем от старых к новым
    my @dates;
    for my $d ( @{$O{date}} ){
        $d = today() if $d eq 'today';
        $d = yesterday() if $d eq 'yesterday';
        die "incorrect date $d" unless $d =~ /^\d{8}$/;
        push @dates, $d;
    }
    $O{date} = [ sort @dates ];

    die "incorrect owner $O{chown}" unless $O{chown} =~ /^[a-zA-Z_0-9\.\-]+$/;

    die "incompatible actions start-db and place-data" if $O{start_db} && $O{place_data};

    my $conf = YAML::LoadFile($CONFFILE);
    my $host = hostname();
    $O{conf_name} = exists $conf->{$host} ? $host : 'DEFAULT';
    write_log("host $host, conf $O{conf_name}");
    $O{DBDIR} = $conf->{$O{conf_name}}->{dbdir} || $conf->{DEFAULT}->{dbdir} || $DEFAULT_DBDIR;
    # prepare секция может быть пустой
    $O{prepare} = $conf->{$O{conf_name}}->{prepare} || $conf->{DEFAULT}->{prepare} || [];

    my %db_filter = map {$_ => 1} @db_filter;
    my $databases = $conf->{$O{conf_name}}->{databases} || $conf->{DEFAULT}->{databases};
    $O{databases} = [
        grep {!%db_filter || $db_filter{$_->{name}}}
        map { ref $_ eq 'HASH' ? $_ : {name => $_} }
        @{$databases || []}
        ];
    die "no databases listed in config, may be no config section for host: $host" if ! @{$O{databases}};

    $_->{dbdir} ||= $O{DBDIR} for @{$O{databases}};
    return \%O;
}


{
    my $log;
sub write_log
{
    $log ||= Yandex::Log->new(
        log_file_name => $LOGFILE,
        date_suf => $DATESUF,
        tee => $VERBOSE,
        msg_prefix => "[$$]",
    );
    $log->out(@_);
}
}

sub usage {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}

=head2 copy_db

    Результат: в каталогах .ready -- готовые данные
    Если что-то ломается посередине -- умирает

=cut

sub copy_db
{
    my %O = @_;
    if (dirs_by_type("ready", %O) > 0 && not $O{remove_ready_set}) {
        write_log("some databases synced, but not started yet");
        die;
    }

    # Убеждаемся, что есть все базы за указанную дату
    for my $db (@{$O{databases}}) {
        $db->{source} = find_source($db, %O);
    }
    rename_dirs(from => 'old', to => 'new', %O) if $O{remove_new_set};
    rename_dirs(from => 'ready', to => 'new') if $O{remove_ready_set};
    rsync_data(%O);
    rename_dirs(from => 'new', to => 'ready', %O);

    for my $db (@{$O{databases}}){
        unless (system("chown", "-R", $O{chown}, data_dir($db, 'ready')) == 0) {
            write_log("can't chown mysql datadir for $db");
            die;
        }
    }
}


=head2 dirs_by_type

=cut

sub dirs_by_type
{
    my ($type, %O) = @_;

    my @res = grep {-e data_dir($_, $type)} @{$O{databases}};

    return @res;
}


=head rename_dirs

=cut

sub rename_dirs
{
    my %O = @_;

    for my $db (@{$O{databases}}){
        my $from = data_dir($db, $O{from});
        my $to = data_dir($db, $O{to});

        if ( -e $to ){
            unless (system("rm -rf $to") == 0) {
                write_log("can't rm -rf $to");
                die;
            }
        }
        if ( -e $from ){
            unless (system("mv $from $to") == 0) {
                write_log("can't mv $from $to");
                die;
            }
        }
    }

    return;
}


=head2

=cut

sub data_dir
{
    my ($db, $type) = @_;

    my $dbname = $db->{name};
    my $base_db_data_dir = $db->{datadir} || "$BASE_DATA_DIR.$dbname";
    return ($base_db_data_dir . ($type ? ".$type" : ""));
}

=head2 socket_filename($db)

=cut

sub socket_filename {
    my ($db) = @_;

    return $db->{socket} || "/var/run/mysqld.$db->{name}/mysqld.sock";
}

=head2 find_source

    Возвращает строку
    'откуда забирать данные'

=cut

sub find_source
{
    my ($db, %O) = @_;

    my $dbname = $db->{name};
    my $dbdir;
    if ($db->{dbdir}) {
        $dbdir = $db->{dbdir}
    } else {
        write_log("no dbdir for database '$dbname'");
        die;
    }

    my $source;

    if ( $dbdir =~ /^rsync:/ ) {
        my $rsync = File::Rsync->new({});

        # каждую базу пытаемся найти во всех заказанных датах. Первую нашедшуюся используем
        my $list = undef;
        for my $d (@{$O{date}}){
            $list = $rsync->list(
                relative => 1,
                exclude => ['*/tmp/*'],
                source => "$dbdir/$dbname/*/$dbname-$d*.pos"
            );
            last if $list;
        }
        unless ($list) {
            write_log("no backup for $dbname on dates ".join(", ", @{$O{date}})." (no files $dbdir/$dbname/*/$dbname-<date>*.pos)");
            die;
        }
        my $rec = $list->[-1];
        chomp($rec);
        my $path = (split " ", $list->[-1])[-1];
        $path =~ s/\Q$dbname\E-.*$//;
        unless ($path) {
            write_log("can't find source path for db '$dbname'");
            die;
        }

        $source = "$dbdir/$path/mysql.$dbname/";
        if (!scalar($rsync->list(source => "$dbdir/$path/mysql.$dbname"))) {
            $source = "$dbdir/$path/";
        }
    } else {
        $source = "$dbdir/$dbname/";
    }

    if ($source) {
        write_log({"source for $dbname" => $source});
    } else {
        write_log("can't find source path for db '$dbname'");
        die;
    }

    return $source;
}


=head2 rsync_data

    Результат: в каталогах .new -- новые данные
    Если что-то ломается посередине -- умирает

=cut

sub rsync_data
{
    my (%O) = @_;
    write_log("rsync_data: trying to copy backups for all dbs with rsync...");

    my $rsync_opts = hash_merge {
            archive => 1,
            delete => 1,
            timeout => 600,
            exclude  => $O{rsync_exclude},
        }, $O{rsync_opts};

    my $rsync = File::Rsync->new($rsync_opts);

    for my $db (@{$O{databases}}){
        write_log("rsync data: copying $db->{name} with following params:");
        # еще раз проверяем источник
        $db->{source} = find_source($db, %O);
        my $param = {
            source => $db->{source},
            dest => data_dir($db, "new"),
        };
        write_log($param);
        my $ok = $rsync->exec( $param );
        if (!$ok){
            write_log("rsync failed",{err=>$rsync->err, out=>$rsync->out});
            die "rsync failed";
        }
        write_log("rsync data: $db->{name} done");
    }
}



sub stop_mysql
{
    my %O = @_;

    for my $db (@{$O{databases}}){
        write_log("stop_mysql for $db->{name}");
        my $output = qx!/etc/init.d/mysql.$db->{name} stop 2>&1!; # !!! system("kill", "-9", "$(cat /var/run/mysqld."$db"/mysqld.pid)");
        my $status = $? >> 8;
        write_log("stop_mysql $db->{name} status: $status output:\n$output");
        write_log("stop_mysql $db->{name} done");
    }
    for my $db (@{$O{databases}}){
        my ($mysql_alive, $i, $output) = (1, 0, '');
        while ($i++ < 200) {
            $output = qx(/etc/init.d/mysql.$db->{name} status);
            $mysql_alive = not ($? >> 8);
            last unless $mysql_alive;
            write_log("wait for stop $db->{name}: sleep #".$i++);
            sleep 30;
        }
        if ($mysql_alive) {
            write_log("can't stop mysql $db->{name}, process still alive: $output");
            die;
        }
        write_log("wait for stop $db->{name} done");
    }
}


sub start_mysql
{
    my %O = @_;
    for my $db (@{$O{databases}}){
        write_log("start_mysql for $db->{name}");
        my $output = qx(/etc/init.d/mysql.$db->{name} start 2>&1) if -d data_dir($db, '');
        my $status = $? >> 8;
        write_log("start_mysql $db->{name} status: $status output:\n$output");
        write_log("start_mysql $db->{name} done");
    }
    for my $db (@{$O{databases}}){
        write_log("wait for start $db->{name}");
        my $i = 0;

        my $socket_filename = socket_filename($db);
        while ( ! -S $socket_filename ){
            last if $i > 100;
            write_log("wait for $db->{name}: sleep #".$i++);
            sleep 60;
        }
        if (-S $socket_filename){
            write_log("wait for start $db->{name}: done");
        } else {
            write_log("wait for start $db->{name}: fail");
        }
    }
}


sub place_new_data
{
    my %O = @_;

    write_log("place_new_data start");
    eval{
        rename_dirs(from => '', to => 'old', %O) if $O{save_old_set};
        rename_dirs(from => 'ready', to => '', %O);
        for my $db (@{$O{databases}}){
    	    unless (system("chown", "-R", $O{chown}, data_dir($db, '')) == 0) {
                write_log("can't chown mysql data dir for $db");
                die;
            }
        }
    };
    write_log("place_new_data done");
}


sub prepare_db_old
{
    my %O = @_;

    for my $db (@{$O{databases}}){
        my $socket_filename = socket_filename($db);
        system(qq!cat "$DBHOME/$db->{name}.sql" | mysql -u root -S "$socket_filename"!);
    }

    write_log("prepare_db done");
}

sub prepare_db_new
{
    my (%O) = @_;

    unless (ref $O{prepare} eq "ARRAY") {
        write_log("'prepare' key should contain an array");
        die;
    }
    write_log("prepare_db start");
    local $ENV{DATABASES} = join " ", map {$_->{name}} @{$O{databases}};

    for my $action ( @{$O{prepare}} ){
        my $output = `$action 2>&1`;
        my $status = $? >> 8;
        write_log("action $action\nstatus $status\noutput:\n$output\n");
    }

    write_log("prepare_db done");
}

sub start_db_new
{
    my %O = @_;

    unless (dirs_by_type("ready", %O) > 0) {
        write_log("no ready dirs, can't start databases");
        die;
    }

    stop_mysql(%O);
    place_new_data(%O);
    start_mysql(%O);

    if( exists $O{prepare} ){
        prepare_db_new(%O);
    } else {
        # TODO после перехода всех Директовых тестовых БД на отдельные подготовительные скрипты -- prepare_db_old выпилить.
        prepare_db_old(%O);
    }

}
