#!/usr/bin/perl

=encoding utf8

=head1 NAME

    sync-juggler-checks.pl - синхронизация плейбука с juggler-проверками директа

=head1 SYNOPSIS

    # из крона
    /usr/local/bin/sync-juggler-checks.pl

    # вручную
    /usr/loca/bin/sync-juggler-checks.pl --force

    Параметры запуска:
        --help          - вывести справку
        --force         - если плейбук не синхронен с juggler-сервером - применяет его несмотря на время суток/день недели
        --ansible-binary <path> - бинарник ansible, по умолчанию /usr/bin/ansible-playbook
        --juggler-service <name> - имя juggler-сервиса, на который будет отправлен статус
        --inventory <path> -- локальный inventory; по умолчанию /dev/null
        --playbook <path>  -- плейбука, которую накатываем; обязательный
        --syslog-prefix <name> -- имя сислог-префикса, в который будет отправляться messages лог скрипта, умолчание sync-juggler-checks

=head1 DESCRIPTION

    Обертка над ansible-playbook. Принцип работы:
      * сначала запускает ansible-playbook --check ...
      * пишет в логи вывод ansible, парсит его. если все хорошо и синхронно - скрипт завершается
      * если вывод не получилось разобрать или часть действий были failed/unreachable - умирает
      * если есть несихронные проверки, запускает ansible-playbook без --check в следующих случаях:
        - при запуске был указан --force
        - сейчас не суббота/воскресенье/праздничный день (по данным из биллинга)
        - сейчас понедельник-четверг и время в интервале с 12 до 19 часов
        - или сейчас пятница и время в интервале с 12 до 17 часов

В каждом сервисе надо завести кронтаб примерно такого содержания (switchman -- опционально): 
41 9,14,16,18 * * * ppc /usr/local/bin/sleep_under_load ; /usr/bin/switchman -g scripts-other --lockname sync-juggler-checks.pl -- /usr/local/bin/sync-juggler-checks.pl --inventory --playbook


=cut

use Getopt::Long;
use Sys::Hostname::FQDN qw/fqdn/;
use POSIX qw(strftime);

use Direct::Modern;
use Yandex::JugglerQueue;
use Yandex::Log::Messages;

my $ansible = '/bin/ansible-juggler';
my $juggler_service = 'juggler_checks.synced';
my $inventory = '/dev/null';
my $playbook = '';
my $force = 0;
$Yandex::Log::SYSLOG_PREFIX = 'sync-juggler-checks';

my $hostname = fqdn();

GetOptions(
    'force' => \$force,
    'ansible-binary=s' => \$ansible,
    'juggler-service=s' => \$juggler_service,
    'inventory=s' => \$inventory,
    'playbook=s' => \$playbook,
    'syslog-prefix=s' => \$Yandex::Log::SYSLOG_PREFIX,
) or die "can't parse options, stop";

my $log = Yandex::Log::Messages->new_without_tracing(trace_service => 'direct-utils.script', trace_method => "sync-juggler-checks");
$log->{tee} = 1 if $force;

$log->out('START');

$log->out('check playbook');
my $playbook_check_result = execute_ansible();
$log->out('get check statistic');
my $check_stat = get_stat($playbook_check_result);
$log->out($check_stat);

my $can_sync = can_sync();

if ($check_stat->{unreachable} || $check_stat->{failed}) {
    $log->out('FAILED on check: Playbook corrupted, check log for ansible-playbook output');
    juggler_crit(service => $juggler_service, description => "Playbook corrupted! Check sync_juggler_check logs on host $hostname.");
} elsif ($check_stat->{changed} == 0) {
    $log->out("OK on check: No differences between playbook and juggler server");
    juggler_ok(service => $juggler_service, description => "No differences between playbook and juggler server. Processed $check_stat->{ok} actions.");
} elsif (!$can_sync) {
    $log->out("WARN on check: Found $check_stat->{changed} unsynced checks, but can't sync it now.");
    $log->out("You can execute script again with --force parameter to sync it");
    # если сейчас синхронизировать не можем - зажигаем лампочку
    juggler_warn(service => $juggler_service, description => "Found $check_stat->{changed} unsynced checks, skip sync");
} else {
    # пробуем накатить плейбук
    $log->out('apply playbook');
    my $playbook_apply_result = execute_ansible('apply');
    $log->out('get apply statistic');
    my $apply_stat = get_stat($playbook_apply_result);
    $log->out($apply_stat);

    if ($apply_stat->{unreachable} || $apply_stat->{failed}) {
        $log->out('FAILED on apply: Error applying playbook, check log for ansible-playbook output');
        juggler_crit(service => $juggler_service, description => "Error on applying playbook! Check sync_juggler_check logs on host $hostname.");
    } elsif ($apply_stat->{changed} == 0) {
        $log->out("WARN on apply: Nothing changed during applying playbook, check log for ansible-playbook output");
        juggler_warn(service => $juggler_service, description => 'Nothing changed during applying playbook - it is strange!');
    } else {
        $log->out("OK on apply: Successfully synced $apply_stat->{changed} checks. Processed $apply_stat->{ok} actions.");
        juggler_ok(service => $juggler_service, description => "Successfully synced $apply_stat->{changed} checks. Processed $apply_stat->{ok} actions.");
    }
}

$log->out('FINISH');

exit 0;

sub read_ansible_output {
    my $fh = shift;

    my @groups;
    my $prev_line_ref = \'';
    my $lines = [];

    my $flush = sub {
        my $last_flush = shift;
        if ((length($$prev_line_ref) == 0 || $last_flush) && @$lines) {
            if (# успешные задачи - пропускаем
                ! ($lines->[0] && $lines->[0] =~ m/^TASK: .*\*{3,} $/
                   && $lines->[1] && $lines->[1] =~ m/^ok: /
                   ) 
                # при ручных запусках пишем все в логи
                || $force
            ) {
                push @groups, $lines;
            }
            $lines = [];
        }
    };

    while (my $line = <$fh>) {
        chomp $line;

        if (length($line) > 0) {
            $flush->();
            push @$lines, $line;
        }

        $prev_line_ref = \$line;
    }
    $flush->('last_flush');

    return \@groups;
}

sub execute_ansible {
    my $apply = shift;

    my $action = '--check';
    $action = '' if $apply;
    my $cmd = "$ansible $action $playbook";

    $log->out("Execute: $ansible $action $playbook");
    local $ENV{ANSIBLE_LOG_PATH} = '/dev/null';
    local $ENV{ANSIBLE_INVENTORY} = $inventory;
    local $ENV{ANSIBLE_RETRY_FILES_ENABLED} = 'False';
    open(my $fh, '-|', $cmd) or $log->die("Can't start ansible-playbook: $!");
    my $data = read_ansible_output($fh);
    {
        local $log->{tee} = 0;
        $log->bulk_out(ansible_output => $data);
    }
    close($fh) or $log->die($! ? "Error closing ansible-playbook pipe: $!" : "Exit status from ansible-playbook: $?");

    return $data;
}

sub can_sync {
    my @time = localtime();
    my $today  = strftime "%Y-%m-%d", @time;
    my ($hour, $wday) = ($time[2], $time[6]);

    if ($force) {
        # ручной запуск - можно всегда
        return 1;
    } elsif (is_public_holiday($today)) {
        # праздник
        return 0;
    } elsif ($wday == 0 || $wday == 6) {
        # выходной
        return 0;
    } elsif ($wday == 5) {
        # пятница
        if ( $hour > 11 && $hour < 18 ) {
            # в дневные часы - можно
            return 1;
        } else {
            return 0;
        }
    } else {
        # будни (понедельник-четверг)
        if ( $hour > 10 && $hour < 21 ) {
            # в дневные часы - можно
            return 1;
        } else {
            return 0;
        }
    }
}

sub get_stat {
    my $groups = shift;

    if (@$groups
        # для failed эта строчка "отбивается" в отдельную группу
        # && $groups->[-1]->[0] && $groups->[-1]->[0] =~ m/^PLAY RECAP \*{3,} $/
        && $groups->[-1]->[-1] && $groups->[-1]->[-1] =~ m/[^:\s]+\s*:\s*ok=([0-9]+)\s*changed=([0-9]+)\s*unreachable=([0-9]+)\s*failed=([0-9]+)\s*$/
    ) {
        my %stat;
        @stat{qw/ok changed unreachable failed/} = ($1, $2, $3, $4);
        return \%stat;
    } else {
        $log->die("can't parse ansible-playbook output! line with changed/unreachable/failed NOT FOUND");
    }

}

sub is_public_holiday
{
    my ($d) = @_;
    my $is_holiday = {
        "2018-01-01" => 1,
        "2018-01-02" => 1,
        "2018-01-03" => 1,
        "2018-01-04" => 1,
        "2018-01-05" => 1,
        "2018-01-06" => 1,
        "2018-01-07" => 1,
        "2018-01-08" => 1,
        "2018-02-23" => 1,
        "2018-03-08" => 1,
        "2018-03-09" => 1,
        "2018-04-28" => 1,
        "2018-04-29" => 1,
        "2018-04-30" => 1,
        "2018-05-01" => 1,
        "2018-05-02" => 1,
        "2018-05-09" => 1,
        "2018-06-09" => 1,
        "2018-06-11" => 1,
        "2018-06-12" => 1,
        "2018-11-04" => 1,
        "2018-11-05" => 1,
        "2018-12-29" => 1,
        "2018-12-31" => 1,
    };
    return $is_holiday->{$d} // 0;
}
