#!/usr/bin/env perl

=head1 NAME

=encoding utf8

clean-temporary-directories - удалить все устаревшие файлы и каталоги из каталогов с установленным TTL

=head1 DESCRIPTION

    Опции:
    --help
        вывести справку (о том, что больше нет опций)

=cut

use strict;
use warnings;
use Fcntl qw(:flock);
use File::Path qw(remove_tree);
use Getopt::Long;
use POSIX qw/strftime/;

GetOptions(
    "help" => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text >&2"); exit 0;},
);

my $locked = 0;
(open FileLock, '>', '/tmp/temp-ttl/ttl_1d/clean-temporary-directories.lock') || exit;
flock (FileLock, LOCK_EX) || exit;
$locked = 1;

my $date_suffix = strftime('%Y%m', localtime);
open Log, '>>', ('/var/log/temporary-directories/clean-temporary-directories.' . $date_suffix . '.log') || exit;
print Log "START\n";

my $SECONDS_IN_A_TYPICAL_DAY = 60 * 60 * 24;
my $epoch_time = time(); # если кто переведёт часы на несколько суток вперёд, мы это не учтём

my $root = '/tmp/temp-ttl';
opendir(my $root_dir, $root) || die "Can't open: $!";
foreach my $id (grep { /^[^\.]/ } readdir($root_dir)) {
    next if $id eq 'README';

    my $ttl_dir_path = "$root/$id";
    if (-d $ttl_dir_path && $id =~ /ttl_([0-9]+)d/) {
        my $ttl = $1 * $SECONDS_IN_A_TYPICAL_DAY;
        opendir(my $ttl_dir, $ttl_dir_path) || die "Can't open: $!";
        foreach my $file (grep { /^[^\.]/ && -e "$ttl_dir_path/$_" } readdir($ttl_dir)) { # файл или каталог, у которого проверим время
            my $mtime = (stat("$ttl_dir_path/$file"))[9];
            if ($epoch_time - $mtime > $ttl) {
                my $name = "$ttl_dir_path/$file";
                print Log "removing $name\n";
                if (-d $name) {
                    remove_tree($name);
                } else {
                    unlink $name;
                }
            }
        }
        closedir $ttl_dir;
    } else {
        print STDERR "Unexpected name: /tmp/temp-ttl/$id\n"; # напишем на рассылку, что мы видим что-то лишнее
    }
}
closedir $root_dir;
close FileLock;
unlink '/tmp/temp-ttl/ttl_1d/clean-temporary-directories.lock';
print Log "FINISH\n";
close Log;

END {
    if ($locked) {
        unlink '/tmp/temp-ttl/ttl_1d/clean-temporary-directories.lock';
    }
}
