#!/usr/bin/python
# -*- coding: utf-8 -*-

import errno
import logging
import json
import os
import re
import socket
import subprocess
import sys
import time

help_msg = '''
Скрипт для выставления флага "не запускать java-скрипты на машине" для "запасных" инстансов java-сервисов для непродакшеновых сред Директа.
Запускается без параметров, пишет лог в stderr.

Для работы нужны файлы:

  /etc/direct/java-instance-np.json
    список с "основными" инстансами для каждой конфигурации БД (json-словарь имя конфигурации => хост или список хостов через запятую). Путь переопределяется переменной окружения JAVA_INSTANCE_NP_FILE

  /var/lib/direct/my-db-conf
    файл с именем конфигурации (devtest, dev7, ...) -- ключом для словаря из /etc/direct/java-instance-np.json. Путь переопределяется переменной окружения MY_DB_CONFIG_FILE

Под "java-скриптами" подразумеваются процессы java-Директа, в общем случае периодически выполняющие какую-то работу с данными. Например, direct-jobs или binlog-to-yt-sync (экспорт бинлогов в YT).
Под "запасными" инстансами подразумеваются не используемые для редиректов запросов в nginx, и не обязательно регулярно обновляемые; на них можно переключиться в случае сбоя/недоступности "основного".
Ещё про стабильные инстансы java-сервисов для непродакшеновых сред Директа тут: https://wiki.yandex-team.ru/direct/development/howto/java-instances-for-betas/

Опция -h/--help -- показать это сообщение.
'''

if '-h' in sys.argv or '--help' in sys.argv:
    print help_msg
    sys.exit(0)

logging.basicConfig(format='%(asctime)s %(message)s', level=logging.INFO)

INSTANCE_INFO_FILE = os.environ.get('JAVA_INSTANCE_NP_FILE', '/etc/direct/java-instance-np.json')
MY_DB_CONFIG_FILE = os.environ.get('MY_DB_CONFIG_FILE', '/var/lib/direct/my-db-conf')

SERVICES = ['direct-jobs', 'binlog-to-yt-sync']

def log_and_die(msg, exit_code=1):
    logging.error(msg)
    logging.warn('exiting abnormally')
    sys.exit(exit_code)

def get_flag_path(service):
    return os.path.join('/', 'var', 'lib', service, 'dont_start')

def run():
    unknown_conf = False
    try:
        my_db_config = open(MY_DB_CONFIG_FILE, 'r').read().rstrip()
        main_instances = json.loads(open(INSTANCE_INFO_FILE, 'r').read())[my_db_config].split(',')
        main_instances = [s.strip() for s in main_instances]
    except IOError as e:
        log_and_die('could not read from {}: {}'.format(e.filename, e.strerror))
    except KeyError as e:
        logging.warn('unknown config "{}"'.format(my_db_config))
        unknown_conf = True
    def resolve_ipv6(host):
        return socket.getaddrinfo(host, None, socket.AF_INET6, socket.SOCK_STREAM)[0][4][0]
    my_ip = resolve_ipv6(socket.getfqdn())
    main_instance_ips = map(resolve_ipv6, main_instances)
    is_main_instance = my_ip in main_instance_ips
    # для неизвестных конфигураций тоже выставляем флаг, на случай, если java-сервис будет использовать ya_environment для определения конфигурации и будет ходить, например, в тестовую базу
    if (not is_main_instance) or unknown_conf:
        for service in SERVICES:
            flag_path = get_flag_path(service)
            flag_dir = os.path.dirname(flag_path)
            if not os.path.exists(flag_path):
                try:
                    os.makedirs(flag_dir)
                except OSError as e:
                    if e.errno == errno.EEXIST and os.path.isdir(flag_dir):
                        pass
                    else:
                        log_and_die('could not create dir {}: {}'.format(flag_dir, e.strerror))
                try:
                    open(flag_path, 'w').close()
                except IOError as e:
                    log_and_die('could not create flag file {}: {}'.format(e.filename, e.strerror))
                logging.info('created flag file {}'.format(flag_path))
                process_pattern = r'^/(usr/.*)?bin/java.*([Jj]obs|[Bb]inlog)'
                subprocess.call(['pkill', '--full', process_pattern, '--signal', 'TERM'])
                time.sleep(10)
                exit_code = subprocess.call(['pkill', '--full', process_pattern, '--signal', 'KILL'])
                if exit_code > 1:   # 1 -- "No processes matched" -- нормальная ситуация
                    logging.warn("pkill error: exited with code {}".format(exit_code))
                else:
                    logging.info('processes killed')
    elif is_main_instance:
        for service in SERVICES:
            flag_path = get_flag_path(service)
            if os.path.exists(flag_path):
                os.remove(flag_path)
                logging.info('removed flag file {}'.format(flag_path))

if __name__ == '__main__':
    run()
