#!/usr/bin/perl

=head1 DESCRIPTION

    Генерация TT-html с фразами в iget(...) из js-файлов.
    файл сохраняется рядом с js, для его подключения в исходный html нужно добавить:
    [% PROCESS Direct/VCardsManager/VCard.i18n.html %]

    поиск всех сгенеренных файлов:
    find data -name '*.i18n.html'

    удаление всех сгенеренных файлов:
    find data -name '*.i18n.html' -delete
    
    Использование:
    generate_i18n_tthtml_from_js.pl -dirs <DIRS_PATH> [-template <TEMPLATE_PATH>] [-force]
    Ключи:
    -dirs      : Обязательный ключ, DIRS_PATH - путь до папок с js файлами, перечисленные через запятую без пробелов.
	-template  : Шаблон, который надо использовать для создания новых файлов. По умолчанию берется универсальный. 
	             В момент переноса скрипта совпадает с шаблоном Директа и Модерации.
	-f, -force : Принудительная перегенерация шаблонов.
    -bundle    : Все фразы в одном файле, результат отдаем в STDOUT

=cut

use warnings;
use strict;

use Getopt::Long;
use List::MoreUtils qw/uniq/;
use File::Basename qw/basename/;
use File::Slurp;
use Template;

use Yandex::Shell;
use Yandex::Svn qw/svn_files/;

use utf8;
use open ':std' => ':utf8';

#-----------------------------------------------------------

my %OPT = (tag => 1);
GetOptions(
    'help' => \&usage,
    'f|force'=> \$OPT{force},
    'template=s' => \$OPT{template},
    'dirs=s' => \$OPT{dirs},
    'git-dirs=s' => \$OPT{git_dirs},
    'bundle' => \$OPT{bundle},
    'tag!' => \$OPT{tag},
) or die "bad options, use --help";
my @JS_DIRS;
my @GIT_JS_DIRS;

my $SCRIPT = basename($0);
my $TEMPLATE;
my $TT = Template->new(FILTERS => {js => \&js_quote});

#-----------------------------------------------------------
sub check_options
{
    die "dirs options have to be defined. Please, read generate_i18n_tthtml_from_js.pl --help\n" unless (defined($OPT{dirs}));
    @JS_DIRS = split '\s*,\s*', $OPT{dirs};
    @GIT_JS_DIRS = split '\s*,\s*', $OPT{git_dirs} || '';
    die "Dirs are not defined" if (! scalar( grep {-d $_} @JS_DIRS));

    if ($OPT{template} && -f $OPT{template}) {
        $TEMPLATE = read_file($OPT{template}, binmode => ':utf8');
    } else {
        $TEMPLATE = join '', <DATA>;
    }
}

#-----------------------------------------------------------
sub main
{
    check_options();
    my @phrases = ();
    for my $js_file (svn_files(\@JS_DIRS), map {split /\n/, `find $_ -type f`} @GIT_JS_DIRS) {
        next unless $js_file =~ /\.(js|bemhtml)$/ && -f $js_file;
        if ($OPT{bundle}) {
            push @phrases, extract_phrases_from_js($js_file);
        } else {
            generate_tt_html_from_js($js_file);
        }
    }
    if ($OPT{bundle}) {
        my $dict = generate_dict_from_phrases([uniq @phrases]);
        if ($OPT{tag}) {
            $dict = wrap_in_tag($dict);
        }
        print $dict;
    }
}

#-----------------------------------------------------------
sub generate_tt_html_from_js
{
    my $js_file = shift;

    my $html_file = $js_file;
    $html_file =~ s/\.js$/.i18n.html/;

    # если время модификации html позже чем js - то пропускаем
    if ( ! $OPT{force} && -e $html_file && (stat($html_file))[9] > (stat($js_file))[9]) {
        return;
    }

    if (my @phrases = extract_phrases_from_js($js_file)) {      
        my $html = generate_dict_from_phrases(\@phrases);
        $html = wrap_in_tag($html);

        write_file($html_file, {atomic => 1, binmode => ':utf8'}, $html);
        yash_system('svn', '-q', add => $html_file);

        warn "created (,added to svn): $html_file\n";
    }
}

#-----------------------------------------------------------
sub extract_phrases_from_js
{
    my $js_file = shift;

    my $js_code = read_file($js_file, binmode => ':utf8');

    my %phrases;
    while ($js_code =~ /iget \s* \( \s* (['"]) ( .+? ) (?<!\\) \1/gx) {
        $phrases{$2}++;
    }

    return keys %phrases;
}

#-----------------------------------------------------------
sub generate_dict_from_phrases
{
    my $phrases = shift;

    my $vars = {phrases => [sort @$phrases]};

    my $html;
    $TT->process(\$TEMPLATE, $vars, \$html);
    return $html;
}

#-----------------------------------------------------------
sub wrap_in_tag
{
    my $content = shift;
    return qq{[%# generated by $SCRIPT -%]<script type="text/javascript">$content</script>};
}

#-----------------------------------------------------------
=head2 usage


=cut

sub usage
{
    system("podselect -section DESCRIPTION -section SYNOPSIS $0 | pod2text-utf8 >&2");
    exit(0);
}

#-----------------------------------------------------------
=head2 js_quote

 quote string for javascript

=cut

sub js_quote
{
    my $str = shift;
    return unless defined $str;

    for ($str) {
        s/(['"\\\/])/\\$1/g;
        s/\r//g;
        s/\n/\\n/g;
    }

    return $str;
}
#-----------------------------------------------------------

main();


__DATA__
[% TAGS <% %> -%]
window.i18n = $.extend(window.i18n || {}, {
<%- FOR ph = phrases %>
<% IF ! loop.first %>,<% END %>"<% ph.replace('"', '\"') %>":"[% iget("<% ph.replace('"', '\"') %>") | js %]"
<%- END %>
});
