#!/usr/bin/perl -w

=head1 NAME

    translations-diff.pl

=head1 DESCRIPTION

    Параметры:
        --old - старая релизная версия (обязателен)
        --new - новая релизная версия (обязателен)
        --ticket - создать тикет, в котором будет дифф
        
    Cкрипт берет базовые ревизии из обеих версий и берет diff файлов с переводами в транке между этими двумя ревизиями.
    По умолчанию выводит на stdout, с параметром --ticket создает тикет и записывает дифф в его описание.

    Пример запуска:
        translations-diff.pl --new 1.154188-1 --old 1.153992.154167-1 --ticket

=head1 COMMENTS

=cut

use strict;
use warnings;

use Getopt::Long;
use Startrek::Client::Easy;
use ProjectSpecific qw/svn_url/;

use open ':std' => ':utf8';
use utf8;

# максимальное количество символов diff'а, которые отправятся в тикет
use constant MAX_CHARACTERS_TO_TICKET => 500000;
my $DIRECT_SVN_URL = svn_url('trunk');
my $LOCALE_PATH = "/locale";
my $DATA3_PATH = "/data3";
my @LANGUAGES = ("ru", "en", "tr", "uk");

my %OPT;
GetOptions(
    'old=s' => \$OPT{old},
    'new=s' => \$OPT{new},
    'ticket' => \$OPT{ticket},
    'help' => \&usage,
);

# номера ревизий старой и новой версии
my $new_rev;
my $old_rev;

die "Can't get number of revision of 'new' release" unless $OPT{new} and $OPT{new} =~ m!^1\.([0-9]+)!;
$new_rev = $1;

die "Can't get number of revision of 'old' release" unless $OPT{old} and $OPT{old} =~ m!^1\.([0-9]+)!;
$old_rev = $1;

my $svn_diff = "";

# берем дифф для файлов с переводами из data3
my $str_langs = join '|', map {$_ . '.js$'} @LANGUAGES;
# всю работу делаем одним большим шельным однострочником
$svn_diff .= `svn diff $DIRECT_SVN_URL$DATA3_PATH -r $old_rev:$new_rev --summarize | awk '{print \$2}' | grep -E '$str_langs' | xargs -r -I% sh -c 'printf "\n%\n"; svn diff % -r $old_rev:$new_rev'`;

# берем дифф для файлов с переводами из locale
$svn_diff .= `svn diff $DIRECT_SVN_URL$LOCALE_PATH -r $old_rev:$new_rev`;

if ($OPT{ticket}) {
    if (length($svn_diff) > MAX_CHARACTERS_TO_TICKET) {
        $svn_diff = "Дифф очень большой, оставлено только ".MAX_CHARACTERS_TO_TICKET." первых символов.\n".(substr $svn_diff, 0, MAX_CHARACTERS_TO_TICKET);
    }
    $svn_diff = "%%".$svn_diff."%%";
    my $st = Startrek::Client::Easy->new();
    my $new_issue_key = $st->do(
        create  => 1,
        queue   => 'DIRECT',
        type    => 'task',
        summary => "Diff переводов между версиями $OPT{old} и $OPT{new}",
        description => $svn_diff,
    );
    print $new_issue_key ? "Startrek ticket has been created: https://st.yandex-team.ru/$new_issue_key\n" : "Error when creating startrek ticket!\n";
} else {
    print $svn_diff;
}


sub usage
{
    system("podselect -section DESCRIPTION -section SYNOPSIS $0 | pod2text-utf8");
    exit(0);
}
