#!/usr/bin/perl -w

=head1 NAME

    update-translations-auto.pl

=head1 DESCRIPTION

    Проделывает все работу с переводами за вас

    Чекаутит чистую рабочую копию транка в /tmp/temp-ttl/ttl_7d/update-translations-auto_XXXXXXX/
    Запускает direct-mk translations-tanker
    Проверяет, что все ок, отсылает оповещение в ямб (если нет, то падает естественно)
    Коммитит в тикет "Очередные переводы", если надо

    Параметры:
        --commit, выставить, если нужно коммитить переводы
        --no-cleanup, не удалять зачекаутенную рабочую копию после завершения работы скрипта
        --no-yamb-notify, выставить, если не надо отправлять оповещения в ямб

    Пример запуска:
    update-translations-auto.pl --no-cleanup (оставляет рабочую копию, не делает коммит, поэтому надо все проверить и закоммитить в ручную)
    update-translations-auto.pl --commit (полностью автоматическая работа вместе коммитом, прийдет оповещение в ямб об успешном завершении или о проблемах)

=head1 COMMENTS

=cut

use strict;
use warnings;

use File::Temp qw/ tempdir /;
use Getopt::Long;
use Startrek::Client::Easy;
use Yandex::Shell qw/ yash_qx yash_system/;
use Yandex::Hostname;
use Yandex::Svn qw/svn_file_status/;
use Yandex::Yamb;
use Yandex::JugglerQueue;
use Date::Parse;
use POSIX qw(strftime);
use Time::Local;
use ProjectSpecific qw/get_startrek_client_token get_sign_comment svn_url/;
use DateTime;
use Yandex::DateTime;
use Data::Dumper;
use File::Basename;

use open ':std' => ':utf8';
use utf8;

our $STARTREK_ROBOT_TOKEN_FILE = '/etc/direct-tokens/startrek';
our $DIRECT_SVN_URL = svn_url('trunk');
our $TICKET_NAME = "Очередные переводы";
our $TICKET_LIFETIME = 60; # дней
our $HOURS_AFTER_PREVIOUS_COMMIT = 20; # количество часов, которое должно пройти с момента прошлого коммита
our $MAX_N_MODIFIED_FILES = 300; # максимальное количество модифицированных файлов
our $MAX_ADDED_LINES = 10000; # максимальное количество добавленных строк по диффу
our $MAX_REMOVED_LINES = 10000; # максимальное количество удаленных строк по диффу
our @DATA3_FILES = ("ru\.js", "en\.js", "tr\.js", "uk\.js", "map\.json"); # допустимые файлы для изменений в /data3
#our @DATA_FILES = ("ru\.js", "en\.js", "tr\.js", "uk\.js"); # допустимые файлы для изменений в /data
our $ARCADIA_URL = 'https://a.yandex-team.ru';
our $domain_login = [getpwuid($<)]->[0];
my $host = Yandex::Hostname::hostfqdn();
my $sign_comment = get_sign_comment(basename(__FILE__));

$ENV{PATH} = "/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/usr/games:/usr/local/games";

my %OPT;
GetOptions(
    'no-yamb-notify' => \$OPT{no_yamb},
    'commit' => \$OPT{commit},
    'no-cleanup' => \$OPT{no_cleanup},
    'help' => \&usage,
);

# чекаутим рабочую копию транка во временную папку
my $temp_dir = tempdir('update-translations-auto_XXXXXXX', CLEANUP => $OPT{no_cleanup} ? 0 : 1, DIR => '/tmp/temp-ttl/ttl_7d');
yash_system('svn', 'co','-q', $DIRECT_SVN_URL, $temp_dir);

#переводам в data3 надо явно сказать, с какой веткой Танкера работать; см. также DIRECT-78348
$ENV{TANKER_BRANCH} = 'master'; 
# запускаем работу с переводами
chdir $temp_dir;
my $translations_log = yash_system("direct-mk translations-tanker");
chdir '/';

if (!$OPT{commit}) {
    if (!$OPT{no_yamb}) {
        my $message;
        if ($OPT{no_cleanup}) {
            $message = "update-translations-auto.pl выполнил работу с переводами. Перейдите в папку $temp_dir на машинке $host, чтобы проверить изменения и закоммитить.";
        } else {
            $message = "update-translations-auto.pl выполнил работу с переводами.";
        }
        send_yamb_message(to => $domain_login, message => $message);
    }
    exit(0);
}

# в DIRECT-113573 решили оставить проверку на частоту коммита только для робота ppc
if ($domain_login eq "ppc") {
    # проверяем, что последний коммит от текущего юзера был достаточно давно
    my $dt = now()->subtract(hours => $HOURS_AFTER_PREVIOUS_COMMIT);
    my $dt_since = $dt->strftime("%Y-%m-%d %H:%M");
    
    # находим коммиты за определенное время, и проверяем даты вручную, т. к. в svn log наблюдается баг в этом моменте
    my $stdout = yash_qx('svn log', $DIRECT_SVN_URL, '-r', "{$dt_since}:HEAD", '--search', $domain_login, '--search-and', $TICKET_NAME, '--xml');
    my @dates = $stdout =~ m!<date>(.+)</date>!g;
    for my $date (@dates) {
        my $dt_rev = iso8601_2_datetime($date);
        if (DateTime->compare($dt, $dt_rev) >= 0) {
            next;
        }
        if (!$OPT{no_yamb}) {
            send_yamb_message(to => $domain_login, message => "update-translations-auto.pl выполнил работу с переводами, но без коммита, т.к. вы уже делали недавно коммит с переводами");
        }
        die "Вы уже делали недавно коммит с переводами";
    }
}

# получаем список модифицированных файлов
my $modified_files = svn_file_status($temp_dir, no_verbose => 1)->{modified};

# проверяем изменения
check_diff($modified_files, $temp_dir, \%OPT);

# проверяем, что тикет с переводами не устарел, иначе создаем новый
my $st;
if ($domain_login ne 'ppc') {
    $st = Startrek::Client::Easy->new();
} else {
    $st = Startrek::Client::Easy->new(token => get_startrek_client_token(file => $STARTREK_ROBOT_TOKEN_FILE));
}
my $ticket = $st->get(query => 'Queue: DIRECT and Summary: #"'.$TICKET_NAME.'" "Sort By": Created DESC', limit => 1);

if ((time() - str2time($ticket->{createdAt})) / 86400 > $TICKET_LIFETIME) {
    my $ticket_key = $st->do(
        create  => 1,
        queue   => 'DIRECT',
        type    => 'task',
        summary => $TICKET_NAME,
        description => "",
    );
    die "Не удалось создать тикет" unless $ticket_key;
    $ticket = {
        key => $ticket_key,
        summary => $TICKET_NAME,
        status => 'open',
    };
}

# если все ок, то коммитим
# если запускается от пользователя ppc, то приходится делать все в ручную :(
if ($domain_login ne 'ppc') {
    chdir $temp_dir;
    yash_system(
        "commit",
        '-j', "$ticket->{key}",
        "-m", "$ticket->{key}: $ticket->{summary}",
        '-c', "Автокоммит переводов\n$sign_comment",
        '-cl'
    );
    chdir '/';
} else {
    my $commit_log = yash_qx(
        'svn ci', $temp_dir,
        '-m', "$ticket->{key}: $ticket->{summary}",
        '--with-revprop', 'arcanum:check-skip=yes'
    );

    # получаем номер ревизии
    $commit_log = substr $commit_log, -100;
    my $revision = 0;
    if ($commit_log =~ m!^(?:committed\s+revision|\w+\s+\w+)\s+([0-9]+)\.\s*$!mi) {
        $revision = $1;
    } else {
        die "Возникла ошибка при коммите";
    }

    # получаем url ревизии в табуле
    my $revision_url = get_svnlog_url($revision);

    # отсылаем комментарий, как в скрипте commit
    my $comment = "Автокоммит переводов";
    $comment .= "\nCommitted revision $revision";
    $comment .= "\n$revision_url";
    $comment .= "\nticket: https://st.yandex-team.ru/$ticket->{key}";
    $comment .= "\n$sign_comment";

    if ($ticket->{status} ne "closed") {
        $st->do(key => $ticket->{key}, comment => $comment, close => 1);
    } else {
        $st->do(key => $ticket->{key}, comment => $comment);
    }
}

# отсылаем оповещение, что все ок
if (!$OPT{no_yamb}) {
    send_yamb_message(to=> $domain_login, message => "update-translations-auto.pl успешно закоммитил переводы.\nТикет: $ticket->{key}");
}

queue_juggler_event(service => 'update-translations-auto', status => 'OK', description => 'OK');

sub usage
{
    system("podselect -section DESCRIPTION -section SYNOPSIS $0 | pod2text-utf8");
    exit(0);
}


sub check_diff
{
    my $modified_files = shift;
    my $temp_dir = shift;
    my $OPT = shift;

    my $err_msg = '';

    if (!$modified_files) {
        $err_msg .= "\nНет изменений в переводах";
    }

    if ($modified_files && scalar @$modified_files > $MAX_N_MODIFIED_FILES) {
        $err_msg .= "\nСлишком много модифицированных файлов.";
    }

    my @bad_files;

    # проверяем название модифицированных файлов
    for my $file (@$modified_files) {
        if ($file =~ m!$temp_dir(.+)!) {
            $file = $1;
        }
        if ($file =~ m!^/locale/!) {
            next;
        }
        my $data3_files_str = join("|", @DATA3_FILES);
        if ($file =~ m!^/data3/.+($data3_files_str)$!) {
            next;
        }

        if ($file =~ m!^/data/block/i-translation/!) {
            next;
        }

        push @bad_files, $file;
    }

    if (@bad_files) {
        $err_msg .= "\nЕсть нераспознанные модифицированные файлы:\n".join("\n", @bad_files);
    }

    my $diff_stat = yash_qx("svn diff $temp_dir | diffstat");

    if ($diff_stat =~ m!([0-9]+) insertions!) {
        if ($1 > $MAX_ADDED_LINES) {
            $err_msg .= "\nСлишком много добавленных строк.";
        }
    }

    if ($diff_stat =~ m!([0-9]+) deletions!) {
        if ($1 > $MAX_REMOVED_LINES) {
            $err_msg .= "\nСлишком много удаленных строк.";
        }
    }

    if ($err_msg) {
        if (!$OPT->{no_yamb}) {
            my $message = "update-translations-auto.pl не удалось завершить работу с переводами. Проблемы:$err_msg";
            if ($OPT->{no_cleanup}) {
                $message .= "\n\nЛежит в $temp_dir на машинке $host";
            }
            send_yamb_message(to=> $domain_login, message => $message);
        }
        die $err_msg;
    }

}

sub get_svnlog_url {
    my ($revision) = @_;
    die "no revision given" unless $revision;
    return join('/', $ARCADIA_URL, 'arc','commit', $revision);
}
