#!/usr/bin/perl -w

# $Id: update_translations.pl 25375 2011-09-07 12:52:01Z eboguslavskaya $

=head1 NAME

    update_translations.pl

=head1 DESCRIPTION

    Забираем из танкера новые переводы
    Проверяем, появились ли в рабочей копии новые непереведенные сообщения, отправляем их в танкер
    
    При ручном запуске выполнять из корня рабочей копии без ключей

    Клюичи для запуска по крону:
        --base-dir=<абсолютный путь>,
          относительно которого приведены все пути в конфиге
        --config=<путь до конфиг-файла>, по умолчанию etc/translation.conf
        --email-report отправлять результат на почту, иначе результаты выдаются в STDERR
        --clean выставить, если нужно оставить в танкере только те фразы, которые есть в коде (по умолчанию устаревшие фразы не удаляются, а переносятся в Outdated)

    Дополнительные опции:
    --debug | --verbose | -v - выводить в stderr отладочные сообщения

    Конфигурационнай файл должен содержать:
        locale_path = <путь до папки с локализационными файлами, со слешем на конце>
        xiget_script = <путь до xiget.pl файла>
        project_id = <название проекта>
        project_id_version = <полное название проекта>
        l10n_api_url = <ссылка до api такера> - необязательный ключ
    опционально:
        send_emails = 1 # посылать в танкер также шаблоны писем
        preprocess = <строка, которая выполнится шеллом из base-dir>

    Пример: translation.conf
        locale_path = /var/www/ppc.yandex.ru/locale/
        xiget_script = /var/www/ppc.yandex.ru/protected/maintenance/xiget.pl
        project_id = direct
        project_id_version = direct.yandex.ru
        l10n_api_url = http://tanker-test.yandex-team.ru:3000

=head1 COMMENTS

=cut

use strict;
use warnings;

use Config::General;
use Cwd qw/getcwd/;
use File::Compare qw/compare/;
use File::Copy qw/copy/;
use Getopt::Long;
use IO::Prompt;
use Net::ZooKeeper qw(:acls :errors :events :node_flags);
use XML::LibXML;
use YAML;
use POSIX 'strftime';

use Yandex::Trace;
use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::I18nTools;
use Yandex::MailTemplate;
use Yandex::SendMail qw/send_alert/;
use Yandex::Shell qw/yash_qx/;
use Yandex::Svn qw/svn_info/;

use open ':std' => ':utf8';
use utf8;

{
    no warnings 'once';
    $Yandex::Log::USE_SYSLOG = 0;
    $Yandex::Log::LOG_ROOT = '.';
}

my $trace = Yandex::Trace->new(service => 'common.scripts', method => 'update_translations.pl');

my $profile = Yandex::Trace::new_profile('prepare');

my $DEFAULT_BRANCH = 'master';
my $DEBUG = 0;
my $ALLOW_BRANCHES = 0;
my $NEED_CLEAN = 0;

my $BASE_DIR = getcwd();
my %OPT;
GetOptions(
    'base-dir=s' => \$BASE_DIR,
    'config=s' => \$OPT{config},
    'email-report' => \$OPT{email_report},
    'debug|verbose|v!' => \$DEBUG,
    'allow-branches' => \$ALLOW_BRANCHES,
    'clean' => \$NEED_CLEAN,
    'help' => \&usage,
);
my $DEFAULT_CONFIG_LOCATION = "$BASE_DIR/etc/translation.conf";
if (!$OPT{config} && -f $DEFAULT_CONFIG_LOCATION) {
    $OPT{config} = $DEFAULT_CONFIG_LOCATION;
}
die "config file is not defined" unless $OPT{config};
my %conf = Config::General->new( -ConfigFile => $OPT{config}, -ForceArray => 1)->getall;
process_config(\%conf);

my $zkh;
get_lock(\%conf);

if ($conf{preprocess}) {
    chdir $BASE_DIR;
    system($conf{preprocess});
}

my $BRANCH = detect_branch($BASE_DIR);
if ($BRANCH ne $DEFAULT_BRANCH && !$ALLOW_BRANCHES){
    my $allow_branch = prompt 
        '-yes',
        '-default' => 'n',
    "\nРабочая копия смотрит на бранч $BRANCH. 
Если продолжить синхронизацию, для нее будет создан бранч в Танкере. 
Создавать бранчи в Танкере следует только в крайних случаях, когда без этого совсем никак. 
Действительно нужен бранч в Танкере? [yN]
(есть ключ --allow-branches на случай, если нужны бранчи, а интерактивное подтверждение вредно)\n",
        '>';
    die "\nне создаем бранч в Танкере, ничего не синхронизуем\n" if !$allow_branch;
}

if ($BRANCH ne $DEFAULT_BRANCH && !grep {$BRANCH eq $_} @{Yandex::I18nTools::get_branches()}) {
    Yandex::I18nTools::create_branch($BRANCH) or die "couldn't create tanker branch $BRANCH";
}
if ($BRANCH eq $DEFAULT_BRANCH && !is_working_copy_suitable($BASE_DIR, \%conf)) {
    die "Won't upload to Tanker's $DEFAULT_BRANCH\n";
}

my @report = ();

sub log_report
{
    push @report, @_;
    return if $OPT{email_report};
    print STDERR ''.(join "\n", map { s!\n*$!!gr; } @_)."\n";
}

sub log_debug
{
    return if $OPT{email_report} || !$DEBUG;
    print STDERR ''.(join "\n", map { strftime("%F %T", localtime(time))." ".(s!\n*$!!gr) } @_)."\n"
}

undef $profile;
$profile = Yandex::Trace::new_profile('xiget');

log_debug('Yandex::I18nTools::get_pot_filename()');
my $pot_filename = Yandex::I18nTools::get_pot_filename();

if ($NEED_CLEAN) {
    log_debug("$conf{xiget_script} | msgcat - --sort-output > $pot_filename");
    `$conf{xiget_script} | msgcat - --sort-output > $pot_filename`;
} else {
    log_debug("Yandex::I18nTools::get_all_phrases_from_tanker($BRANCH)");
    Yandex::I18nTools::get_all_phrases_from_tanker($BRANCH);
    log_debug("$conf{xiget_script} | msgcat --use-first - $pot_filename  -o $pot_filename");
    `$conf{xiget_script} | msgcat --use-first - $pot_filename --sort-output -o $pot_filename`;
}

my $pot_lines = int(yash_qx("wc -l $pot_filename |grep -o '^[0-9]*'"));
log_report("updated $pot_filename, got $pot_lines lines");
die "too short .pot file $pot_filename: $pot_lines lines, expected at least min_pot_lines=$conf{min_pot_lines}" if $pot_lines < $conf{min_pot_lines};

undef $profile;
$profile = Yandex::Trace::new_profile('update_po');

my $lang_option = $conf{lang} ? "-lang=$conf{lang}" : "";
log_debug("update_po.pl -locale_path $conf{locale_path} -project_id $conf{project_id} $lang_option");
if (my $update_error = `update_po.pl -locale_path $conf{locale_path} -project_id $conf{project_id} $lang_option`) {
    log_report('update_po error:', $update_error);
} else {
    log_report('update_po OK');
}

undef $profile;
$profile = Yandex::Trace::new_profile('main_phrases');

log_debug('Yandex::I18nTools::get_translated_stats()');
my %old_stats = Yandex::I18nTools::get_translated_stats();
log_debug("Yandex::I18nTools::merge_everything_from_l10n_team('$BRANCH')");
Yandex::I18nTools::merge_everything_from_l10n_team($BRANCH);
log_debug('Yandex::I18nTools::get_translated_stats()');
my %new_stats = Yandex::I18nTools::get_translated_stats();
log_debug("compile_mo.pl -locale_path $conf{locale_path} -project_id $conf{project_id} $lang_option");
if (my $compile_error = `compile_mo.pl -locale_path $conf{locale_path} -project_id $conf{project_id} $lang_option`) {
    log_report('compile_mo error:', $compile_error);
} else {
    log_report('compile_mo OK');
    log_debug("Yandex::I18nTools::send_everything_to_l10n_team('$BRANCH')");
    Yandex::I18nTools::send_everything_to_l10n_team($BRANCH);
}

log_debug('Yandex::I18n::get_other_langs()');
my %diff_by_lang = map {$_ => $new_stats{$_}->{translated} - $old_stats{$_}->{translated}} Yandex::I18n::get_other_langs();
my @changed_langs = grep {$diff_by_lang{$_} > 0} sort keys %diff_by_lang;
log_report(map { $diff_by_lang{$_}.' new translations for '.$_ } @changed_langs);
if (@changed_langs) {
    log_report('svndiff '. join(' ', map {Yandex::I18nTools::get_po_filename($_)} @changed_langs));
}
# убрать проверку после обновления libwww-perl в Директе и Модерации
if ($HTTP::Request::Common::VERSION >= 6) {
if ($BRANCH eq $DEFAULT_BRANCH) {
    # удаляем неактуальные кейсеты
    my %preserve = map { $_ => 1 } @{$conf{preserve_keyset} || []};
    log_debug("Yandex::I18nTools::get_keysets('$BRANCH')");
    my @to_delete = grep {! exists $new_stats{hints}->{$_} && ! $preserve{$_} } Yandex::I18nTools::get_keysets($BRANCH);
    die "too many to delete:\n".join("\n", @to_delete) if @to_delete >= keys %{$new_stats{hints}};
    for my $keyset (@to_delete) {
        log_debug("Yandex::I18nTools::delete_keyset('$BRANCH', '$keyset'");
        if (Yandex::I18nTools::delete_keyset($BRANCH, $keyset)) {
            log_report("deleted keyset $keyset");
        } else {
            warn "failed to delete keyset $keyset";
        }
    }
}
}

undef $profile;
$profile = Yandex::Trace::new_profile('emails');

if ($conf{send_emails}) {
    local $Yandex::I18n::PROJECT_ID = "$Yandex::I18n::PROJECT_ID-emails";

    log_debug('Yandex::I18nTools::get_branches()');
    if ($BRANCH ne $DEFAULT_BRANCH && !grep {$BRANCH eq $_} @{Yandex::I18nTools::get_branches()}) {
        log_debug("Yandex::I18nTools::create_branch('$BRANCH')");
        Yandex::I18nTools::create_branch($BRANCH) or die "couldn't create tanker branch $BRANCH";
    }
    log_debug("Yandex::I18nTools::send_emails_to_l10n_team('$BRANCH')");
    Yandex::I18nTools::send_emails_to_l10n_team($BRANCH);
    log_debug("Yandex::I18nTools::get_emails_translation('$BRANCH')");
    Yandex::I18nTools::get_emails_translation($BRANCH);
    log_report('updated '.Yandex::I18nTools::get_emails_filename());

    # удаляем неактуальные кейсеты с письмами
if ($HTTP::Request::Common::VERSION >= 6) {
    if ($BRANCH eq $DEFAULT_BRANCH) {
        log_debug("Yandex::MailTemplate::get_email_template_list_names()");
        my %templates = map {$_ => 1} Yandex::MailTemplate::get_email_template_list_names();
        log_debug("Yandex::I18nTools::get_keysets('$BRANCH')");
        for my $keyset (grep {not exists $templates{$_}} Yandex::I18nTools::get_keysets($BRANCH)) {
            log_debug("Yandex::I18nTools::delete_keyset('$BRANCH', '$keyset')");
            if (Yandex::I18nTools::delete_keyset($BRANCH, $keyset)) {
                log_report("deleted keyset $keyset");
            } else {
                warn "failed to delete keyset $keyset";
            }
        }
    }
}
}

undef $profile;
$profile = Yandex::Trace::new_profile('postprocess');

if ($conf{postprocess}) {
    chdir $BASE_DIR;
    log_debug("$conf{postprocess}");
    system($conf{postprocess});
}

my $report = join "\n", @report;
my $BROADCAST_EMAIL = $conf{broadcast_email};
if ($OPT{email_report} && $BROADCAST_EMAIL) {
    local $Yandex::SendMail::FAKEMAIL = undef;
    local $Yandex::SendMail::FROM_FIELDS_FOR_ALERTS = $BROADCAST_EMAIL;
    Yandex::SendMail::send_alert($report, 'translations update', $BROADCAST_EMAIL);
}

undef $profile;

=head2 process_config(\%conf)

    Конфигурируем модули для работы с Танкером и gettext

=cut

sub process_config
{
    my $conf = shift;

    my @missing_params = grep {!defined $conf->{$_}} qw/
        locale_path
        project_id
        project_id_version
        xiget_script
    /;
    die "Missing params in config: ".join(', ', @missing_params) if @missing_params;

    $conf->{locale_path} = "$BASE_DIR/$conf->{locale_path}";
    die "locale_path $conf->{locale_path} does not exist" unless -d $conf->{locale_path};

    $conf->{xiget_script} = "$BASE_DIR/$conf->{xiget_script}";
    die "xiget_script $conf->{xiget_script} is invalid" unless -x $conf->{xiget_script} && -r $conf->{xiget_script};

    $Yandex::I18n::LOCALE_PATH = $conf->{locale_path};
    if( exists $conf->{locales} ){
        $Yandex::I18n::LOCALES{$_} = $conf->{locales}->{$_} for keys %{$conf->{locales}};
    }
    %Yandex::I18n::LOCALES = %{hash_cut \%Yandex::I18n::LOCALES, split /\s*,\s*/, $conf->{lang}} if $conf->{lang};
    $Yandex::I18n::PROJECT_ID = $conf->{project_id};
    if ($conf->{authorization}) {
        if ($conf->{authorization} =~ m!file:(.+)!) {
            my $token_path = $1;
            open my $token_file, $token_path or die "Can't open $token_path: $!";
            chomp($Yandex::I18nTools::AUTHORIZATION = <$token_file>);
            close $token_file;
        } else {
            $Yandex::I18nTools::AUTHORIZATION = $conf->{authorization};
        }
    }
    $Yandex::I18nTools::PROJECT_ID_VERSION  = $conf->{project_id_version};
    $Yandex::Tanker::L10N_API_URL = $conf->{l10n_api_url} if $conf->{l10n_api_url};
    $Yandex::I18nTools::SURROGATE_EMAILS_KEYS = 1 if $conf->{surrogate_emails_keys};
    if ($conf->{email_templates_folder}) {
        $Yandex::MailTemplate::EMAIL_TEMPLATES_FOLDER = "$BASE_DIR/$conf->{email_templates_folder}";
        die "$Yandex::MailTemplate::EMAIL_TEMPLATES_FOLDER dir does not exist" unless -d $Yandex::MailTemplate::EMAIL_TEMPLATES_FOLDER;
    }

    $conf->{min_pot_lines} //= 10;
}


=head2 usage

    Показываем справку

=cut

sub usage
{
    system("podselect -section DESCRIPTION -section SYNOPSIS $0 | pod2text-utf8 >&2");
    exit(0);
}


=head2 detect_branch($dir)

    Определяем, откуда запустили скрипт

=cut

sub detect_branch
{
    my $dir = shift;

    my $url = eval {
        my $svn_info = svn_info($dir);
        $svn_info->{url}
    };
    if ($url && $url =~ /branches\/([^\/]+)/) {
        return $1;
    }
    return $DEFAULT_BRANCH;
}


=head2 is_working_copy_suitable($dir, $conf)

    Проверяем, что рабочая копия актуальная и без изменений,
    чтобы не затереть свежие фразы из транка

=cut

sub is_working_copy_suitable
{
    my $dir = shift;
    my $conf = shift;

    my $result = 1;
    my $svn_info = svn_info($dir);
    if ($svn_info->{revision} < svn_info($svn_info->{url})->{last_change_rev}) {
        warn "Your working copy is out of date\n";
        $result = 0;
    }

    my $xml = yash_qx("svn", "st", "--xml", $dir);
    my $root = XML::LibXML->new()->parse_string($xml)->documentElement();            
    my @modifications = map {$_->getAttribute('path')} $root->findnodes('//entry[wc-status[@item="modified"]]');

    my @ignore_modifications = ('locale/', 'etc/translation.conf');
    if (ref $conf->{ignore_modifications}) {
        push @ignore_modifications, @{$conf->{ignore_modifications}};
    } elsif ($conf->{ignore_modifications}) {
        push @ignore_modifications, $conf->{ignore_modifications};
    }

    my $RE = join '|', map {quotemeta $_} @ignore_modifications;
    if (my @fatal_modifications = grep {$_ !~ m!^$dir/*(?:$RE)!} @modifications) {
        warn "Your working copy has modifications:\n".Dump(@fatal_modifications);
        $result = 0;
    }

    return $result;
}


=head2 get_lock($conf)

    берем лок в зукипере против повторного запуска update_translations.pl

=cut

sub get_lock
{
    my $conf = shift;
    
    $zkh = Net::ZooKeeper->new(@{$conf->{zk_servers}}) or die "error when connecting to zk_servers: $!\n";
    
    chomp (my $hostname = `hostname -f`);
    my $login = $ENV{LOGNAME} || getpwuid($<) || $ENV{USER};
    my $current_time = strftime "%H:%M", localtime;
    my $lock_data = "host: $hostname\nlogin: $login\ntime: $current_time\n";
    
    unless ($zkh->create($conf->{zk_node}, $lock_data, flags => (ZOO_EPHEMERAL), acl   => ZOO_OPEN_ACL_UNSAFE)) {
        my $get_data = $zkh->get($conf->{zk_node});
        if ($get_data) {
            die "Can't get lock, update_translations.pl is already running:\n$get_data";
        } else {
            die "Can't get lock, unexpected error";
        }
    }    
}
