#!/usr/bin/perl

=head1 NAME

migrate_emails.pl

=head1 DESCRIPTION

Первоначальный экспорт переведенных писем в выделенный проект
Переносим только те переводы, у которых количество абзацев совпадает с оригиналом

Запускается разово, сразу после переноса писем в отдельный проект, вручную

=cut

use strict;
use warnings;

use Config::General;
use Cwd qw/getcwd/;
use Getopt::Long;
use List::MoreUtils qw/each_arrayref/;
use List::Util qw/reduce/;
use Yandex::HashUtils qw/hash_cut hash_merge/;
use Yandex::I18n;
use Yandex::I18nTools;
use Yandex::MailTemplate;
use Yandex::Tanker;

my $BASE_DIR = getcwd();
my %OPT;
GetOptions(
    'base-dir=s' => \$BASE_DIR,
    'config=s' => \$OPT{config},
    'help' => \&usage,
);
my $DEFAULT_CONFIG_LOCATION = "$BASE_DIR/etc/translation.conf";
if (!$OPT{config} && -f $DEFAULT_CONFIG_LOCATION) {
    $OPT{config} = $DEFAULT_CONFIG_LOCATION;
}
die "config file is not defined" unless $OPT{config};
my %conf = Config::General->new($OPT{config})->getall;
$Yandex::I18n::LOCALE_PATH = $conf{locale_path};
%Yandex::I18n::LOCALES = %{hash_cut \%Yandex::I18n::LOCALES, split /\s*,\s*/, $conf{lang}} if $conf{lang};
$Yandex::I18n::PROJECT_ID = $conf{project_id}.'-emails';
$Yandex::I18nTools::AUTHORIZATION = $conf{authorization};
$Yandex::MailTemplate::EMAIL_TEMPLATES_FOLDER = "$BASE_DIR/$conf{email_templates_folder}";
die "$Yandex::MailTemplate::EMAIL_TEMPLATES_FOLDER dir does not exist" unless -d $Yandex::MailTemplate::EMAIL_TEMPLATES_FOLDER;
$Yandex::Tanker::L10N_API_URL = $conf{l10n_api_url} if $conf{l10n_api_url};

my @templates = Yandex::MailTemplate::get_email_template_list_names();

my $keysets = {};
my $default_lang = Yandex::I18n::default_lang();
for my $name (@templates) {
    my $content = get_raw_email_template($name, $default_lang);
    my @paragraphs = grep {$_->{type} eq 'text'} @{Yandex::MailTemplate::get_email_template_paragraphs($content)};

    my @translations = ([map {{$default_lang => $_->{content}}} @paragraphs]);
    for my $lang (Yandex::I18n::get_other_langs()) {
        my $translated_content = get_raw_email_template($name, $lang) or next;
        my @translated_paragraphs = grep {$_->{type} eq 'text'} @{Yandex::MailTemplate::get_email_template_paragraphs($translated_content)};
        if (@translated_paragraphs != @paragraphs) {
            warn "skipping $lang $name\n";
            next;
        }
        push @translations, [map {{$lang => $_->{content}}} @translated_paragraphs];
    }
    my $ea = each_arrayref(@translations);
    while (my @t = $ea->()) {
        my $data = reduce {no warnings 'once'; hash_merge($a, $b)} {}, @t;
        $keysets->{$name}->{Yandex::I18nTools::escape_key($data->{$default_lang})} = $data;
    }
}
my $keysets_xml = Yandex::Tanker::hashref_to_xml($keysets);
Yandex::I18nTools::send_to_l10n_team('master', $templates[0], $keysets_xml, 'no_language');
