#!/usr/bin/perl

=head1 DESCRIPTION

    Скрипт запускается из postinst'а при установке пакета, и генерирует файл ~updater/.ssh/authorized_keys
    Важно! Старый файл удаляется, все ключи каждый раз перезаписываются заново. 

=cut

use strict;
use warnings;

use Getopt::Long;
use List::MoreUtils qw/ uniq /;
use List::Util qw/max/;
use Path::Tiny;
use Pid::File::Flock qw/:auto/;
use Sys::Hostname;
use YAML;

#=======================================================================
# Настройки: ключи + какие из них на каких хостах использовать

my %HOSTS = (
    direct => [qw/ppctest-ts1-mysql
                  ppctest-ts1-front
                  ppctest-ts2-front
                  ppctest-ts3-front
                  ppctest-sandbox-front
                  ppctest-sandbox1-front
                  ppctest-sandbox2-front
                  ppctest-load-front
                  ppcdev1
                  ppcdev2
                  ppcdev3
                  ppcdev4
                  ppcdev5
                  ppcdev6
                  ppcdev7
                  ppctest-cloud-front
                  ppctest-cloud2-front
                  vla1-4273-8609.vm.search.yandex.net
                  vla1-5083-8609.vm.search.yandex.net
                  vla1-5093-8609.vm.search.yandex.net
                  vla1-5102-8609.vm.search.yandex.net
               /],
    directmod => [qw/ppcmoddev1
                     ppcmoddev2
                     direct-mod-test
                     direct-mod-test2
                  /],
    tabula => [qw/ppctabula-front01i
                  ppctabula-front01i.ppc.yandex.ru
                  tabula1.yandex.ru
                  ppctabula-front01f
                  ppctabula-front01f.ppc.yandex.ru
                  tabula2.yandex.ru
                  direct-tabula-man-1.man.yp-c.yandex.net
                  direct-tabula-man-2.man.yp-c.yandex.net
                  direct-tabula-vla-1.vla.yp-c.yandex.net
                  direct-tabula-vla-2.vla.yp-c.yandex.net
                  tabula-dev.man.yp-c.yandex.net
                  ppcdirdev2/],
    direct_mysql => [qw/ppctest-devtest-mysql.ppc.yandex.ru
                        ppctest-dev7-mysql.ppc.yandex.ru
                        ppctest-load-mysql.ppc.yandex.ru
                        ppctest-ts1-mysql.ppc.yandex.ru
                        ppctest-ts2-mysql.ppc.yandex.ru/],
    # группа в кондукторе direct_java_dev_rtc
    direct_java_dev => [qw/man1-4631-21550.vm.search.yandex.net
                           man1-7202-14850.vm.search.yandex.net
                           sas1-5092-21360.vm.search.yandex.net
                           sas1-9028-14430.vm.search.yandex.net
                           man2-0301-man-direct-java-test-23095.gencfg-c.yandex.net
                           man2-0374-man-direct-java-test-23095.gencfg-c.yandex.net
                           man2-0499-man-direct-java-test-23095.gencfg-c.yandex.net
                           vla1-0215-vla-direct-java-test-23065.gencfg-c.yandex.net
                           vla1-0229-vla-direct-java-test-23065.gencfg-c.yandex.net
                           vla1-0862-vla-direct-java-test-23065.gencfg-c.yandex.net
                           /],
    # группа в кондукторе direct_java_test_rtc
    direct_java_test => [qw/sas1-9069-7120.vm.search.yandex.net
                            vla1-0256-vla-direct-java-test-23065.gencfg-c.yandex.net
                            man2-0434-man-direct-java-test-23095.gencfg-c.yandex.net/,
    # группа в кондукторе direct_testnets_java ("новая java-ТС")
                          qw/man1-9616-man-ppc-di-test-27875.gencfg-c.yandex.net
                            man1-9652-man-ppc-di-test-27875.gencfg-c.yandex.net
                            man1-9599-man-ppc-di-test-27875.gencfg-c.yandex.net
                            man1-9639-man-ppc-di-test-27875.gencfg-c.yandex.net
                            vla1-5102-vla-ppc-di-test-18274.gencfg-c.yandex.net
                            vla1-5111-vla-ppc-di-test-18274.gencfg-c.yandex.net
                            vla1-5082-vla-ppc-di-test-18274.gencfg-c.yandex.net
                            vla1-5095-vla-ppc-di-test-18274.gencfg-c.yandex.net
                            /],
);
my @dirinf_group = qw/lena-san yukaba rivik dspushkin pe4kin palasonic/;
my @dmove_stable = qw/
yukaba
pe4kin 

ppc

zhur

ppalex

raketa
sudar
l-jul
sonch

gerdler

pavryabov
buhter
mexicano
xy6er

a-dubov
aleran
andreypav
avzaykov
darkkeks
dimitrovsd
elwood
khuzinazat
kozobrodov
maxlog
mspirit
pavelkataykin
ssdmitriev
zakhar

alkaline
evolkowa
/;

my @CONF = (
    { hosts => $HOSTS{direct},          confs => ['direct.conf'],     users_except_groups => [ 'qa-special-users' ], },
    { hosts => $HOSTS{directmod},       confs => ['directmod.conf'],  users_except_groups => [ 'qa-special-users' ], },
    { hosts => $HOSTS{tabula},          confs => ['tabula.conf'],     users_except_groups => [ 'qa-special-users' ], },
    # jenkins -- имя пользователя на машине с jenkins (используется исторически), robot-irt-jenkins -- на стаффе, ключ у них один
    { hosts => ['ppcdev5', ],  confs => ['direct.conf'],     users => [qw/ jenkins robot-twilight robot-irt-jenkins /] },
    { hosts => ['ppcdev5', ],  confs => ['mediaplan-test.conf'],     users => [qw/ robot-irt-jenkins jenkins /] },
    { hosts => [map {"ppcdev$_"} 1..7], confs => ['dmove-testing.conf'], users_except_groups => [ 'qa-special-users' ], },
    { hosts => [map {"ppcdev$_"} 1..7], confs => ['dmove-prestable.conf'], users_except_groups => [ 'qa-special-users' ], },
    { hosts => [map {"ppcdev$_"} 1..7], confs => ['dmove-stable.conf'], users => \@dmove_stable },
    { hosts => [map {"ppcdev$_"} 1..7], confs => ['dmove-stable-dmp.conf'], users_except_groups => [ 'qa-special-users' ], },
    # TODO отселить правила для mysql в отдельный файл
    { hosts => $HOSTS{direct_mysql},    confs => ['direct.conf'],     users => [qw/zhur/, @dirinf_group] },
    { hosts => ["ppcbuild", "ppcbuild01i"], confs => ['ppcbuild.conf'],   users => [qw/zhur ppalex/, @dirinf_group] },
    { hosts => [map {"ppcdev$_"} 1..7], confs => ['service-control.conf'], users => [qw/zhur ppalex/, @dirinf_group] },
    { hosts => $HOSTS{direct_java_dev}, confs => ['direct.conf'], users_except_groups => [ 'qa-special-users' ], },
    { hosts => $HOSTS{direct_java_test}, confs => ['direct.conf'], users_except_groups => [ 'qa-special-users' ], },
);

my $user = 'updater';
my $auth_keys_file = "/home/$user/.ssh/authorized_keys";

#=======================================================================

my %O = (
    files => [],
    dirs => [],
);

GetOptions(
    "f|file=s@" => \$O{files},
    "d|dir=s@" => \$O{dirs},
    "force"    => \$O{force},
) or die "can't parse options, stop";

#если никаких каталогов и файлов не дано -- используем стандартный 
if ( ! @{$O{dirs}} && !@{$O{files}} ){
    $O{dirs} = ["/etc/ts-updater-keys.d"];
}

for my $d ( @{$O{dirs}} ){
    push @{$O{files}}, map {"".$_} path($d)->children;
}
@{$O{files}} = uniq @{$O{files}};

if( $ARGV[0] eq 'create' && !$O{force} ){
    my $conf_change_time = max map { (stat($_))[9] } @{$O{files}} ;
    my $auth_keys_change_time = (stat($auth_keys_file))[9];
    exit 0 if $auth_keys_change_time > $conf_change_time;
}

my %KEYS;
my %GROUPS;
for my $f ( sort @{$O{files}} ){
    my $short_name = $f =~ s!^.*/!!r;
    $short_name =~ s/\.yaml$//;
    my $group_keys = YAML::LoadFile($f);
    $GROUPS{ $short_name } = [ keys %$group_keys ]; 
    for my $l ( keys %$group_keys ){
        $KEYS{$l} = $group_keys->{$l};
    }
}

my $host = hostname();

if( $ARGV[0] eq 'create' ){
    my @lines;
    push @lines, "#autogenerated by add-del-updater-user-keys.pl";
    push @lines, @{authorized_keys_for_host($host)};
    open(my $fh, ">", $auth_keys_file) or die "Can't open $auth_keys_file: $!";
    print $fh join( "\n", @lines);
    chmod 0644, $auth_keys_file;
} elsif( $ARGV[0] eq 'generate-all' ){
    my @hosts = sort(uniq(map { @{ $_->{hosts} } } @CONF));
    my @lines;
    for my $host (@hosts) {
        push @lines, "# $host";
        push @lines, @{ authorized_keys_for_host($host) };
        print join("\n", @lines, '');
    }
} elsif( $ARGV[0] eq 'remove' ){
    unlink $auth_keys_file;
    ! -e $auth_keys_file or die "can't delete file $auth_keys_file"; 
} else {
    die "unknown parameter $ARGV[0]";
}

exit(0);

sub authorized_keys_for_host {
    my ($host) = @_;
    my %confs_by_login;
    for my $item (@CONF) {
        next unless grep { $_ eq $host } @{ $item->{hosts} };
        my @users;
        if ( exists $item->{users} ){
            @users = @{$item->{users}};
        } elsif ( exists $item->{users_except_groups} ){
            my %except = map { $_ => 1 } @{$item->{users_except_groups}};
            @users = map { @{$GROUPS{$_}} } grep { !$except{$_} } keys %GROUPS; 
        } else {
            die "neither 'users' nor 'users_except_groups' specified";
        }
        push @{ $confs_by_login{$_} }, @{ $item->{confs} } for @users;
    }

    my @lines;
    for my $login ( sort keys %confs_by_login ){
        my @keys = ref $KEYS{$login} eq 'ARRAY' ? @{$KEYS{$login}} : ( $KEYS{$login} );
        for my $key (@keys){
            my $conf_options = join " ", (map { "--conf=/etc/ts-updater/$_" } sort +uniq @{ $confs_by_login{ $login } } );
            push @lines, qq!command="sudo ORIGINAL_LOGIN=$login /usr/local/bin/ts-update.pl $conf_options \$SSH_ORIGINAL_COMMAND" $key!;
        }
    }
    return \@lines;
}
