#!/usr/bin/perl -w

=head1 DESCRIPTION

    Установка пакетов для различных test-update

    Опции 

    --conf <path/to/conf/file1>
        Настроечный файл.
        Можно передавать несколько опций --conf:

        ts-updater.pl --conf /etc/ts-updater/first.conf --conf /etc/ts-updater/second.conf

        В этом случае файлы будут читаться последовательно и будет использовано первое подходящее правило.


    ssh updater@ppctest-ts1-front 1.34458.34715-1

    Если первым позиционным аргументом указать ASYNC, будет запущен direct-async-run с соответствующей командой. Например:

    ssh updater@ppctest-ts1-front ASYNC 1.34458.34715-1

    Спецкоманды для просмотра истории: 

    ssh updater@ppctest-ts1-front upd last
    ssh updater@ppctest-ts1-front upd last 1

=cut

use strict;
use warnings;

use Carp;
use Sys::Hostname;
use Net::Domain qw/hostfqdn/;
use YAML;
use Getopt::Long qw(:config require_order);
use POSIX qw/strftime/;

our $LOGFILE = "/var/log/ts-updater/ts-updater.log";
my $ANSIBLE_MARKS = '/var/spool/ansible/marks';

run() unless caller();

sub run
{
    $|++;

    my $opt = parse_options();

    $ENV{PATH} = "/usr/local/bin:/usr/bin:/bin:/usr/local/sbin:/usr/sbin:/sbin";

    my $run_async = 0;
    if ($ARGV[0] eq 'ASYNC') {
        shift @ARGV;
        $run_async = 1;
    }

    my $cmd = join " ", @ARGV;

    my $to_run = '';
    my $write_log = 1;

    my @marks;
    if (-e $ANSIBLE_MARKS) {
        if (open (my $fh, $ANSIBLE_MARKS)) {
            @marks = map { "ansible-mark-$_" } grep { $_ } map { chomp $_; $_ } <$fh>;
            close ($fh);
        }
    }

    my $host = hostname();
    my $fqdn = hostfqdn();

    for my $conf_file (@{ $opt->{conf_files} }) {
        my $conf = YAML::LoadFile($conf_file);
        my $rules = $conf->{ $host } || $conf->{ $fqdn } || [];
        for my $mark (@marks) {
            if (my $candidate = $conf->{ $mark }) {
                push @$rules, (ref $candidate eq 'ARRAY' ? @$candidate : $candidate);
            }
        }
        for my $rule (map { ref $_ eq 'ARRAY' ? @$_ : $_ } @$rules) {
            if ( $rule->{type} eq 'main_version' 
                && $cmd =~ /^\d+(?:\.\d+)+(?:~[\w\-]+)?-\d+$/ 
            ) {
                $to_run = $rule->{to_run};
                $to_run =~ s/\$VERSION\b/$cmd/g;
                last;
            } elsif ( $rule->{type} eq 'package'
                && $rule->{to_run} !~ /\$VERSION\b/
                && $cmd =~ /^($rule->{package_regexp})$/
            ) {
                my $package = $1;
                $to_run = $rule->{to_run};
                $to_run =~ s/\$PACKAGE\b/$package/g;
                last;
            } elsif ( $rule->{type} eq 'package' 
                && $rule->{to_run} =~ /\$VERSION\b/
                && $cmd =~ /^($rule->{package_regexp})=([\w\-~\.]+)$/
            ) {
                my ($package, $version) = ($1, $2);
                $to_run = $rule->{to_run};
                $to_run =~ s/\$PACKAGE\b/$package/g;
                $to_run =~ s/\$VERSION\b/$version/g;
                last;
            } elsif ( $rule->{type} eq 'cmd' 
                && $cmd =~ /$rule->{cmd_regexp}/
            ) {
                $to_run = $rule->{to_run};
                $to_run =~ s/\$CMD\b/$cmd/g;
                last;
            } elsif ( $rule->{type} eq 'ext_cmd'
                && $cmd =~ /$rule->{cmd_regexp}/
            ) {
                my %repl = %+;
                $write_log=0;
                $to_run = $rule->{to_run};
                while ( my ($key, $val) = each %repl ) {
                    my $count = $to_run =~ s/\$$key\b/$val/g;
                    croak "Key <$key> is unused"  if !$count;
                }
                last;
            } elsif ( $rule->{type} eq 'cocaine_app'
                && $cmd =~ /^($rule->{cocaine_app_regexp})=([\w\.-]+)(\s+[0-9]+)?$/
            ) {
                my ($app, $version, $runtime_port) = ($1, $2, $3);
                my $runtime_opt = '';
                if ($runtime_port) {
                    $runtime_port =~ s/^\s+//;

                    $runtime_opt = "-p $runtime_port";
                }
                $to_run = $rule->{to_run};
                $to_run =~ s/\$APP\b/$app/g;
                $to_run =~ s/\$VERSION\b/$version/g;
                $to_run =~ s/\$OPT\b/$runtime_opt/g;
            } elsif ( 
                $cmd =~ /^upd\s+(.*)\s*$/
            ) {
                my $special_cmd = $1;
                my @tokens = split /\s+/, $special_cmd;
                if ($tokens[0] eq 'last' && @tokens <= 2){
                    my $cnt = $tokens[1] || 10;
                    last unless $cnt =~ /^[\+\-]?\d+$/;
                    $to_run = "tail -n $cnt $LOGFILE";
                    $write_log = 0;
                }
            }
        }
        last if $to_run;
    }
    die "can't find appropriate rule for cmd $cmd for hostname $host\nsearched config files:\n" . (join "\n", @{$opt->{conf_files}}) . "\n" if !$to_run;

    if ($write_log){
        open(my $fh, ">>", $LOGFILE) or die "Can't open $LOGFILE: $!";
        print $fh join("\t", 
            strftime("%Y-%m-%d %H:%M:%S", localtime),
            "$opt->{login}/$opt->{original_login}",
            $cmd,
            $to_run."\n"
        );
    }

    print STDERR "to run: $to_run\n...\n";
    exit system($run_async ? ('direct-async-run', 'start', 'sh', '-c', $to_run) : $to_run)>>8;
}


sub parse_options
{
    my %O = (
    );
    my @conf_files;
    GetOptions (
        "h|help"             => \&usage,
        "conf=s@"            => \@conf_files,
    ) or die $@;

    $O{login} = $ENV{SUDO_USER} || '-';
    $O{original_login} = $ENV{ORIGINAL_LOGIN} || '-';
    die 'at least one conf file should be specified' unless @conf_files;
    $O{conf_files} = \@conf_files;

    return \%O;
}


sub usage {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}
