#!/usr/bin/python

# Provide: walle_host_certificate

import os
import re
import time
import glob
import subprocess
import argparse
import sys
import calendar
import datetime
import socket
import logging
import requests

TMP_DIR = "/run/shm/"
STATUS = {0: "OK", 1:"WARN", 2:"CRIT"}

def report(status, reason=""):
    timestamp = time.time()
    result = {"status":STATUS[status], "timestamp":timestamp, "reason":reason}
    print("PASSIVE-CHECK:walle_host_certificate;{};{}".format(status, result))
    sys.exit(0)

def run_cmd(cmd):
    logging.debug(cmd)
    proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    (out, err) = proc.communicate()
    if err:
        logging.debug(err)
        return (False, err)
    else:
        logging.debug(out)
        return (True, out)

def get_crl(crl_url, crl_path):
    if os.path.exists(crl_path):
        logging.debug('Get modify time for CRL: %s' % crl_path)
        mtime = os.path.getmtime(crl_path)
        mtime_gmt = datetime.datetime.fromtimestamp(mtime).strftime('%a, %d %b %Y %H:%M:%S GMT')
    else:
        mtime_gmt = None
    logging.debug('Modified time CRL: %s' % mtime_gmt)
    logging.debug('Get CRL from %s' % crl_url)
    resp = requests.get(crl_url, headers={'If-Modified-Since': mtime_gmt})
    if resp.status_code == 200:
        last_modified = resp.headers['last-modified']
        logging.debug('Last-Modified CRL: %s' % last_modified)
        last_modified_t = time.mktime(datetime.datetime.strptime(resp.headers['last-modified'], '%a, %d %b %Y %H:%M:%S GMT').timetuple())
        with open(crl_path, 'wb') as crl_file:
            crl_file.write(resp.content)
            logging.debug('Save CRL to %s' % crl_path)
        os.utime(crl_path, (last_modified_t, last_modified_t))
    elif resp.status_code == 304:
        logging.debug('CRL not modified')
    else:
        report(1, 'CRL didn\'t download from %s' % crl_url)


def check_serial_in_crl(crl_path, cert_path):
    (status, serial_txt) = run_cmd('sudo /usr/bin/openssl x509 -noout -serial -in '+cert_path)
    if status:
        serial = serial_txt.split('=')[1]
        logging.debug('Serial: '+serial)
        (status, crl_txt) = run_cmd('/usr/bin/openssl crl -noout -text -inform DER -in '+crl_path)
        if status:
            ret = re.search('Serial Number: ('+serial+').+Revocation Date: (.+) GMT', crl_txt)
            if ret:
                revoke_date = ret.groups()[1]
                report(2, "Revocation Date:"+revoke_date)
        else:
            report(1, 'Error getting Revoke List from %s' % crl_path)
    else:
        report(1, 'Error getting SN from %s' % cert_path)

def check_certificate_date(cert_path, cert_notbefore, cert_notafter):
    (status, date) = run_cmd('sudo /usr/bin/openssl x509 -noout -startdate -enddate -in '+cert_path)
    if not status:
        report(1, "Start/End date not calculated")
    date_list = re.findall("not.+=(.*)\n", date)
    if not date_list:
        report(1, "Start/End date not calculated")
    else:
        current_unixtime = int(time.time())
        start_date = int(calendar.timegm(time.strptime(date_list[0], '%b %d %H:%M:%S %Y %Z')))
        end_date = int(calendar.timegm(time.strptime(date_list[1], '%b %d %H:%M:%S %Y %Z')))
        if current_unixtime > end_date:
            report(2, 'Certificate %s expired' % cert_path)
        if current_unixtime < start_date:
            report(2, 'Certificate %s from future' % cert_path)
        if cert_notbefore and current_unixtime < start_date + cert_notbefore:
            report(1, 'Certificate %s notbefore reached' % cert_path)
        if cert_notafter and current_unixtime > end_date - cert_notafter:
            report(1, 'Certificate %snotafter reached' % cert_path)

def check_subject_in_cert(cert_path):
    fqdn = socket.getfqdn()
    logging.debug('FQDN: '+fqdn)
    (status, subject) = run_cmd('sudo /usr/bin/openssl x509 -noout -subject -in '+cert_path)
    if not status:
        logging.debug(status)
        report(1, "Error getting subject")
    subject_list = re.findall("subject=.*/CN=(.*)", subject)
    logging.debug('Subject: '+subject_list[0])
    if subject_list[0] != fqdn:
        report(1, "Certificate %s has wrong subject: %s" % (cert_path, subject_list[0]))

def main():
    parser = argparse.ArgumentParser(add_help=True)
    parser.add_argument('-c', dest='cert_path', default='/etc/certs/capi.pem', required=True, help="/etc/certs/capi.pem")
    parser.add_argument('--debug', dest='debug', action='store_true', help="additional logging")
    parser.add_argument('--notbefore', dest='cert_notbefore', type=int, required=False, help="86400 (1 day)")
    parser.add_argument('--notafter', dest='cert_notafter', type=int, required=False, help="604800 (1 week)")
    args = parser.parse_args()

    if args.debug:
        logging.basicConfig(format='%(levelname)s[LINE:%(lineno)d]: %(message)s', level = logging.DEBUG)

    if not os.path.exists(args.cert_path):
        report(2,"Not found certificate")

    crl_path = TMP_DIR+"monitor_yandexrcca.crl"
    crl_url = 'http://crls.yandex.ru/YandexRCCA/YandexRCCA.crl'
    try:
        get_crl(crl_url, crl_path)
        check_serial_in_crl(crl_path, args.cert_path)
        check_certificate_date(args.cert_path, args.cert_notbefore, args.cert_notafter)
        check_subject_in_cert(args.cert_path)
    except Exception as e:
        report(1, str(e))
    else:
        report(0, '')

if __name__ == '__main__':
    main()

