#!/usr/bin/perl

=head1 NAME

    Скрипт для бекапа mysql баз.

=head1 DESCRIPTION

    Читает конфиги в /etc/mysql-create-backup.conf.d, и пытается забекапить
    базы /opt/mysql.*. 

    --sync-binlogs
        Копирует только бинлоги в backup_dest/$db-binlogs (по-умолчанию 
        копируются только данные)

    --rsync-adt-opts
        Дополнительные опции rsync, например, --rsync-adt-opts="--bwlimit=20000"

=cut

use strict;
use warnings;

use 5.010;

use Getopt::Long;
use YAML;
use Hash::Merge::Simple qw/ merge /;
use Yandex::Shell;
use File::Temp;
use Pid::File::Flock;
use Data::Dumper;
use Date::Format;
use DBI;
use File::stat;

my $DB_CONF = {
    '_default'  => {
        innodb_recover => 0,
        dump_schema => 0,
        do_lvm_snapshot => 1, # делать ли честный lvm snapshot, или только flush tables with read lock (только для backup_method = mylvmbackup)
        backup_method => 'mylvmbackup', # or local_zfs_snapshot
        backup_src_host_check => 1,
        backup_dest => "none",
        backup_src_host => "none",
        backup_binlogs => 1, # синхронизировать ли бинлоги, если запущен с опцией --sync-binlogs, или просто выйти
        backup_snapshot => 0, # тащить ли снапшот, если запущен с опцией --sync-snapshot ZFS ONLY
        delete_snapshot => 0, # удалять ли снапшот ZFS ONLY
        max_snapshots_num => 7,
        stop_before_snapshot => 1,
        step_backup => 172800, # время через которое обязательно должен сделаться бекап;
        #even_days => 0/1 - выполнять бекапы по четным/нечетным дням
    },
};
my $LOG_PREFIX = "main";
my $RSYNC_COMMON_OPTS = "--archive --numeric-ids --sockopts=SO_SNDBUF=9000000,SO_RCVBUF=9000000 --whole-file --ignore-times --timeout=1800 --contimeout=20";
my $CONF_DIR = "/etc/mysql-create-backup.conf.d";
my $HOSTNAME = yash_qx("hostname -f");
chomp $HOSTNAME;
my $curent_timestamp = time();

my %O = (
    sync_binlogs => 0,
    sync_snapshot => 0,
    rsync_adt_opts => "",
);
GetOptions (
    "h|help"             => \&usage,
    "sync-binlogs"       => \$O{sync_binlogs},
    "sync-snapshot"      => \$O{sync_snapshot},
    "rsync-adt-opts:s"   => \$O{rsync_adt_opts}
) or die $@;

my @db_list;
for my $datadir ( glob '/opt/mysql.*' ) {
    if ( $datadir =~ m{^/opt/mysql\.(.*)$} && -d $datadir ) {
        push @db_list, $1;
    }
}
die "No databases to backup" if not @db_list;

for my $file (glob "$CONF_DIR/*.conf") {
    my $next_conf = YAML::LoadFile($file);
    $DB_CONF = merge $DB_CONF, $next_conf;
}

for my $db (@db_list) {
    my $conf = $DB_CONF->{$db} ? merge($DB_CONF->{_default}, $DB_CONF->{$db}) : $DB_CONF->{_default};
    my @backup_dest = ref($conf->{backup_dest}) eq 'ARRAY' ? @{$conf->{backup_dest}} : ($conf->{backup_dest});
    #TODO: get datadir from mysql
    $conf->{datadir} = "/opt/mysql.$db";
    $conf->{hooksdir} = "/usr/share/mylvmbackup/$db";
    $conf->{instance} = $db;
    my_log("$conf->{instance}");
    $conf->{monitoring} = "/var/spool/mysql-monitor/" . $db . "-status";

    my $even_days = ref($conf->{even_days}) eq 'SCALAR' ? undef : $conf->{even_days};
    my $create_backup = 1;

    # если файл со статусом прошлого выполнения существует, то проверяем не 
    # расходятся ли бекапы больше чем на $step_backup и выполняется ли условие четности дня.
    if ((-f $conf->{monitoring}) && (defined $even_days) && (not $O{sync_binlogs})) {
        $create_backup = 0;
        
        my $sb = stat($conf->{monitoring});
        my $modify_file_age = $sb->mtime;
        
        open my $fh, '<', "/var/spool/mysql-monitor/$db-status";
        my $backup_success = <$fh>; chomp $backup_success;
        my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);

        if ($curent_timestamp - $modify_file_age >= $conf->{step_backup})  {
            $create_backup = 1;
	    my_log("бекап старее заданного значения. Делаем вне зависимости от четности дня.");
        } elsif ($backup_success eq 'FAILURE') {
            $create_backup = 1;
            my_log("прошлый бекап завершился с неудачей. Делаем вне зависимости от четности дня.");
        } elsif ($backup_success eq 'PROGRESS') {
            my_log("прошлый бекап застрял в незавершенном состоянии. Ничего не делаем, выставляем FAILURE.");
            yash_system($conf->{hooksdir} . "/backupfailure");
        } elsif ($yday%2 == $conf->{even_days} ) {
            $create_backup = 1;
        } else {
            my_log("бекап не делается, т.к. четный/нечетный день не совпадает с текущим.");
        }
    }

    if ($conf->{backup_src_host_check} and $HOSTNAME ne $conf->{backup_src_host}) {
        my_log("This db is configured to backup from $conf->{backup_src_host}, not $HOSTNAME. May be no config for $db in $CONF_DIR");
        next;
    }
    unless ( -f "$conf->{datadir}/ibdata" || -f "$conf->{datadir}/data/ibdata" ) {
        my_log("Database $db corrupt");
        next;
    }

    my $do_sync;

    &create_hooks($conf);

    if ($O{sync_binlogs} && $conf->{backup_binlogs}) {
        $do_sync = \&sync_binlogs;
    } elsif ($O{sync_binlogs}) {
        my_log("backup_binlogs option is disabled for $db");
        exit 0;
    } elsif (($conf->{backup_method} eq 'mylvmbackup') && ($create_backup == 1)) {
        $do_sync = \&sync_full_db;
    } elsif ($conf->{backup_method} eq 'local_zfs_snapshot') {
        yash_system("$conf->{hooksdir}/backupprocess");
        if (!$conf->{backup_snapshot}) { 
                @backup_dest = ('local zfs'); #Снапшот делается один раз, если никуда не увозится
                                        }
        $do_sync = \&local_zfs_snapshot;
    } else {
        my_log("no such backup method: " . ($conf->{backup_method} // "undef"));
        next;
    }

    for (@backup_dest) {
        $conf->{backup_dest} = $_;
        my_log("Try to backup $db to $conf->{backup_dest}");
        eval {
	    $do_sync->($conf);
	    if (not $O{sync_binlogs}) { 
	        yash_system("$conf->{hooksdir}/backupsuccess");
	    };
	};
	last if not $@;

	my_log("Failed to back up $db to $_: $@");
	if (not $O{sync_binlogs}) {
	    yash_system("$conf->{hooksdir}/backupfailure");
	};
    };

    if ((not $O{sync_binlogs}) && ($create_backup == 1)) {
        if (!defined(my $pid = fork())) {
            die "cannot fork: $!";
        } elsif ($pid == 0) {
            my $mysqlbackupmds = "/usr/bin/mysql-backup-mds create $db";
            my_log("start $mysqlbackupmds");
            exec("$mysqlbackupmds");
        } else {
            my_log("start child mysql-backup-mds with pid: $pid");
        };
     };
}

sub create_hooks {
    my ($conf) = @_;

    if (not -e $conf->{hooksdir}) {
        mkdir $conf->{hooksdir},0755;
    }

    my $backupfailure_path = $conf->{hooksdir} . "/backupfailure";
    my $backupsuccess_path = $conf->{hooksdir} . "/backupsuccess";
    my $backupprocess_path = $conf->{hooksdir} . "/backupprocess";

    open( FN, '>', $backupfailure_path ) or die;
    print FN "#!/bin/sh -e\n" . 
             "#The script is generated in create_backup.\n" .
             "echo 'FAILURE' > /var/spool/mysql-monitor/" . $conf->{instance} . "-status\n";
    close FN;
    chmod( 0755, $backupfailure_path );

    open( FN, '>', $backupsuccess_path ) or die;
    print FN "#!/bin/sh -e\n" .
             "#The script is generated in create_backup.\n" .
             "echo 'SUCCESS' > /var/spool/mysql-monitor/" . $conf->{instance} . "-status\n";
    close FN;
    chmod( 0755, $backupsuccess_path );

    open( FN, '>', $backupprocess_path ) or die;
    print FN "#!/bin/sh -e\n" .
             "#The script is generated in create_backup.\n" .
             "echo 'PROGRESS' > /var/spool/mysql-monitor/" . $conf->{instance} . "-status\n";
    close FN;
    chmod( 0755, $backupprocess_path );

    my $premount_path = $conf->{hooksdir} . "/premount";
    my $presnapshot_path = $conf->{hooksdir} . "/presnapshot";

    open( FN, '>', $premount_path ) or die;
    print FN "#!/bin/sh\n" .
             "#The script is generated in create_backup.\n\n" .
             "RETVAL=0\n\n" .
             "/bin/sleep 10\n" .
	     "for i in {1..3}; do\n" .
             "/etc/init.d/mysql." . $conf->{instance} . " start\n" .
             "RETVAL=\$?\n\n" .
	     "[ \$RETVAL == 0 ] && break\n" .
	     "done\n" .
             "exit \$RETVAL\n";
    close FN;
    chmod( 0755, $premount_path );

    open( FN, '>', $presnapshot_path ) or die;
    print FN "#!/bin/sh\n" .
             "#The script is generated in create_backup.\n\n" .
             "RETVAL=0\n\n" .
             "/etc/init.d/mysql." . $conf->{instance} . " stop\n" .
             "RETVAL=\$?\n\n" .
             "exit \$RETVAL\n";
    close FN;
    chmod( 0755, $presnapshot_path );
}

sub local_zfs_snapshot {
    Pid::File::Flock->new(name => 'local_zfs_snapshot', debug=>1, dir=>'/tmp');
    $LOG_PREFIX = "local_zfs_snapshot";
    $Yandex::Shell::PRINT_COMMANDS = 1;

    my ($conf) = @_;
    my $db = $conf->{instance};
    my_log("started with config " . Dumper($conf));

    my_log("die if running on master");
    my @lfw_closed = grep { /-> X/ } split /\n/, qx(lfw $db);
    die "lfw is open" unless scalar @lfw_closed;

    my $dbh = db_connect($db);
    my $datadir = @{$dbh->selectcol_arrayref("SHOW VARIABLES LIKE 'datadir'", { Columns=>[2] })}[0];
    my ($zfs_dataset, $mounted_on) = map { chomp; $_ } split / /, qx(df -T $datadir | grep -w zfs | awk '{ print \$1,\$NF }');

    my_log("get repl status for $db with datadir $datadir (dataset $zfs_dataset mounted on $mounted_on)");
    print("get repl status for $db with datadir $datadir (dataset $zfs_dataset mounted on $mounted_on)");
    dbh_do($dbh, "FLUSH TABLES WITH READ LOCK");
    chomp(my $backup_date = qx(date +%Y%m%d_%H%M%S));
    yash_system("lm $db status-json > $mounted_on/$conf->{backup_method}.lm-status.after_flush.json");
    my $date = time2str('%Y%m%d_%H%M%S', time);
    my $pos_file = '/tmp/'.$db.'-'.$date.'_mysql' . '.pos';
    my_log("Save $db state to $pos_file");
    open(my $pos_file_fh, '>', $pos_file) or my_log("Opening $pos_file failed: $!");
    _create_posfile_single($dbh, 'SHOW MASTER STATUS', $pos_file_fh, 'Master');
    _create_posfile_single($dbh, 'SHOW SLAVE STATUS', $pos_file_fh, 'Slave');
    close $pos_file_fh or my_log("Closing $pos_file failed: $!");
    if ($conf->{stop_before_snapshot}) {
        my_log("stop instance $db");
        yash_system("lm $db server-stop -f");
	sleep 60;
    }

    my_log("do zfs snapshot");
    yash_system("zfs snapshot $zfs_dataset\@$backup_date");

    if ($conf->{stop_before_snapshot}) {
        my_log("start instance $db");
        yash_system("lm $db server-start -f --skip-slave-start");
        sleep 120; # база может стартовать после того, как lm отдаст управление, в идеале, нужно дожидаться старта в цикле
        my_log("take status-json from $db");
        yash_system("lm $db status-json > $mounted_on/$conf->{backup_method}.lm-status.after_snapshot.$backup_date.json");
        my_log("start slave $db");
        yash_system("lm $db slave-start");
    } else {
        dbh_do($dbh, "UNLOCK TABLES");
    }
    if ($O{sync_snapshot} || $conf->{backup_snapshot}) { 
        
        sync_snapshot($db, "$zfs_dataset\@$backup_date", "$conf->{backup_dest}/$db", "$conf->{delete_snapshot}", $pos_file);

    }
    my_log("Remove old snapshots.");
    yash_system(q(zfs list -t snap | grep ) . $zfs_dataset . q( | awk '{ print $1 }' | sort | head -n-) . $conf->{max_snapshots_num} . q( | xargs -n 1 -I{} zfs destroy -v {}));

}


sub sync_full_db {
    Pid::File::Flock->new(name => 'sync_full_db', debug=>1, dir=>'/tmp');
    $LOG_PREFIX = "sync_full_db";
    my ($conf) = @_;
    my $db = $conf->{instance};
    my $tmpdir = File::Temp->newdir();

    my $backupdir = "$conf->{backup_dest}/$db"; 
    my_log("Ensure directory $backupdir exists ...");
    yash_system("rsync $RSYNC_COMMON_OPTS /dev/null $backupdir/ >/dev/null 2>&1");

    # ensure empty remote tmp dir
    yash_system("rsync $RSYNC_COMMON_OPTS -a --delete $tmpdir/ $backupdir/tmp/ >/dev/null 2>&1");
    $backupdir .= "/tmp";

    # mark backup as failed before start (backupsuccess handled by mylvmbackup)
    if ( -x "$conf->{hooksdir}/backupprocess" ) {
        yash_system("$conf->{hooksdir}/backupprocess");
    }

    my @mylvmbackup_opts = (
        "--mycnf=/etc/mysql/$db.cnf",
        $conf->{innodb_recover} ? ' --innodb_recover' : (),
        "--socket=/var/run/mysqld.$db/mysqld.sock",
        "--prefix=$db",
        "--backuptype=rsync",
        "--rsync_to_backupdir",
        qq(--rsyncarg=$RSYNC_COMMON_OPTS $O{rsync_adt_opts} --exclude=.rsnap_prot0 --exclude="*-bin.*" --exclude=auto.cnf),
        "--backupdir=$backupdir",
        "--hooksdir=$conf->{hooksdir}",
    );
    if ($conf->{do_lvm_snapshot}) {
        push @mylvmbackup_opts, ("--do_lvm_snapshot", "--lvname=backup-$db", "--skip_flush_tables");
    } else {
        push @mylvmbackup_opts, ("--nodo_lvm_snapshot", "--sourcedir=/opt/mysql.$db");
    }

    my_log("Run /usr/lib/yandex-direct-mysql-create-backup/mylvmbackup.patched " . join " ", @mylvmbackup_opts);
    yash_system('/usr/lib/yandex-direct-mysql-create-backup/mylvmbackup.patched', @mylvmbackup_opts);

    # some db may not have backupsuccess hooks (need default hooks templates)
    my $backup_success = "FAILED";
    if (-f "/var/spool/mysql-monitor/$db-status") {
        open my $fh, '<', "/var/spool/mysql-monitor/$db-status";
        $backup_success = <$fh>; chomp $backup_success;
    }
    if ($backup_success eq "FAILED") {
        yash_system($conf->{hooksdir} . "/backupfailure");
        die "mylvmbackup for $db failed";
    }

    if ( $conf->{dump_schema} ) {
        if ( -S "/var/run/mysqld.$db/mysqld.sock" ) {
            my_log("Do mysqldump --no-data and rsync it to $backupdir/$db-\$(date +\"%Y%m%d\").sql");
            yash_system("mysqldump -u root --no-data --opt -S /var/run/mysqld.$db/mysqld.sock $db > $tmpdir/$db.sql");
            yash_system("rsync $RSYNC_COMMON_OPTS $O{rsync_adt_opts} $tmpdir/$db.sql $backupdir/$db-\$(date +\"%Y%m%d\").sql");
        }
    }

    yash_system("touch $tmpdir/.rsnap_prot0");
    yash_system("rsync $RSYNC_COMMON_OPTS -a $tmpdir/.rsnap_prot0 $backupdir/ >/dev/null 2>&1");
    my_log("Db $db backed up to $backupdir");
}

sub sync_binlogs {
    Pid::File::Flock->new(name => 'sync_binlogs', debug=>1, dir=>'/tmp');
    $LOG_PREFIX = "sync_binlogs";
    my ($conf) = @_;
    my $db = $conf->{instance};
    my $tmpdir = File::Temp->newdir();

    my $binlogs_basename = yash_qx(qq(set -o pipefail; echo "show variables like 'log_bin_basename'" | mysql -S /var/run/mysqld.$db/mysqld.sock -sN | awk '{ print \$2 }'));
    chomp $binlogs_basename;
    $binlogs_basename = $binlogs_basename ? "$binlogs_basename*" : "/opt/mysql.$db/*-bin*";
    my @to_backup = glob $binlogs_basename;
    if (not @to_backup) {
        my_log("no binlogs to backup in $binlogs_basename");
        return;
    }
    
    my $backupdir = "$conf->{backup_dest}/$db-binlogs"; 
    my_log("Ensure directory $backupdir exists ...");
    yash_system("rsync $RSYNC_COMMON_OPTS /dev/null $backupdir/ >/dev/null 2>&1");

    my $cmd = qq(rsync $RSYNC_COMMON_OPTS $O{rsync_adt_opts} $binlogs_basename $backupdir);
    my_log("Start sync $binlogs_basename to $backupdir: $cmd");
    yash_system($cmd);
    my_log("Binlogs $binlogs_basename synced to $backupdir");
}

sub my_log {
    chomp(my $date = qx(date));
    print "$date [$LOG_PREFIX] @_\n";
}

sub sync_snapshot {
    $LOG_PREFIX = "sync_snapshot";
    my ($db, $snapshot_name, $backupdir, $delete_snapshot, $pos_file) = @_;
    my $tmpdir = File::Temp->newdir();
    $backupdir .= '/tmp/';
    my_log("mount $snapshot_name to $tmpdir");
    yash_system("mount -t zfs $snapshot_name $tmpdir");
    my_log("Ensure directory $backupdir exists ...");
    yash_system("rsync $RSYNC_COMMON_OPTS /dev/null $backupdir/ >/dev/null 2>&1");
    my $cmd = qq(rsync $RSYNC_COMMON_OPTS $O{rsync_adt_opts} --exclude=.rsnap_prot0 --exclude="*-bin.*" --exclude=auto.cnf $tmpdir/ $backupdir);
    my_log("Start sync $snapshot_name to $backupdir: $cmd");
    yash_system($cmd);
    my_log("Snapshot $snapshot_name synced to $backupdir");
    yash_system("umount $tmpdir");
    yash_system("touch $tmpdir/.rsnap_prot0");
    yash_system("rsync $RSYNC_COMMON_OPTS $pos_file $backupdir >/dev/null 2>&1");
    my_log("rsync $pos_file to $backupdir");
    yash_system("rsync $RSYNC_COMMON_OPTS $tmpdir/.rsnap_prot0 $backupdir >/dev/null 2>&1");
    my_log("rsync $tmpdir/.rsnap_prot0 to $backupdir");
    if ($delete_snapshot) {
        my_log("Snapshot $snapshot_name will be deleted");
        yash_system("zfs destroy $snapshot_name");
        }
}

sub _create_posfile_single {
	my $dbh = shift; my $query = shift; my $fh = shift; my $pos_prefix = shift;
	my $sth = $dbh->prepare($query) or my_log($DBI::errstr);
	$sth->execute or my_log($DBI::errstr);
	while (my $r = $sth->fetchrow_hashref) {
		foreach my $f (@{$sth->{NAME}}) {
			my $v = $r->{$f};
			$v = '' if (!defined($v));
			my $line = "$pos_prefix:$f=$v\n";
			print $fh $line or my_log("Writing position record failed: $!");
		}
 }
 $sth->finish;
}

sub db_connect {
    my ($dbname) = @_;
    return DBI->connect("DBI:mysql:;mysql_socket=/var/run/mysqld.$dbname/mysqld.sock;mysql_enable_utf8=1", "root", '', {RaiseError => 1});
}

sub dbh_do {
    my ($dbh, @sql) = @_;
    for my $s (@sql) {
        my_log("mysql: $s");
        $dbh->do($s);
    }
}

sub usage {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}
