#!/usr/bin/env perl

use Direct::Modern;
use feature 'say';
use utf8;
use open ':std' => ':utf8';

use Getopt::Long;
use JSON;
use POSIX 'strftime';
use Try::Tiny;

use Yandex::HTTP;


=head1 NAME

=encoding utf-8

direct-check-release-in-aqua.pl - проверка запуска паков в Акве

=head1 SYNOPSIS

    direct-check-release-in-aqua.pl -p api -d 2016-10-28
    direct-check-release-in-aqua.pl -p api --tag mysql_5.7_20161108
    direct-check-release-in-aqua.pl -p api -t mysql_5.7_20161108 -b
    direct-check-release-in-aqua.pl --diff 5844d847e4b05385ac45be2e:58459d23e4b05385ac461522
    direct-check-release-in-aqua.pl -p api -d 2016-12-13 -b --include-not-ready | sort -rn

=head1 DESCRIPTION

    Скрипт, который по захардкоженному набору названий лончей и дате релиза
    идет в Акву и провеяет, что все паки были запущены. Для запущенных паков
    выводит тесты, которые не прошли хотя бы в одном из запусков.

    Набор лончей задается профилем; при запуске сприпта можно выбрать, в каком профиле работаем.

    В результате выводится название теста и ссылка на последний запуск, в котором он сфейлил.

    Параметры запуска:
        -h|--help - эта справка
        -l|--list - выводит список доступных профилей (наборов лончей)
        --last - ищет для профиля последний по дате релиз и выводит все для него
        --list-last - выводит список релизов профиля за последние N дней
        --days-back - задает N для --list-last, если не задано используется константа $DAYS_TO_LOOK_BACK_FOR_LAST_RELEASES
        -d|--date - дата релиза в формате YYYY-MM-DD, будут выбираться запуски в Акве по тэгу direct-release-$DATA
        -t|--tag - полное название тега запуска в Акве, имеет приоритет перед --date
        -p|--profile - профиль (задает набор лончей с которыми работаем,
                например, api - для запусков api, cmd - для бэкэнда, transport - для
                транспорта)
        -b|--brief — краткий формат отчета, только количество упавших сьютов в каждом паке
        -sh|--show_missing — в сочетании с профилем all в конце пишет, какие ещё паки были запущены с указанным тегом,
            но при этом не были проверены скриптом из-за отсутствия в списке
        -n|--no-strict —  не строгий режим, если в рамках тега тест хотябы один
                раз прошел, то более поздние по времени падения будут проигнорированы,
                в противном случае поздние падение перекрывают предыдущее зеленое
                прохождение, сигнализируя тем самым что состояние ухудшилось
        -e|--debug — вывод отладочной информации
        -l|--diff launch_id1:launch_id2 - вывод разницы в сьютах между двумя лончами
        --include-not-ready - влючаем в подсчет прошедших сьютов незавершенные
            паки, удобно чтобы понять как в принципе идут паки, а также для
            правильного подсчета результатов, когда повисли большие паки всего с
            парой незавершенных тестов (перезапускать весь пак дорого)

=head1 EXAPMPLES

  список релизов за последнее время:
  direct-check-release-in-aqua.pl --list-last -p all

  короткий отчет по известной дате, сортируем по кол-ву оставшихся пробем:
  direct-check-release-in-aqua.pl -d 2017-12-20 -p all --brief --sh |sort -nr

  то же самое, по известному тегу:
  direct-check-release-in-aqua.pl -t direct-release-2017-12-20 -p all --brief --sh |sort -nr

  более дотошный короткий отчет -- учитываются прошедшие тесты, даже если отчет еще не готов
  direct-check-release-in-aqua.pl -d 2017-12-20 -p all --brief --sh --include-not-ready |sort -nr

=head1 CODE

=cut

use constant {
    STATUS_FINISHED => 'FINISHED',
    STATUS_REVOKED => 'REVOKED',
    STATUS_FAILED => 'FAILED'
};

my $PROFILES = {
    api_java => [
        'Autotests Direct.API5 AudienceTargets',
        'Autotests Direct.API5 RetargetingLists',
        'Autotests Direct.API5 Clients',
        'Autotests Direct.API5 AgencyClients',
        'Autotests Direct.API5. KeywordsResearch',
        'Autotests Direct.API5. Bids',
        'Autotests Direct.API5. Sitelinks',
        'Autotests Direct.API5. VCards',
        'Autotests Direct.API5. AdExtensions',
        'Autotests Direct.API5. AdGroups',
        'Autotests Direct.API5. BidModifiers',
        'Autotests Direct.API5. Ads',
        'Autotests Direct.API5 Leads',
        'Autotests Direct.API5. Changes',
        'Autotests Direct.API5. Keywords',
        'Autotests Direct.API5. Dictionaries',
        'Autotests Direct.API5. Creatives',
        'Autotests Direct.API5 DynamicTextAdTargets',
        'Autotests Direct.API. Sandbox',
        'Autotests Direct.API5. Campaigns',
    ],
    api => [
        'Autotests Direct API Mediaplan (Media)',
        'Autotests Direct API Mediaplan (Teamleader)',
        'Autotests Direct. API Images',
        'Autotests Direct. API Retargeting',
        'Autotests Direct.API Finance',
        'Autotests Direct.API. Banners',
        'Autotests Direct.API. Campaigns',
        'Autotests Direct.API. Clients',
        'Autotests Direct.API. Sandbox',
        'Autotests Direct.API. Statistics',
        'Autotests Direct.API5. AdImages',
        'Autotests Direct.API5. Campaigns',
        'Autotests Direct.API5. Changes',
        'Autotests Direct.API5 Reports',
        'Autotests Direct.API5 Reports: back-to-back tests',
    ],
    cmd => [
        'Direct Backend',
    ],
    transport => [
        'Autotests Direct B2B BS Transport',
        'Autotests Direct BS Transport Full Export',
        'Autotests Direct B2B Transport Moderation',
        'Autotests Direct BS Transport',
        'Autotests Direct Moderation Transport',
    ],
    intapi => [
        'Autotests Direct INTAPI'
    ],
    java_intapi => [
        'direct java intapi',
    ],
    java_web => [
        'Direct Java Web Api',
    ],
};

$PROFILES->{all} = [@{$PROFILES->{api}}, @{$PROFILES->{cmd}}, @{$PROFILES->{transport}}, @{$PROFILES->{intapi}},];
$PROFILES->{perl_regression_duty} = [@{$PROFILES->{api}}, @{$PROFILES->{cmd}}, @{$PROFILES->{transport}}, @{$PROFILES->{intapi}},];

my $DAYS_TO_LOOK_BACK_FOR_LAST_RELEASE = 10;
my $DAYS_TO_LOOK_BACK_FOR_LAST_RELEASES = 20;

# хеш со всеми паками
my $PACKS;

my $URL = "https://aqua.yandex-team.ru/aqua-api/";

my $days_back = $DAYS_TO_LOOK_BACK_FOR_LAST_RELEASES;
my ($date, $tag, $profile) = ('', '', '');
my ($list, $last, $list_last, $help, $debug, $no_strict, $diff);
my $brief = 0;
my $show_missing = 0;
my $include_not_ready = 0;

Getopt::Long::GetOptions(
    'include-not-ready' => \$include_not_ready,
    'last' => \$last,
    'list-last' => \$list_last,
    'days-back' => \$days_back,
    'b|brief' => \$brief,
    'e|debug' => \$debug,
    'n|no-strict' => \$no_strict,
    'list' => \$list,
    'd|date=s' => \$date,
    'l|diff=s' => \$diff,
    'tag=s' => \$tag,
    'profile=s' => \$profile,
    'help' => \$help,
    'show_missing' => \$show_missing,
);

if ($help) {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION -section EXAPMPLES $0 | pod2text");
    exit(0);
} elsif ($list) {
    say "$_" foreach sort keys %$PROFILES;
    exit(0);
}

die "Usage:\n$0 -p <profile> -d <release_data>\n$0 --list# for profiles list\n" unless $diff || (($date =~ /^\d\d\d\d-\d\d\-\d\d$/ || $tag || $last || $list_last) && $profile);

my $json = JSON->new();
$json->utf8;

if ($diff) {
    my ($launch1, $launch2) = split(':', $diff);

    die "launches for diff should be specified as --diff launchid1:launchid2" unless $launch2;
    print_launches_diff($launch1, $launch2);
} else {
    die "profile $profile not found, check --list" unless exists $PROFILES->{$profile};
    die "no packs configured" unless @{$PROFILES->{$profile}};

    if ($list_last) {
        my $releases = find_last_releases($days_back);
        die "no releases found for $days_back past days" unless @$releases;
        say "Found " . @$releases . " releases for $profile for $days_back past days. Tags:";
        foreach my $release (@$releases) {
            say qq!\t$release->{tag}!;
        }
    } else {
        $PACKS = { map { $_ => undef } @{$PROFILES->{$profile}} };

        $show_missing = 0 unless $profile eq 'all';
        # хеш с паками, которых не хватает в списке $PACKS (а в отчетах они есть)
        my %missing;

        my $launches = $last
            ? find_last_release_launches()
            : ($tag ? get_release_launches_by_tag($tag)->{launches} : get_release_launches_by_date($date)->{launches});
        my $results = parse_launches($launches, \%missing);

        print_not_launched_packs($results);
        print_suites_results($results, $brief);

        if ($show_missing && %missing) {
            local $, = "\n";
            say "Missing packs:";
            say sort keys %missing;
        }
    }
}

exit(0);

sub parse_launches {
    my ($launches, $missing) = @_;

    my $results;
    foreach my $launch (@$launches) {
        my $launch_name = $launch->{pack}{name};

        unless (exists $PACKS->{$launch_name}) {
            # получили отчет про что-то, чего не знаем
            $missing->{$launch_name} = 1;
            next;
        }
        next if $launch->{launchStatus} ne "REPORT_READY" && !$include_not_ready;

        my $projects = $launch->{pack}{projects};
        foreach my $project (@$projects) {
            my $project_title = $project->{title};
            my $rp = $results->{$launch_name}{$project_title} //= {};
            foreach my $suite (@{$project->{launchSuites}}) {
                my $suite_name = $suite->{suite}{name};
                my $id = $suite->{suite}{id};
                my $was_met = exists $rp->{$id} ? 1 : 0;
                my $replaced;
                # нет записей - берем любую, иначе более значимую
                if ( !$was_met || is_parsed_prevail($no_strict, $rp->{$id}, $suite) ) {
                    $replaced = 1;
                    $rp->{$id} = {
                        name => $suite_name,
                        finishArrived => $suite->{finishArrived},
                        status => $suite->{launchStatus},
                        launchId => $suite->{launchId}
                    };

                }
                if ($debug) {
                    my $id_tail = substr($id, -4);
                    warn localtime(int($suite->{finishArrived}/1000))
                        . qq~ launch: $suite->{launchId}\t$suite->{launchStatus}\t\t$suite_name\[..$id_tail\]~
                        . ($was_met ? ( $replaced ? " ..replaced" : "") : " ..new") . "\n";
                }

            }
        }
    }
    return $results;
}

=head2

    Выводим паки, которые мы ожидаем (в соответствии с списокм $PACKS), но для которых нет результата.
    Сюда могут попадать устаревшие паки, которые уже удалили из тестов. Такие надо удалять из $PACKS

=cut
sub print_not_launched_packs {
    my $results = shift;
    foreach my $pack (sort keys %$PACKS) {
        say "(!!!) <$pack> wasn't launched or finished once" unless exists $results->{$pack};
    }
}


=head2

    выводим статистику по запускам

=cut
sub print_suites_results {
    my ($results, $brief) = @_;
    foreach my $pack (sort keys %$results) {
        my $header_printed = 0;
        foreach my $project (sort keys %{$results->{$pack}}) {
            my @failed_suits;
            foreach my $suite_id (sort keys %{$results->{$pack}{$project}}) {
                my $suite = $results->{$pack}{$project}{$suite_id};
                next if $suite->{status} eq 'FINISHED';
                push @failed_suits, $suite;
            }
            if (scalar @failed_suits <= 0){
                # ничего: упавших сьютов нет
                # TODO: может быть, по отдельному ключу показывать и полностью прошедшие паки тоже?
            } elsif ( ! $brief ){
                # полный отчет с именами всех упавших сьютов
                say "$pack - FAILED:";
                for my $s ( @failed_suits ){
                    say qq~\thttps://aqua.yandex-team.ru/#/launch?id=$s->{launchId}\t$s->{name}~;
                }
            } elsif ( $brief ){
                # краткий отчет с количество упавших сьютов в каждом паке
                # может быть, сделать это умолчанием, а полный отчет -- по ключу -v, verbose?
                say "".(scalar @failed_suits)." \tsuites failed in pack '$pack'";
            } else {
                die "";
            }
        }
    }
}

sub is_parsed_prevail {
    my ($no_strict, $stored, $parsed) = @_;
    my @args = ($stored->{status}, $stored->{finishArrived}, $parsed->{launchStatus}, $parsed->{finishArrived});
    if ($no_strict) {
        return need_to_overwrite_suite_status(@args);
    } else {
        return need_to_overwrite_suite_status_strict(@args);
    }
}

sub need_to_overwrite_suite_status_strict {
    my ($stored_status, $stored_time, $parsed_status, $parsed_time) = @_;
    # если режим строгий всегда перезаписываем последним по времени статусом
    # REVOKED игнорируется если есть другие статусы
    # Из двух REVOKED-ов выбираем более поздний
    if ($stored_time < $parsed_time) {
        return 1 unless $stored_status ne STATUS_REVOKED && $parsed_status eq STATUS_REVOKED;
    }

    return 0;
}

sub need_to_overwrite_suite_status {
    my ($stored_status, $stored_time, $new_status, $new_time) = @_;

    if ($stored_time < $new_time) { # новая запись более поздняя
        # выбираем более поздний FINISHED
        return 1 if $new_status eq STATUS_FINISHED;
        # REVOKED обновляем а любую более позднюю запись
        return 1 if $stored_status eq STATUS_REVOKED;

        if ($new_status eq STATUS_REVOKED) {
            # игнорируем новый REVOKED, за исключением когда сохранен ранний REVOKED
            return 0;
        } elsif ($stored_status eq STATUS_FINISHED){
            return 0;
        } else {
            return 1;
        }
    } elsif ($stored_status eq STATUS_REVOKED && $new_status ne STATUS_REVOKED) {
        # Обновляем более ранним статусом только в случае если был запомнен
        # REVOKED и получили не-REVOKED статус
        return 1;
    } elsif ($stored_status ne STATUS_FINISHED && $new_status eq STATUS_FINISHED) {
        # в не строгом режиме одного успеха достаточно, перезаписываем только если уже не записано более позднего успеха
        return 1;
    } else {
        return 0;
    }
}

sub print_launches_diff {
    my ($lid1, $lid2) = @_;

    my $suites1 = get_suites_hash_by_launch_id($lid1);
    my $suites2 = get_suites_hash_by_launch_id($lid2);

    say "$lid1 " . (keys %$suites1) . " suites";
    say "$lid2 " . (keys %$suites2) . " suites";

    print_substract('-', $suites1, $suites2);
    print_substract('+', $suites2, $suites1);
}

sub print_substract {
    my ($t, $h1, $h2) = @_;

    foreach my $k1 (keys %$h1) {
        say "$t $k1" unless exists $h2->{$k1};
    }
}

sub get_suites_hash_by_launch_id {
    my ($launch_id) = @_;

    my $suites_by_name;

    foreach my $p ( @{ get_launch_by_id($launch_id)->{'pack'}{projects} }) {
        foreach my $s ( @{$p->{launchSuites}} ) {
            my $suite_full_name = qq~$p->{id}: $s->{suite}{name}~;
            $suites_by_name->{$suite_full_name} = $_;
        }
    }
    return $suites_by_name;
}

sub get_release_tag_by_date_for_current_profile {
    my $date = shift;
    if ($profile eq 'api_java') {
        return 'direct-release-java_api5-' . $date;
    } else {
        return 'direct-release-' . $date;
    }
}

sub get_release_launches_by_date {
    my $date = shift;
    return get_release_launches_by_tag(
        get_release_tag_by_date_for_current_profile($date)
    );
}

sub get_release_launches_by_tag {
    my $tag = shift;
    return $json->decode(_get('services/launch/tag/' . $tag));
}

sub get_launch_by_id {
    my $id = shift;
    my $data = _get('services/launch/' . $id) or die "launch id: $id not found";
    return $json->decode($data);
}

sub find_last_release_launches {
    my $releases = find_last_releases($DAYS_TO_LOOK_BACK_FOR_LAST_RELEASE, 1);
    die "No launches for $profile releases last $DAYS_TO_LOOK_BACK_FOR_LAST_RELEASE days" unless @$releases;
    my $release = $releases->[0];
    say qq!found release for $release->{date} with tag $release->{tag}!;
    return $release->{launches};
}

=head2 find_last_releases($look_back_days, $limit)

    Look for N<=$limit releases not further than $look_back_days days back.
    if $limit is false, then look for any amount of releases for past $look_back_days days;

=cut

sub find_last_releases {
    my ($look_back_days, $limit) = @_;
    my $days_count=1;
    my $day_seconds = 24*60*60;
    my $releases = [];
    while($days_count <= $look_back_days) {
        my $date = strftime '%Y-%m-%d', (localtime(time() - $days_count*$day_seconds));
        my $tag = get_release_tag_by_date_for_current_profile($date);
        my $result;
        try {
            $result = get_release_launches_by_tag($tag);
        };
        if ($result && exists $result->{launches} && @{$result->{launches}}) {
            push @$releases, { tag => $tag,  date => $date, launches => $result->{launches} } if defined $result;
        }
        last if $limit && @$releases == $limit;
        $days_count++;
    }
    return $releases;
}

sub _get {
    my $response = _call('GET', @_);
    die "failed to get information from aqua: " . $response->status_line unless $response->is_success;
    return $response->decoded_content;
}

sub _call {
    my ($method, $handler, $params) = @_;
    return submit_form($method, "$URL$handler", $params);
}

