#!/usr/bin/perl

use strict;
use warnings;
use File::Basename;

# Common databases - thresholds in hours
my $age_warn_daily = 48;  # hours
my $tmp_warn_daily = 36;  # hours
my $age_crit_daily = 96;  # 4 days

my $age_warn_binlogs = 48; 
my $age_crit_binlogs = 96; # hours

# Weekly backup databases
my @weekly = (qr(april-db.*));

my $age_warn_weekly = 240;  # 10 days
my $tmp_warn_weekly = 24;   # 1 day
my $age_crit_weekly = 360;  # 15 days

# Duplicity backups
my $age_warn_duplicity = 36;   # hours
my $age_crit_duplicity_fullbackup = 240; # 10 days

my $dir_rsnap = "/local/backup/rsnap";
my $dir_dupl = "/local/backup/duplicity";

my (@warn, @crit);

sub modified_before {
  my ($file, $hours) = @_;
  my $age = time() - (stat $file)[9];
  return $age >= $hours*3600;
}

sub is_empty {
  opendir my $dir, $_[0] or die $!;
  return (scalar grep ! /^\.\.?$/, readdir $dir) == 0;
}

## main
-d $dir_rsnap || push @crit, "$dir_rsnap is not a directory";

for my $dbtype_path (glob "$dir_rsnap/*") {
  if (is_empty $dbtype_path) {
    push @crit, "$dbtype_path directory is empty";
    next;
  }

  for my $instance_path (glob "$dbtype_path/*") {
    my $instance = basename($instance_path);

    my ($age_warn, $age_tmp, $age_crit) = ($age_warn_daily, $tmp_warn_daily, $age_crit_daily);
    if (scalar grep { $_ } map { $instance =~ /$_/ } @weekly) {
      ($age_warn, $age_tmp, $age_crit) = ($age_warn_weekly, $tmp_warn_weekly, $age_crit_weekly);
    }

    if ($instance_path =~ /-binlogs$/) {
      ($age_warn, $age_crit) = ($age_warn_binlogs*60, $age_crit_binlogs*60);
      my $binlogs_newer_age_warn = qx(find $instance_path -mmin -$age_warn 2>/dev/null | wc -l);
      my $binlogs_newer_age_crit = qx(find $instance_path -mmin -$age_crit 2>/dev/null | wc -l);
       
      if ($binlogs_newer_age_crit == 0) {
        push @crit, "$dbtype_path $instance: no binlogs for $age_crit_binlogs hours found";
      } elsif ($binlogs_newer_age_warn == 0) {
        push @warn, "$dbtype_path $instance: stalled binlogs backup: > $age_warn_binlogs hours";
      }
      next;
    }

    if (-d "$instance_path/tmp" && modified_before("$instance_path/tmp", $age_tmp)) {
      push @warn, "$dbtype_path $instance: tmp age > $age_tmp hours";
    }

    if (! -d "$instance_path/0000") {
      push @crit, "$dbtype_path $instance: 0000 in not directory";
    }
    elsif (! -f "$instance_path/0000/.rsnap_prot0") {
      if (modified_before("$instance_path/0000", $age_crit)) {
        push @crit, "$dbtype_path $instance: 0000 backup broken: no .rsnap_prot0 and age > $age_crit hours";
      } else {
        push @warn, "$dbtype_path $instance: 0000 backup broken: no .rsnap_prot0";
      }
    }
    else {
      if (modified_before("$instance_path/0000/.rsnap_prot0", $age_crit)) {
        push @crit, "$dbtype_path $instance: 0000 backup age > $age_crit hours";
      }
      elsif (modified_before("$instance_path/0000/.rsnap_prot0", $age_warn)) {
        push @warn, "$dbtype_path $instance: 0000 backup age > $age_warn hours";
      }
    }
  }
}

for my $instance_path (glob "$dir_dupl/*") {
  my $instance = basename($instance_path);
  my ($age_warn, $full_crit) = ($age_warn_duplicity*60, $age_crit_duplicity_fullbackup*60);

  if (is_empty($instance_path)) {
    push @crit, "$instance_path directory is empty";
    next;
  } else {
    my $files_newer_age_warn = qx(find $instance_path -mmin -$age_warn 2>/dev/null | wc -l);
    chomp $files_newer_age_warn;
    my $full_backups_newer_age_crit = qx(find $instance_path -type f -mmin -$full_crit -name 'duplicity-full.*' 2>/dev/null | wc -l);
    chomp $full_backups_newer_age_crit;

    if ($full_backups_newer_age_crit == 0) {
      push @crit, "$instance: duplicity full-backup newer than $age_crit_duplicity_fullbackup hours not found";
    } elsif ($files_newer_age_warn == 0) {
      push @warn, "$instance: stalled duplicity backup age > $age_warn_duplicity hours";
    }
  }
}

if (@crit) {
  print "2;ERRORS: " . join('; ', @crit) . " WARNINGS: " . join('; ', @warn) . "\n";
}
elsif (@warn) {
  print "1;WARNINGS: " . join('; ', @warn) . "\n";
}
else {
  print "0;OK\n";
}
