#!/usr/bin/perl
use strict;
use warnings;
use utf8;

# $Id: remote-log-client-rfs.pl 3326 2014-02-18 09:41:25Z andy-ilyin $

=head1 NAME

remote-log-client-rfs.pl

=head1 SYNOPSIS

    # из-под root
    remote-log-client-rfs.pl monitoring
    remote-log-client-rfs.pl status
    remote-log-client-rfs.pl mount
    remote-log-client-rfs.pl umount

=head1 DESCRIPTION

Скрипт отвечает за монтирование remotefs-логов с серверов.

Списки серверов: /etc/remote-log-client/*.conf
Логи: /var/log/remote-log-client-rfs

http://wiki.yandex-team.ru/direkt/gluster-logs

=cut

use autodie;
use English qw( -no_match_vars );
use Fcntl qw( :flock );
use File::Basename qw( basename );
use File::Slurp qw( read_file write_file );
use File::Path qw(make_path);
use Guard;
use IO::Handle;
use JSON;
use List::MoreUtils qw( any );
use POSIX qw( strftime );
use Readonly;
use Try::Tiny;
use LWP::Simple;
use Data::Dumper;
use Socket;

use Yandex::Log;
use Yandex::Shell;

Readonly my $LOCK_FILE_NAME  => '/var/run/remote-log-client-rfs.pid';
Readonly my $STATE_FILE_NAME => '/var/spool/remote-log-client-rfs/status';
Readonly my $MOUNT_BASE      => '/mnt/remote-log-rfs';
Readonly my $CONFIG_DIR      => '/etc/remote-log-client';
Readonly my $TIMEOUT_CMD     => 'timeout -k 20 15';
Readonly my $REMOTEFS_CMD   => 'mount.rfs';
Readonly my $REMOTEFS_OPTS => '-o allow_other,nonempty,_netdev,ro';
Readonly my $REMOTE_MOUNT_POINT   => '/var/remote-log';

Readonly my $CMD_FILTER =>
    q{ 2>&1 | perl -MPOSIX=strftime -npe 'print strftime("%Y-%m-%d\\t%H:%M:%S\\t", localtime)' };

$Yandex::Log::LOG_ROOT = '/var/log/remote-log-client-rfs';

my ( @mounted_directories_cache, $mounted_directories_cached );
my %group_for_host;

my $result = run();
exit $result;

=head1 SUBROUTINES/METHODS

=head2 get_lock

=cut

sub get_lock {
    open my $fh, '>', $LOCK_FILE_NAME;

    my $locked = flock $fh, LOCK_EX | LOCK_NB;
    die "Couldn't lock $LOCK_FILE_NAME" unless $locked;

    $fh->autoflush;

    print $fh "$PROCESS_ID\n";

    return $fh;
}

=head2 release_lock

=cut

sub release_lock {
    my ($fh) = @_;

    close $fh;
    unlink $LOCK_FILE_NAME;
}

=head2 get_command_log_filename

=cut

sub get_command_log_filename {
    my $date = strftime( '%Y-%m-%d', localtime );
    return "$Yandex::Log::LOG_ROOT/commands.log.$date";
}

=head2 get_state

=cut

sub get_state {
    my $content = read_file($STATE_FILE_NAME);
    return decode_json($content);
}

=head2 save_state

=cut

sub save_state {
    my ($state) = @_;

    my $content = encode_json($state);
    write_file( $STATE_FILE_NAME, $content );
}

=head2 get_all_mountnames

=cut

sub get_all_mountnames {
    my @conffiles = glob("$CONFIG_DIR/*.conf");

    my @mountnames;
    foreach my $conffile (@conffiles) {
        foreach my $line ( read_file($conffile) ) {
            chomp $line;

            # пропускаем пустые строки и комментарии
            next if $line =~ /^\s*$/ || $line =~ /^\s*#/;

            # удаляем начальные и концевые пробелы
            $line =~ s/^\s+//;
            $line =~ s/\s+$//;

	    # если группа, то определяем хосты в ней
	    if ( $line =~ "\^%.*" ) {
		$line =~ s/^%//;
		my $hosts = get("https://c.yandex-team.ru/api/groups2hosts/$line");
		foreach my $host ( split /\n/, $hosts ) {
            # если хост в двух группах из конфига, то он примонтируется в последнюю из них
            # можно переделать на хранение массива групп, но тогда нужно переделывать и get_mount_status
            $group_for_host{$host} = $line;
			push @mountnames, $host;
			}
		next;
		}
            push @mountnames, $line;
        }
    }

    return @mountnames;
}

=head2 get_mounted_directories

=cut

sub get_mounted_directories {
    unless ($mounted_directories_cached) {
        my $mount_output = yash_qx('mount');

        foreach my $line ( split /\n/, $mount_output ) {
            if ( $line =~ m{ on (\S+) type fuse\.rfs} ) {
                push @mounted_directories_cache, $1;
            }
        }
    }

    $mounted_directories_cached = 1;
    return @mounted_directories_cache;
}

=head2 reset_mounted_directories_cache

=cut

sub reset_mounted_directories_cache {
    undef $mounted_directories_cached;
    @mounted_directories_cache = ();
}

=head2 get_mount_status

=cut

sub get_mount_status {
    my ($mountname) = @_;

    my $dirname = "$MOUNT_BASE/$mountname";
    if ( $mountname =~ "(gencfg-c|yp-c).yandex.net\$" ) {
    my $group = $group_for_host{$mountname};
    if (!$group) {
        my @conductor_ans = decode_json(get("https://c.yandex-team.ru/api/hosts/$mountname?format=json"));
#	print Dumper($conductor_ans[0][0]->{group});
        $group = $conductor_ans[0][0]->{group};
    }
	$dirname = "$MOUNT_BASE/$group/$mountname";
    }
    #unless ( any { $_ eq $dirname || $_ =~ /^$dirname(\/|$)/ } get_mounted_directories() ) {
    unless ( any { m!^$dirname(/|$)! } get_mounted_directories() ) {
        return 'not_mounted';
    }

    my $mount_stalled = 0;
    try {
        yash_system("$TIMEOUT_CMD stat -t $dirname >/dev/null 2>/dev/null");
    } catch {
        $mount_stalled = 1;
    };

    if ($mount_stalled) {
        return 'stalled';
    }

    return 'mounted';
}

=head2 do_mount
=head2 do_unmount

Процедуры do_* собственно делают действия, которые нужно - монтируют или отмонтируют каталог.
Это всего лишь небольшие обёртки вокруг yash_system($cmd).

=cut

sub do_mount {
    my ($mountname) = @_;

    my $dirname = "$MOUNT_BASE/$mountname";

    my $hostname = $mountname;

    # не на всех серверах resolv.conf позволяет разрешить имена вроде ci01d как ci01d.yandex.ru;
    # заставлять всех настраивать сервер не стоит, лучше решим эту задачу за пользователя
    unless ( $hostname =~ /\./ ) {
        $hostname = "$hostname.yandex.ru";
    }

	my $command_log = get_command_log_filename();
#для рантайма делаем отдельную магию 
            if ( $hostname =~ "(gencfg-c|yp-c).yandex.net\$" ) {
        my $group = $group_for_host{$hostname};
        if (!$group) {
		my @conductor_ans = decode_json(get("https://c.yandex-team.ru/api/hosts/$hostname?format=json"));
		die if !@conductor_ans;
		$group = $conductor_ans[0][0]->{group};
        }
		my @REMOTE_MOUNT_POINTS = split /\n/,yash_qx("$REMOTEFS_CMD", ("-l","$hostname")); 
		$dirname = "$MOUNT_BASE/$group/$hostname/";
	    	unless ( -d $dirname ) {
	        	make_path($dirname);
    					}
		foreach my $remote_mount_point_runtime (@REMOTE_MOUNT_POINTS) {
			next if $remote_mount_point_runtime !~ m!^/.*!;
			$remote_mount_point_runtime =~ s/\s.*$//;
			my $local_mount_point_runtime = $remote_mount_point_runtime;
			$local_mount_point_runtime =~ s/^\/|\s.*$//;
			$local_mount_point_runtime =~ tr/\//_/;
			$dirname .= $local_mount_point_runtime;
	        	make_path($dirname);
			yash_system("$TIMEOUT_CMD $REMOTEFS_CMD $hostname:$remote_mount_point_runtime $dirname $REMOTEFS_OPTS $CMD_FILTER >> $command_log");
			$dirname = "$MOUNT_BASE/$group/$hostname/";
		}
            } else {
		    unless ( -d $dirname ) {
        		mkdir $dirname;
    					}
    	    		yash_system("$TIMEOUT_CMD $REMOTEFS_CMD $hostname:$REMOTE_MOUNT_POINT $dirname $REMOTEFS_OPTS $CMD_FILTER >> $command_log");
		   }
}

sub do_unmount {
    my ( $mountname, %opts ) = @_;

    my $dirname = "$MOUNT_BASE/$mountname";
    my $command_log = get_command_log_filename();

    if ( $opts{force} ) {
        yash_system("umount -l $dirname $CMD_FILTER >> $command_log");
    } else {
        yash_system("$TIMEOUT_CMD umount $dirname $CMD_FILTER >> $command_log");
    }
}

=head2 run_monitoring
=head2 run_status
=head2 run_mount
=head2 run_umount

Процедуры run_* - это ветки выполнения процедуры run(); run() разбирается, что хотел пользователь,
и вызывает одну из этих четырёх, чтобы выполнить конкретное действие.

=cut

sub run_monitoring {
    my $state = get_state();

    my $stalled_mounts = $state->{stalled};
    if ( $stalled_mounts && @$stalled_mounts ) {
        print "1; Some mounts are stalled. See service remote-log-client-rfs status\n";
    } else {
        print "0; OK\n";
    }

    return 0;
}

sub run_status {
    my $state = get_state();

    my $ok_mounts      = $state->{ok}      || [];
    my $stalled_mounts = $state->{stalled} || [];

    print "ok_mounts:\n";
    foreach my $mountname (@$ok_mounts) {
        print "$mountname\n";
    }

    print "stalled_mounts:\n";
    foreach my $mountname (@$stalled_mounts) {
        print "$mountname\n";
    }

    return 0;
}

sub run_mount {
    my $lock_fh = get_lock();
    scope_guard {
        release_lock($lock_fh);
    };

    my $log = Yandex::Log->new( log_file_name => 'client.log', date_suf => '%Y%m%d' );
    $log->out('start');
    $log->out('mount');

    my ( @ok_mounts, @stalled_mounts );

    my @mountnames = get_all_mountnames();
    foreach my $mountname (@mountnames) {
       try {
            my $mount_status = get_mount_status($mountname);

            if ( $mount_status eq 'not_mounted' ) {
                $log->out("$mountname not mounted, mounting");
                do_mount($mountname);
            } elsif ( $mount_status eq 'stalled' ) {
                $log->out("$mountname stalled, mounting");
                do_unmount( $mountname, force => 1 );
                do_mount($mountname);
            }
        } catch {
            $log->out("error handling $mountname: $_");
        };
    }

    reset_mounted_directories_cache();

    my $state = { ok => [], stalled => [] };
    foreach my $mountname (@mountnames) {
        try {
            my $mount_status = get_mount_status($mountname);

            if ( $mount_status eq 'mounted' ) {
                push @{ $state->{ok} }, $mountname;
            } else {
                push @{ $state->{stalled} }, $mountname;
            }
        } catch {
            $log->out("error collecting status of $mountname: $_");
        };
    }

    save_state($state);

    my %is_mountname = map { $_ => 1 } @mountnames;
    my %is_mounted   = map { $_ => 1 } get_mounted_directories();

    foreach my $entry ( glob("$MOUNT_BASE/*") ) {
        my $entry_basename = basename($entry);

	my ( $err, @res ) = Socket::getaddrinfo( "$entry_basename", "", { 'protocol' => Socket::IPPROTO_TCP, 'family' => Socket::AF_INET6 } );
	if ( $err ) {
		my $hosts = get("https://c.yandex-team.ru/api/groups2hosts/$entry_basename");
                next if $hosts;
	} else {
#    		print Dumper($entry_basename);
        next if $is_mountname{$entry_basename};
	}
        try {
            $log->out("cleaning up $entry");

            unless ( -d $entry ) {
                die "cannot clean up $entry: is a file";
            }

            if ( $is_mounted{$entry} ) {
                $log->out("unmounting $entry");
                do_unmount( $entry_basename, force => 1 );
            }

            rmdir $entry;

            if ( -d $entry ) {
                die "failure cleaning up $entry: persists after rmdir";
            }
        } catch {
            $log->out("error cleaning up $entry: $_");
        };
    }

    $log->out('finish');
    return 0;
}

sub run_umount {
    my $lock_fh = get_lock();
    scope_guard {
        release_lock($lock_fh);
    };

    my $log = Yandex::Log->new( log_file_name => 'client.log', date_suf => '%Y%m%d' );
    $log->out('start');
    $log->out('umount');

    my ( @ok_mounts, @stalled_mounts );

    my @mountnames = get_all_mountnames();
    foreach my $mountname (@mountnames) {
        try {
            my $mount_status = get_mount_status($mountname);

            if ( $mount_status eq 'mounted' ) {
                $log->out("$mountname mounted, unmounting");
                do_unmount($mountname);
            } elsif ( $mount_status eq 'stalled' ) {
                $log->out("$mountname stalled, forcing an umount");
                do_unmount( $mountname, force => 1 );
            }
        } catch {
            $log->out("error handling $mountname: $_");
        };
    }

    $log->out('finish');
    return 0;
}

=head2 run

=cut

sub run {
    my ($action) = @ARGV;

    if ( !$action || @ARGV > 2 ) {
        print "Usage: $0 monitoring|status|mount|umount\n";
        return 1;
    }

    if ( $action eq 'monitoring' ) {
        return run_monitoring();
    } elsif ( $action eq 'status' ) {
        return run_status();
    } elsif ( $action eq 'mount' ) {
        return run_mount();
    } elsif ( $action eq 'umount' ) {
        return run_umount();
    } else {
        die "Invalid action $action\n";
    }
}
