#!/usr/bin/env python2
# -*- coding: utf-8 *-*
"""
Generate rfsd exports file from multiple files in directory.
Resolve groups in conductor to IPs.
Use cache if conductor is hanging.
Reload service if fiile changed.

Groups in acl entries should have prefix '%'.

Wiki: https://wiki.yandex-team.ru/users/techpriest/yandex-du-rfsd
"""

import logging
import os
import pprint
import re
import socket
import sys
import urllib2

from glob import glob
from optparse import OptionParser
from subprocess import call

CONFIG_DIR = '/etc/rfs-exports.d'
CONFIG_FILE = '/etc/rfs-exports'
SWAP_DIR = '/var/cache/yandex-du-rfsd'
# URL_RESOLVE_GROUP = 'http://c.yandex-team.ru/api/groups2hosts/%s'
URL_RESOLVE_GROUP = 'http://c.yandex-team.ru/api-cached/groups2hosts/%s'
SERVICE_RELOAD_COMMAND = '/usr/sbin/service rfsd restart >/dev/null 2>&1'


def parse_config(filename):
    """Parse config file and get listed items.
    Return: [ (dir, [acl_entry, acl_entry], options), ... ]
    Example config line:
    /tmp 127.0.0.1, alex, 10.0.0.2 (ro,user=alex)"""
    result = []
    lines = file(filename).read().splitlines()
    # Remove trailing and leading spaces, empty lines, comments
    lines = [line.strip() for line in lines if len(line) > 0 and line[0] != '#']
    for line in lines:
        path = re.match('^(\\S+)\\s+.*', line)
        if path is None:
            continue
        else:
            path = path.groups()[0]
            line = re.sub('^\\S+\\s+', '', line)
        options = re.match('.*(\\(.*\\))', line)
        if options is None:
            options = ''
        else:
            options = options.groups()[0]
            line = re.sub('\\(.*\\)$', '', line)
        line = line.replace(',', ' ')
        acl_entries = line.split()
        result.append([path, acl_entries, options])
    return result


def get_host_ips(host):
    result = []
    try:
        host_ips = socket.getaddrinfo(host, 80, 0, socket.SOCK_STREAM)
        result = [x[4][0] for x in host_ips]
    except socket.error as e:
        pass
    return result


def get_conductor_group_hosts_by_ip(group, url, swap_dir, log):
    """Get conductor group hosts and resolve to ip.
    Get values from cache files in swap dir, if conductor fails.
    Return: (conductor_query_success, [ip1, ip2, ...])"""
    log.debug('Getting hosts in group %s' % group)

    conductor_query_success = False
    group_file_name = os.path.join(swap_dir, group)
    group_ips = []
    response = None

    try:
        response = urllib2.urlopen(url % group)
        conductor_query_success = True
    except urllib2.HTTPError as e:
        if e.code == 404:
            return True, []

    # Conductor query ok
    if conductor_query_success:
        # Get IPs for every host
        hosts = response.read().splitlines()
        # For independence of hosts order in conductor response
        hosts.sort()
        for host in hosts:
            host_ips = get_host_ips(host)
            group_ips.extend(host_ips)
        # Update cache file if group changed
        data = '\n'.join(group_ips)
        group_file = os.path.join(swap_dir, group)
        if os.path.isfile(group_file):
            cached_data = file(group_file).read()
        else:
            cached_data = ''
        if data != cached_data:
            log.info('Group %s changed. Updating cache file.' % group)
            open(group_file_name, 'w+').write(data)
    # Conductor query failed
    else:
        if os.path.isfile(group_file_name):
            group_ips = file(group_file_name).read().splitlines()
            log.info('Conductor query failed. Return cached results for group %s' % group)
        else:
            log.warn('Conductor query failed and no cached result found for group %s' % group)

    return conductor_query_success, group_ips


def main():
    # Get command-line options
    opt_parser = OptionParser()
    opt_parser.add_option("-c", "--config-dir", dest="config_dir", help="Configuration files directory", type="string",
                          default=CONFIG_DIR)
    opt_parser.add_option("-u", "--url-resolve-group", dest="url_resolve_group",
                          help="URL for resolving group hosts. Use %s for group name", type="string",
                          default=URL_RESOLVE_GROUP)
    opt_parser.add_option("-f", "--config-file", dest="config_file", help="Resulting configuration file", type="string",
                          default=CONFIG_FILE)
    opt_parser.add_option("-s", "--swap-dir", dest="swap_dir", help="Directory for caching results", type="string",
                          default=SWAP_DIR)
    opt_parser.add_option("-r", "--reload-command", dest="reload_command", help="Command to reload rfsd service",
                          type="string", default=SERVICE_RELOAD_COMMAND)
    opt_parser.add_option("-v", "--verbose", dest="verbose", help="Verbose messages", action="store_true",
                          default=False)
    opt_parser.add_option("-d", "--debug", dest="debug", help="Debug messages", action="store_true", default=False)
    (opts, cmd_args) = opt_parser.parse_args()

    # Enable logging
    logging.basicConfig()
    log = logging.getLogger('rfsd-generate-acls')
    if opts.debug:
        log.setLevel(logging.DEBUG)
    elif opts.verbose:
        log.setLevel(logging.INFO)
    else:
        log.setLevel(logging.WARNING)

    # Find config files
    conf_files = []
    if os.path.isdir(opts.config_dir):
        for filename in glob(os.path.join(opts.config_dir, '*.conf')):
            if os.path.isfile(filename):
                conf_files.append(filename)

    if len(conf_files) == 0:
        log.info('No configs found in %s. Terminating' % opts.config_dir)
        sys.exit(0)

    log.debug('Found config files:' + str(conf_files))

    # Parse config files
    acls = []

    for conf in conf_files:
        try:
            log.debug('Parsing config file %s' % conf)
            result = parse_config(conf)
            acls.extend(result)
        except Exception, e:
            log.warning('Failed to process config %s' % conf)
            log.warning(str(e))

    log.debug('Parsed ACLs:\n' + pprint.pformat(acls))

    # Resolve group names to hosts IPs
    for i in range(len(acls)):
        groups = acls[i][1]
        groups_resolved = []
        for gr in groups:
            # conductor group name
            if gr[0] == '%':
                # print(gr[1:] + ' ' + opts.url_resolve_group + ' ' + opts.swap_dir + ' ' + str(log))
                groups_resolved.extend(
                    get_conductor_group_hosts_by_ip(gr[1:], opts.url_resolve_group, opts.swap_dir, log)[1])
            elif gr[0] == '*':
                groups_resolved.append(gr)
            # user@host
            elif '@' in gr:
                groups_resolved.append(gr)
            # hostname - resolve to IP
            else:
                groups_resolved.extend(get_host_ips(gr))
        acls[i][1] = groups_resolved

    log.debug('Resolved ACLs:\n' + pprint.pformat(acls))

    # Generate rfsd config
    new_conf = []
    for acl in acls:
        if len(acl[1]) == 0:
            log.info('Skip empty acl: ' + pprint.pformat(acl))
        else:
            line = acl[0] + ' ' + ','.join(acl[1]) + ' ' + acl[2] + '\n'
            new_conf.append(line)

    # Read old config
    old_conf = file(opts.config_file).readlines() if os.path.exists(opts.config_file) else list()

    # If config changed - write new and reload service
    if new_conf == old_conf:
        log.info('No config changes - nothing to do')
    else:
        log.info('Writing changed config')
        open(opts.config_file, 'w+').writelines(new_conf)
        log.info('Reloading rfsd service')
        call(opts.reload_command, shell=True)


if __name__ == "__main__":
    main()
