#!/usr/bin/env python

from subprocess import Popen, PIPE
import syslog
from telnetlib import *
import socket
import time
import pyudev
import optparse

DEBUG = 0
SENDER_HOST = '127.0.0.1'
SENDER_PORT = '42000'
FQDN = socket.getfqdn().replace('.', '_')
PREFIX = 'one_min'

def logger(err):
    syslog.syslog(syslog.LOG_ERR | syslog.LOG_SYSLOG, str(err))

def getBlockDevices():
    context = pyudev.Context()
    devices = context.list_devices(subsystem="block", sys_name="sd[a-zA-Z]*")
    for device in devices:
        disk_type = device.get('DEVTYPE', 'partition').lower()
        if not disk_type.count('partition'):
            yield device.get('DEVNAME')

def getSmartDevice(device_name, debug=DEBUG):
    try:
       smart_device_dict = dict()
       if debug: logger(device_name)
       device_name_short = device_name.split('/')[-1]
       smart_device_dict.setdefault(device_name_short, dict())
       cmd = '/usr/sbin/smartctl -A %s' % device_name
       command = Popen( cmd, stdout=PIPE, stdin=None, stderr=PIPE, bufsize=0, shell=True, preexec_fn=None)
       _outstd, _errstd = command.communicate()
       prefix, smart_stat = _outstd.split("RAW_VALUE\n", 1)
       if _errstd: raise ValueError(str(_errstd))
       if debug: logger(smart_stat)
       smart_value_list = smart_stat.rstrip('\n\n').rsplit('\n')
       smart_value_list = ['\t'.join(_line.split()).rsplit('\t') for _line in smart_value_list]
       if debug: logger(smart_value_list)

       attrib_names = ['VALUE', 'WORST', 'THRESH', 'RAW_VALUE']
       smart_value_list = [ dict( [( _line[1], dict( zip(attrib_names, [_line[3], _line[4], _line[5], _line[9]]) ) )] ) for _line in smart_value_list ]
       [ smart_device_dict[device_name_short].update(smart_value) for smart_value in smart_value_list ]
       if debug: logger(smart_device_dict)
    except  Exception as err:
       logger(err)
       if debug: raise
    return smart_device_dict

def sendGraphite( smart_device_dict, sender_host=SENDER_HOST, sender_port=SENDER_PORT, 
                  prefix=PREFIX, fqdn=FQDN, debug=DEBUG):
    try:
        carbon = Telnet(sender_host, sender_port)
        timestamp = int(time.time())
        for device_name in smart_device_dict:
            for key_name in smart_device_dict[device_name]:
                for attrib_name in smart_device_dict[device_name][key_name]:
                    try:
                        smart_value = int(smart_device_dict[device_name][key_name][attrib_name])
                        result_string = '{0}.{1}.{2}.{3}.{4}.{5} {6} {7}\n'.format( prefix, fqdn, 'smart',  
                                                                                    device_name, key_name, 
                                                                                    attrib_name, smart_value,
                                                                                    timestamp) 
                        if debug: logger(result_string)
                        carbon.write(result_string)
                    except ValueError as err:
                        logger(err) 
    except Exception as err:
        logger(err)
        raise

if __name__ == '__main__':

    usage = "usage: smartmon.py "
    parser = optparse.OptionParser(usage=usage)
    parser.add_option("--debug", "-d", action="store_true", dest="debug",
                      default=DEBUG, help="Enable debug")
    parser.add_option("--prefix", action="store", dest="prefix", 
                      default=PREFIX, help="Prefix for graphite")
    parser.add_option("--fqdn", action="store", dest="fqdn",
                      default=FQDN, help="Server hostname")
    parser.add_option("--sender-host", action="store", dest="sender_host", 
                      default=SENDER_HOST, help="Graphite sender host")
    parser.add_option("--sender-port", action="store", dest="sender_port", 
                      default=SENDER_PORT, help="Graphite sender port")
    (options, args) = parser.parse_args()

    
    smart_result_dict = dict()
    for device_name in getBlockDevices():
        sendGraphite(getSmartDevice(device_name, options.debug), 
                     options.sender_host, options.sender_port, 
                     options.prefix, options.fqdn, options.debug)
