package Yandex::ABC;

=head1 NAME

    Yandex::ABC
    Perl обертка для API ABC: https://abc-back.yandex-team.ru/api/v4/

=head1 DESCRIPTION

    Для работы необходим OAuth токен:
    our $ABC_OAUTH_TOKEN;
    Получить OAuth токен можно здесь: https://wiki.yandex-team.ru/intranet/dev/oauth

=cut

use base qw/Exporter/;
use strict;
use warnings;
use utf8;

use Data::Dumper qw( Dumper );
use Carp qw( cluck confess );

use JSON;
use HTTP::Headers;
use HTTP::Request;
use LWP::UserAgent;
use Yandex::HTTP;

our @EXPORT = qw();
our @EXPORT_OK = qw(get_data_from_abc);

our $ABC_OAUTH_TOKEN;
our $ABC_API_URL ||= 'https://abc-back.yandex-team.ru/api/v4/';

use constant MAX_PAGE_SIZE => 1000; # максимальное количество объектов на странице
use constant DEFAULT_PATH_TO_TOKEN_FILE => "/etc/direct-tokens/startrek"; # токен для startrek подходит и для ABC


sub load_token_from_file {
    my $path_to_token = shift;
    $path_to_token ||= DEFAULT_PATH_TO_TOKEN_FILE;

    open(my $fh, "<", $path_to_token) || ($path_to_token eq DEFAULT_PATH_TO_TOKEN_FILE ? return : confess "Can't open $path_to_token: $!");
    my $token = <$fh>;
    $ABC_OAUTH_TOKEN = $token;
    close($fh) || confess "Can't close $path_to_token: $!";
}


BEGIN {
    load_token_from_file();
}


=head2 get_data_from_abc($method, %params)

    Небольшая функция обертка для получения данных из api ABC.
    Принимает метод, к которому надо обратиться в api (services, services/members, roles и т.д.) и
    хэш параметров для данного метода (посмотреть список можно здесь https://wiki.yandex-team.ru/intranet/abc/api/).
    Возвращает ссылку на массив с полученными данными.
    Если при работе функции произошла ошибка, то ругнется в STDERR и вернёт undef.

=head2 SYNOPSIS

    Пример использования:
    my $data = get_data_from_abc("services/members", service__slug => "direct", fields => "person.login,service.slug");

    my $data = get_data_from_abc("duty/on_duty", service__slug => "direct", schedule__slug => "dna-autotest-duty");

=cut

sub get_data_from_abc {
    my ($method, %params) = @_;
    return () unless (%params || $method);

    if (substr $method, -1 ne "/") {
        $method .= "/";
    }
    
    my @result_data = ();

    unless ($ABC_OAUTH_TOKEN) {
        confess "Not set global variable \$ABC_OAUTH_TOKEN!\n";
    }
    
    $params{page_size} = MAX_PAGE_SIZE unless exists $params{page_size};
    
    my $url = Yandex::HTTP::make_url("$ABC_API_URL$method", \%params);
    my $req = HTTP::Request->new(
        'GET' => $url,
        HTTP::Headers->new('Authorization' => 'OAuth '.$ABC_OAUTH_TOKEN),
    );

    my $ua_response = _http_get($req);
    my $data = eval {JSON->new->allow_nonref->utf8->decode($ua_response)};

    my $has_results_in_hash = ref($data) eq 'HASH' && exists($data->{results}) && ref($data->{results}) eq 'ARRAY';
    # некоторые методы возвращают просто массив, например duty/on_duty
    my $has_results_in_array = ref($data) eq 'ARRAY';

    if ($@ || !defined($data) || !($has_results_in_hash || $has_results_in_array)) {
        cluck "ABC parser error or bad request. ".Dumper($ua_response)." for request ".
                "\$url = '$url'\n".($@ ? "$@" : '');
        return;
    }

    if ($has_results_in_hash) {
        return \@{$data->{results}};
    } else {
        return $data;
    }
}

sub _http_get {
    my ($req) = @_;

    my $ua = LWP::UserAgent->new();
    $ua->timeout(5);

    my $response = $ua->request($req);
    my $ua_response = '';
    
    if ($response->is_success()) {
        $ua_response = $response->content();
    } else {
        cluck "ABC API Error: ".$response->message()."\nresponse-code: ".$response->code();
        return;
    }
    
    return $ua_response;
}

1;
