package Yandex::Advmon;

=head1 NAME

    Yandex::Advmon - запись значений для мониторинга

=head1 DESCRIPTION

    Одна маленькая функция, которая умеет записывать в специальную директорию
    файл в формате YAML с метриками для графиков advmon.

    Файлики подбирает и отправляет в cacti другая утилита, advmon-client.

=head1 SYNOPSIS

    advmon(
        name => 'bsClientData',
        check_files => ['/etc/cron.d/yandex-direct-scripts-auto'],
        max_age => 30*60,
        data => {bsexport => {
            std => {age => 12, size => 4323},
            }
        },
    );

    monitor_values({
        'direct.overdraft.last_success' => time,
    });
    # equivalent to
    monitor_values({
        direct => {overdraft => {last_success => time}},
    });

=cut

use strict;
use warnings;

use Data::Leaf::Walker;
use IO::Socket::INET6;
use Params::Validate qw(:all);
use YAML;

use Yandex::HashUtils;
use Yandex::Trace;
use Yandex::SendMail qw/send_alert/;
use IPC::SysV qw/ftok/;
use IPC::ShareLite qw( :lock );

use base qw/Exporter/;
our @EXPORT = qw/advmon monitor_values/;


=head2 $ADVMON_DIR

    директория, в которой будут создаваться фалы с метриками

=cut
our $ADVMON_DIR ||= '/opt/advmon';

our $GRAPHITE_TMP_DIR ||= '/tmp/yandex_advmon_graphite';

=head2 $GRAPHITE_PREFIX

    коллбэк, возвращающий ссылку на массив из частей префикса названия метрики

=cut
our $GRAPHITE_PREFIX ||= sub {[]};

=head2 $GRAPHITE_SENDER

    адрес, куда отправлять данные

=cut
our $GRAPHITE_SENDER ||= 'localhost:42000';

=head2 $SOLOMON_AGENT_URL

    адрес solomon-агента, в который пушим метрики

=cut
our $SOLOMON_AGENT_URL ||= 'localhost:19092';

=head2 advmon

    Записывать в специальную директорию файл в формате YAML с метриками.
    Принимает именованные параметры:
    - name - название создаваемого файла
    - max_age - через сколько секунд считать данные недействительными
    - check_files - если нет ни одного из указанных файлов - считать данные устаревшими
    - data - хэш с метриками, может содержать вложенные хэши, в этом случае ключи 
            склеиваются через точку

=cut
sub advmon {
    my %O = @_;
    validate(@_, {
        name => {type => SCALAR},
        data => {type => HASHREF},
        check_files => {type => ARRAYREF, optional => 1},
        max_age => {type => SCALAR, optional => 1},
    });
    my $r = hash_merge {}, hash_cut \%O, qw/check_files max_age/;

    # плющим хэши
    $r->{data} = _flat_hash($O{data});

    # пишем файл
    my $filename = "$ADVMON_DIR/$O{name}.yaml";
    open(my $fh, ">:encoding(utf-8)", "$filename.tmp") || die "Can't open $filename.tmp: $!";
    print $fh YAML::Dump($r) or die "Can't print to $filename.tmp: $!";
    close($fh) || die "Can't close $filename.tmp: $!";
    rename("$filename.tmp", $filename) || die "Can't rename $filename.tmp -> $filename: $!";
}

# "Уплощение" хэша,
# из {k1 => {k2 => v2}, k3 => v3}
# делаем {"k1.k2" => v2, k3 => v3}
sub _flat_hash {
    my $h = shift;
    my %ret;
    while(my ($k, $v) = each %$h) {
        if (ref $v && ref($v) eq 'HASH') {
            my $fh = _flat_hash($v);
            while(my ($k1, $v1) = each %$fh) {
                $ret{"$k.$k1"} = $v1;
            }
        } else {
            $ret{$k} = $v;
        }
    }
    return \%ret;
}


=head2 monitor_values({$path => $value, ...})

    Отправляем данные через graphite-sender

    Параметры:
        $values - ссылка на хеш значений.
            Ключи вложенных хешей склеиваем через точку для формирования пути
        %params - хеш с необязательными параметрами.
            time - временной штамп в формате unix (результат функции time() ).
                Если не передать, то берётся текущий timestamp.
            timeout - таймаут в секундах на различные операции с сокетом (см. IO::Socket::INET параметр Timeout).
                Если не передать, то timeout будет равен 7 секундам.
            combine_concurrently_records - Объединяет значения  которые записываются на один и тот же график в одно и тоже время (возможно разными процессами)
                Если не передать, то на груфик попадет только одно из значений.

=cut
sub monitor_values {

    my ( $values, %params ) = @_;
    my $profile = Yandex::Trace::new_profile('advmon:monitor_values');
    my $time = $params{ 'time' };
    my $timeout = $params{ 'timeout' } || 7;

    $time = time() if ( ! $time || $time !~ m/^\d+$/ );

    my ($sender_host, $sender_port) = split(/:/, $GRAPHITE_SENDER);
    my $socket = IO::Socket::INET6->new(
        PeerHost => $sender_host,
        PeerPort => $sender_port,
        Proto => 'tcp',
        Timeout => $timeout,
    );

    unless ($socket) {
        send_alert("Failed to open graphite-sender socket: $!", 'graphite-sender');
        return;
    }

    my $walker = Data::Leaf::Walker->new($values);
    my $cnt = 0;
    my $prefix = $GRAPHITE_PREFIX->();
    
    my %graphs;

    while (my ($path, $value) = $walker->each) {
        $graphs{ join('.', @$prefix, @$path) } = $value;
        $cnt += 1;
    }
    
    if ($params{combine_concurrently_records}) {
        foreach my $gr (keys %graphs) {
            my ($last_access_time, $last_value) = _get_last_access_time_to_graph($gr);
            
            if ($last_access_time && $last_access_time == $time) {
                $graphs{ $gr } += $last_value;
            }

            _update_last_access_time_to_graph($gr, $time, $graphs{ $gr });
        }
    }

    my $data = join("\n", map { $_." ".$graphs{$_}." ".$time } keys %graphs)."\n";
    $profile->obj_num($cnt);
    local $| = 1;
    print $socket $data;
    $socket->close();

    _send_data_to_solomon($data, $timeout);
    
    _unlock_memory(keys %graphs) if $params{combine_concurrently_records};
}

sub _send_data_to_solomon
{
    my $data = shift;
    my $timeout = shift;

    my ($sender_host, $sender_port) = split(/:/, $SOLOMON_AGENT_URL);
    my $socket = IO::Socket::INET6->new(
        PeerHost => $sender_host,
        PeerPort => $sender_port,
        Proto => 'tcp',
        Timeout => $timeout,
    );

    unless ($socket) {
        send_alert("Failed to open solomon-agent socket: $!", 'solomon-agent is unavailable');
        return;
    }

    local $| = 1;
    print $socket $data;
    $socket->close();
}

{
my %memory_cache;
sub _get_last_access_time_to_graph
{
    my ($graph) = @_;

    if (!defined $memory_cache{$graph}) {
        my $path = $graph;
        $path =~ s#/#_#g;

        if (! -e $GRAPHITE_TMP_DIR) {
            mkdir ($GRAPHITE_TMP_DIR) or die "mkdir($GRAPHITE_TMP_DIR): $!";
        }

        $path = "$GRAPHITE_TMP_DIR/$path";
        open (my $tmp_fd, '>', $path) or die "open($GRAPHITE_TMP_DIR/$graph): $!";

        $memory_cache{$graph} = IPC::ShareLite->new(   -key     => ftok($path),
                                                       -create  => 'yes',
                                                       -destroy => 'no',
                                                       -exclusive => 'no',
                                                       -size => 4096 );
    }
    
    $memory_cache{$graph}->lock(LOCK_EX);
    return split(/:/,$memory_cache{$graph}->fetch(), 2);
}

sub _update_last_access_time_to_graph
{
    my ($graph, $time, $new_value) = @_;
    $memory_cache{$graph}->store(join(":", $time, $new_value));
}

sub _unlock_memory
{
    my (@graphs) = @_;
    $memory_cache{$_}->unlock() for @graphs;
}
}

1;
