package Yandex::Antispam::RTChecker;

use Direct::Modern;

use Encode 'encode_utf8';
use HTTP::Headers;
use JSON::RPC2::Client ();

use Yandex::HTTP 'http_fetch';

use constant {
    PACK_SIZE => 1_000,
};

our $API_URL //= 'http://direct-checker-rt.n.yandex-team.ru/jsonrpc';

=head1 NAME

    DirectChecker - Класс клиента для выполнения удалённых запросов при взаимодействии с API DirectChecker
    https://wiki.yandex-team.ru/users/diseaz/DirectChecker/RealTime/
    Пример: C<DirectChecker->check_urls(['http://ya.ru', 'http://вести.рф/'])>

=head1 METHODS

=head2 _client

    Возвращает объект JSON::RPC2::Client

=cut

sub _client { state $client = JSON::RPC2::Client->new() }

=head2 _headers

    Возвращает объект HTTP::Headers

=cut

sub _headers {
    my ($self) = @_;

    state $headers;
    unless( $headers ){
        $headers = HTTP::Headers->new();
        $headers->push_header(
            'Content-Type' => 'application/json',
            'Accept'       => 'application/json'
        );
    }
    return $headers;
}

=head2 _check_urls( ArrayRef[Str] )

    Проверка урлов. Приватный метод.
    $urls - Массив строк проверяемых урлов
    Возвращает результат check_urls из API. Это должен быть ArrayRef[HashRef], где в каждом элементе есть ключ url и verdict

=cut

sub _check_urls {
    my ($self, $urls) = @_;

    my $json_text = $self->_client->call_named( 'check_urls', 'urls' => [ map {{ 'url' => $_ }} @$urls ] );

    my $res = http_fetch(
        'POST',
        $API_URL,
        $json_text,
        'headers'       => $self->_headers,
        'ipv6_prefer'   => 1,
        'handle_params' => {'keepalive' => 1},
    );

    confess('JSON-RPC request failed') unless $res;

    $res = encode_utf8( $res ) if utf8::is_utf8( $res );

    my ( $failed, $result, $error ) = $self->_client->response( $res );

    # немного странный порядок проверки результата, но так по документации JSON::RPC2::Client
    if($failed) {
        confess( 'Bad response: '.$failed );
    }
    elsif($result) {
        return $result;
    }
    elsif($error && ref $error eq 'HASH') {
        confess( 'Request failed. Code: '.($error->{'code'} // '').' Message: '.($error->{'message'} // '') );
    }
    else {
        confess( 'Unknown error' );
    };
}

=head2 check_urls( ArrayRef[Str] )

    Проверка урлов
    $urls - Массив строк проверяемых урлов
    Возвращает HashRef[ArrayRef[Str]]:
        ключи - урлы
        значения - массив строк вердиктов (только негативных)
    Если урл "чистый" то он в результат не попадёт

=cut

sub check_urls{
    my ($self, $urls) = @_;

    confess('Need URLs') unless $urls && ref $urls eq 'ARRAY';
    return {} unless @$urls;
    # скопируем массив урлов, чтобы не попортить его splice
    my @urls = @$urls;
    my $res = {};
    while(my @portion = splice @urls, 0, PACK_SIZE){
        my $p_res = $self->_check_urls(\@portion);
        confess('Incorrect response format') unless ref $p_res eq 'ARRAY';
        $res = {%$res, map {$_->{'url'} => $_->{'verdicts'}} grep {$_->{'url'} && $_->{'verdicts'}} @$p_res};
    }
    return $res;
}

1;