package Yandex::Avatars;

# $Id$

=head1 NAME

    Yandex::Avatars

=head1 DESCRIPTION

    Функции для работы с аватарницей
    http://wiki.yandex-team.ru/Avatars

=head1 SYNOPSIS

    my $get_host  = 'avatars-fast.yandex.net:80';
    my $put_host  = 'avatars-int-bs.yandex.net:14000';
    my $namespace = 'direct';

    my $data;

    # Вариант 1: настройки в глобальных переменных
    # подсказка: можно сделать local
    $Yandex::Avatars::GET_HOST  = $get_host;
    $Yandex::Avatars::PUT_HOST  = $put_host;
    $Yandex::Avatars::NAMESPACE = $namespace;
    $data = Yandex::Avatars->avatars_get('FOOBAR');

    # Вариант 2: настройки в объекте
    my $avatars = Yandex::Avatars->new(
        get_host  => $get_host,
        put_host  => $put_host,
        namespace => $namespace,
    );
    $data = $avatars->avatars_get('FOOBAR');

=cut

=head1 AUTHORS

    Никита Билоус <icenine@yandex-team.ru>
    Андрей Ильин <andy-ilyin@yandex-team.ru>
    Андрей Луковенко <aluck@yandex-team.ru>

=cut

use strict;
use warnings;

use base 'Exporter';

use Carp;
use LWP::UserAgent;
use URI::Escape qw/uri_escape_utf8/;
use MIME::Base64 qw/encode_base64/;

use Yandex::Trace;
use Yandex::HTTP qw/http_get http_post http_parallel_request submit_form/;
use Yandex::Retry qw/retry/;
use Yandex::Log;
use XML::LibXML;

our $AVATARS_GET_HOST ||= 'avatars-fast.yandex.net:80';
our $AVATARS_PUT_HOST ||= 'avatars-int-bs.yandex.net:14000';

our $AVATARS_NAMESPACE ||= 'direct';

my %LOG_OPTIONS = (
    date_suf => '%Y%m',
    log_file_name => 'avatars',
    auto_rotate => 1,
);

=head2 new

Конструктор; превращает список ( поле => значение, ... ) в объект.
Поля объекта:

* get_host  (по умолчанию AVATARS_GET_HOST)
* put_host  (по умолчанию AVATARS_PUT_HOST)
* namespace (по умолчанию AVATARS_NAMESPACE)

=cut

sub new {
    my ($class, %fields) = @_;

    return bless {
        get_host  => $fields{get_host}  || $AVATARS_GET_HOST,
        put_host  => $fields{put_host}  || $AVATARS_PUT_HOST,
        namespace => $fields{namespace} || $AVATARS_NAMESPACE,
    }, $class;
}

=head2 avatars_get

Получить картинку из аватарницы.
Параметры
    image_id - id картинки
    format_id - id формата
Именованые параметры
    url - ссылка на картинку, которую нужно выкачать и сохранить в аватарнице. Картинка будет сохранена под указаным image_id

=cut

sub avatars_get
{
    my ($self, $id, $format, %opt) = @_;
    my $url = $self->avatars_url('get')."/$id/$format";
    if ($opt{url}) {
        $url .= "/".uri_escape_utf8(encode_base64($opt{url}));
    }
    my $profile = Yandex::Trace::new_profile('avatars:get');
    return http_get($url);
}

=head2 avatars_check

Проверить, есть ли картинка в аватарнице.
Параметры
    image_id - id картинки
    format_id - id формата

=cut

sub avatars_check
{
    my ($self, $id, $format, %opt) = @_;
    my $url = $self->avatars_url('check')."/$id/$format";
    my $profile = Yandex::Trace::new_profile('avatars:check');

    # здесь используется submit_form, а не http_get, потому что последний безусловно выдает
    # сообщение в stderr в случае получения ответа с 404; в рамках этой функции такой ответ
    # считается корректным поведением, и лишних сообщений в stderr не требуется
    my $response = submit_form('GET', $url, {});
    return $response->is_success;
}

=head2 avatars_getinfo  

    Получает мета-данные картинки из аватарницы.
    Параметры
        image_id - id картинки

    В мета-данные входят результаты классификации изображений в Компьютерном Зрении:
    лица, а также вероятности порно, трагичности, желтухи и т. п.

    my $xml = Yandex::Avatars->avatars_getinfo($image_id);
  
=cut

sub avatars_getinfo
{
    my ($self, $id) = @_;

    my $profile = Yandex::Trace::new_profile('avatars:getinfo');

    my $bulk_results = $self->avatars_getinfo_bulk( ids => [$id] );

    my $response = $bulk_results->{$id};
    return unless $response && $response->{is_success};
    return $response->{content};
}

=head2 _avatars_processmeta_bulk

    Bulk method для meta-запросов, используется в C<avatars_getinfo_bulk>, C<avatars_delinfo_bulk>

=cut

sub _avatars_processmeta_bulk {
    my ($self, $method, %params) = @_;

    my @ids = grep{ $_ } @{ delete( $params{'ids'} ) || [] };
    my $request_id = 1;
    my %requests = map { $request_id++ => { 'url' => $self->avatars_url($method)."/$_/meta" } } @ids;
    my $profile = Yandex::Trace::new_profile('avatars:meta', obj_num => scalar @ids);
    my $responses = http_parallel_request( 'GET', \%requests, %params ) || {};
    my %result = ();

    foreach my $request_id ( sort { $a <=> $b } keys( %{ $responses } ) ) {
        my $image_hash = shift( @ids );
        $result{ $image_hash } = $responses->{ $request_id };
    }

    return( \%result );
}

=head2 avatars_getinfo_bulk

    Получает мета-данные картинок из аватарницы.
    В качестве параметра принимает хеш со следующими ключами:
        ids - ссылка на массив с id картинок (undef и пустые значения будут проигнорированны)
        все параметры, которые принмает Yandex::HTTP::http_parallel_request

    В мета-данные входят результаты классфикации изображений в Компьютерном Зрении: лица, а так же вероятности порно, трагичности, желтухи и т.п.

    my $responses = Yandex::Avatars->avatars_getinfo_bulk( ids => \@image_ids, maxreq => 10 );
  
    По словам izhidkov@ "Дергать можно с нагрузкой 50-80rps для прода, и не более 10rps в тестинге"
  
=cut

sub avatars_getinfo_bulk
{
    my $self = shift;

    return $self->_avatars_processmeta_bulk( 'getinfo', @_ );
}

=head2 avatars_delinfo_bulk

    Удаляет мета-данные картинок в аватарнице. При обновлении классификаторов в Компьютерном Зрении для
    получения новых классов по уже загруженным в Аватарницу картинкам необходимо удалить по ним метаданные 
    и перезапросить снова при помощи avatars_getinfo либо avatars_getinfo_bulk

    В качестве параметра принимает хеш со следующими ключами:
        ids - ссылка на массив с id картинок (undef и пустые значения будут проигнорированы)
        все параметры, которые принмает Yandex::HTTP::http_parallel_request

    my $responses = Yandex::Avatars->avatars_delinfo_bulk( ids => \@image_ids, maxreq => 50 );
  
    По словам izhidkov@ "удалять можно старые метаданные можно 300 rps в проде, 50 rps в тестинге"

=cut

sub avatars_delinfo_bulk
{
    my $self = shift;

    return $self->_avatars_processmeta_bulk( 'delete', @_ );
}

=head2 avatars_put

Положить картинку в аватарницу
Параметры:
    image_id - id картинки
    image - картинка (бинарные данные)
Именованые параметры
    url - ссылка на картинку, по аналогии с avatars_get

=cut

sub avatars_put
{
    my ($self, $image_id, $raw_image, %opt) = @_;

    my $namespace = ref $self ? $self->{namespace} : $AVATARS_NAMESPACE;

    my $url = $self->avatars_url('put');
    my $profile = Yandex::Trace::new_profile('avatars:put');
    my $resp_xml;
    my $log = Yandex::Log->new(%LOG_OPTIONS);
    retry tries => 3, pauses => [0.1, 1], sub {
        if ($opt{url}) {
            $resp_xml = http_get($url, { id => $image_id, url => $opt{url} });
        }
        else {
            my $file_field_name = $opt{file_field_name} || 'file';

            my $request = LWP::UserAgent->new(timeout => 60)->post($url, Content_Type => 'form-data',
                Content => [
                    id => $image_id,
                    $file_field_name => [ undef, $image_id, Content => $raw_image ],
                ],
            );
            if ($request->is_success) {
                $resp_xml = $request->content;
            }
            else {
                $log->die("Failed to post '$image_id' to avatars ($url): (".$request->code.") ".$request->content);
            }
        }
    };
    undef $profile;
    $log->out("image '$image_id' saved: $resp_xml");
    my $doc = XML::LibXML->new()->parse_string($resp_xml)->documentElement();
    my $xpc = XML::LibXML::XPathContext->new($doc);
    my @sizes = $xpc->findnodes("/$namespace/sizes/size");
    my $resp = [];
    for my $size (@sizes) {
        push @$resp, { map { $_ => $size->getAttribute($_) } qw/id width height path/ };
    }
    return $resp;
}

=head2 avatars_delete

Удалить изображение из аватарницы
Параметры:
  id - id картинки
  format - id формата. Если не указан - будут удалены все размеры

=cut

sub avatars_delete
{
    my ($self, $id, $format) = @_;
    my $log = Yandex::Log->new(%LOG_OPTIONS);
    my $url = $self->avatars_url('delete')."/$id";
    if ($format) {
        $url .= "/$format";
    }
    my $res = http_get($url);
    $log->out("deleted image '$id'");
    return $res;
}

=head2 avatars_url

Функция для составления общей части url к аватарнице
$method - 'get', 'put' или 'delete'

Именованые параметры:
    short => 1 - если GET URL не должен содержать протокола и порта

=cut

sub avatars_url
{
    my ($self, $method, %opt) = @_;

    my $get_host  = ref $self ? $self->{get_host}  : $AVATARS_GET_HOST;
    my $put_host  = ref $self ? $self->{put_host}  : $AVATARS_PUT_HOST;
    my $namespace = ref $self ? $self->{namespace} : $AVATARS_NAMESPACE;

    my $url;
    if ($method =~ /^(put|delete|getinfo)$/) {
        $url = "http://$put_host/$method-$namespace";
    }
    elsif ($method =~ /^(get|check)$/) {
        my $host = $get_host;
        if ($opt{short}) {
            $host =~ s/:\d+$//;
            $host = '//'.$host;
        }
        else {
            $host = "http://$host";
        }
        $url = "$host/$method-$namespace";
    }
    else {
        croak "unknown avatars method '$method'";
    }
    return $url;
}

1;

__END__
