package Yandex::Avatars::MDS;

=head1 NAME

Yandex::Avatars::MDS

=head1 DESCRIPTION

https://wiki.yandex-team.ru/mds/avatars/

Модуль для работы с МДС-Аватарницей
Обертка над http api
Хранение ключа (group_id) ложится на пользователя этой библиотеки

Все методы умирают при ошибке

=head1 SYNOPSYS

    use Yandex::Avatars::MDS;

    my $avatars = Yandex::Avatars::MDS->new(namespace => 'direct');
    my $meta = $avatars->put($image_id => $image_data);
    my $group_id = $meta->{'group-id'};

    my $image_data2 = $avatars->get($group_id, $image_id);

    $avatars->delete($group_id, $image_id);

=cut

our $GET_HOST //= 'avatars.mds.yandex.net';
our $PUT_HOST //= 'avatars-int.mds.yandex.net:13000';

my %LOG_OPTIONS = (
    date_suf => '%Y%m%d',
    log_file_name => 'avatars_mds',
    auto_rotate => 1,
);

our $LOG_PUT_REQUESTS;
our $DEFAULT_PUT_RPS ||= 50;

my %RETRY_OPTIONS = (
    tries => 3,
    pauses => [0.1, 1],
);

use Direct::Modern;
use Mouse;

use Carp;
use JSON;
use List::MoreUtils qw/any/;
use HTTP::Request::Common;
use Time::HiRes;

use Yandex::Log;
use Yandex::Trace;
use Yandex::HTTP qw/http_parallel_request http_fetch/;
use Yandex::Retry qw/retry/;
use Yandex::HashUtils qw/hash_merge/;

has get_host => (
    is => 'ro',
    isa => 'Str',
    default => $GET_HOST,
);

has put_host => (
    is => 'ro',
    isa => 'Str',
    default => $PUT_HOST,
);

has put_rps => (
    is => 'ro',
    isa => 'Int',
    default => sub { $DEFAULT_PUT_RPS },
);

has prioritypass => (
    is => 'ro',
    isa => 'Bool',
    default => 1,
);

has num_attempts => (
    is => 'ro',
    isa => 'Int',
    default => 3,
);

has json => (
    is => 'ro',
    isa => 'JSON',
    default => sub { JSON->new()->utf8(1) },
);

has namespace => (
    is => 'ro',
    isa => 'Str',
    required => 1,
);

has log => (
    is => 'ro',
    isa => 'Object',
);

=head2 new

Именованые параметры:
    namespace => $namespace -- required -- неймспейс аватарницы
    log => $log -- логгер
    get_host => $get_host
    put_host => $put_host


=cut

sub BUILDARGS
{
    my ($self, %opt) = @_;
    unless ($opt{log}) {
        $opt{log} = Yandex::Log->new(%LOG_OPTIONS);
    }
    return \%opt;
}

=head2 couplelist

https://wiki.yandex-team.ru/mds/avatars/#couplelist

=cut

sub couplelist
{
    my ($self) = @_;
    return $self->_http_request('couplelist', GET => $self->avatars_url('couplelist'));
}

=head2 delete

https://wiki.yandex-team.ru/mds/avatars/#delete

=cut

sub delete
{
    my ($self, $group_id, $image_id, $size) = @_;
    # /delete-$namespace/$group-id/$imagename/$size
    my $url = $self->avatars_url('delete')."/$group_id/$image_id".($size ? "/$size" : "");
    return $self->_http_request('delete', GET => $url, json => 0);
}

=head2 get

https://wiki.yandex-team.ru/mds/avatars/#get

my $image_data = $avatars->get($group_id, $image_hash, 'orig')

=cut

sub get
{
    my ($self, $group_id, $image_id, $size) = @_;
    $size //= 'orig';
    # /get-$namespace/$group-id/$imagename/$size
    my $url = $self->avatars_url('get')."/$group_id/$image_id/$size";
    return http_fetch(GET => $url, undef, log => $self->log);
}

=head2 getimageinfo

https://wiki.yandex-team.ru/mds/avatars/#getimageinfo

=cut

sub getimageinfo
{
    my ($self, $group_id, $image_id) = @_;
    # /getimageinfo-$namespace/$group-id/$filename
    my $url = $self->avatars_url('getimageinfo')."/$group_id/$image_id";
    return $self->_http_request('getimageinfo', GET => $url);
}

=head2 getinfo

https://wiki.yandex-team.ru/mds/avatars/#getinfo

=cut

sub getinfo
{
    my ($self, $group_id, $image_id) = @_;
    # /getinfo-$namespace/$group-id/$imagename/meta
    return $self->getinfo_multi([ [$group_id, $image_id] ])->{$image_id};
}

=head2 _meta_method_multi

Массовые методы над $group_id/$image_id/meta

=cut

sub _meta_method_multi
{
    my ($self, $method, $request) = @_;
    my $profile = Yandex::Trace::new_profile("avatars:$method", obj_num => scalar @$request);
    my $extra_params = $method eq "getinfo" && $self->prioritypass ? "?prioritypass=1" : "";
    my %req = map {
        $_->[1] => { url => $self->avatars_url($method)."/$_->[0]/$_->[1]/meta" . $extra_params }
    } @$request;
    my $resp = http_parallel_request(GET => \%req);
    my %result;
    for my $row (@$request) {
        my $response = $resp->{$row->[1]};
        if ($response->{is_success}) {
            $result{$row->[1]} = $self->json->decode($response->{content});
        }
        else {
            utf8::decode($response->{headers}->{Reason});
            utf8::decode($response->{headers}->{Status});
            $self->log->die("getinfo: $row->[1]: ERROR $response->{headers}->{Status} $response->{headers}->{Reason}");
        }
    }
    return \%result;

}

=head2 getinfo_multi

my $info_by_hash = $avatars->getinfo_multi([ [ $group_id, $image_hash], [ $group_id2, $image_hash2 ], ... ]);

=cut

sub getinfo_multi
{
    my ($self, $request) = @_;
    return $self->_meta_method_multi('getinfo', $request);
}


=head2 deleteinfo_multi

https://st.yandex-team.ru/MDS-2530#1461856784000
delete-direct/$coupe_ip/$imagename/meta

$request = [ [$group_id, $image_id], [...] ]

returns { $image_id => $result };

=cut

sub deleteinfo_multi
{
    my ($self, $request) = @_;
    return $self->_meta_method_multi('delete', $request);
}

=head2 put_url

Залить изображение в аватарницу по ссылке

    my $reply = $avatars->put_url('my_image_id', 'http://apod.nasa.gov/apod/image/1604/ngc7635bubble_hubble26.jpg');

Возвращает распаршеный ответ аватарницы, либо умирает

=cut

sub put_url
{
    my ($self, $image_id, $image_url) = @_;
    my $url = Yandex::HTTP::make_url($self->avatars_url('put')."/$image_id", { url => $image_url });
    if ($self->prioritypass) {
        $url = Yandex::HTTP::make_url($url, { prioritypass => 1 });
    }
    return $self->_http_request('put_url', GET => $url, retry_on => [qw/429 5\d{2}/])->{attrs};
}

=head2 put

https://wiki.yandex-team.ru/mds/avatars/#put

Залить изображение в аватарницу

    my $reply = $avatars->put($image_id, $image_data);

Опции прокидываются в put_multi

Возвращает распаршеный ответ аватарницы, либо умирает

=cut

sub put
{
    my ($self, $image_id, $image_data, %O) = @_;

    my $res = $self->put_multi({$image_id => $image_data}, %O)->{$image_id};
    die "avatars error: $res->{http_code} $res->{http_message}"  if !$res->{'group-id'};

    return $res;
}

=head2 put_multi

    https://wiki.yandex-team.ru/mds/avatars/#put_multi

    Залить пачку изображений в аватарницу

    my $reply = $avatars->put_multi({
        $image_id_1 => $image_data_1,
        $image_id_2 => $image_data_2,
        ...
    });

    image_data может быть:
        - скаляр (бинарные данные) или ссылка на скаляр
        - хеш вида {data => $data} или {url => $url}, может содержать опции

    Возвращает распаршеные ответы аватарницы, либо параметры ошибки
    {
        $image_id_1 => $image_resp_1,
        $image_id_2 => $image_resp_2,
        ...
    }
    где значения эквиваленты тем, которые возвращает метод put

    Опции:
        file_field_name - название параметра для данных, по умолчанию file

=cut

sub put_multi{
    my ($self, $images, %O) = @_;

    my $profile = Yandex::Trace::new_profile( 'avatars:put_multi', obj_num => scalar(keys %$images) );

    my $res = {};
    # сделаем массив индексов для пакетной обработки, это проще, чем считать на лету
    my @indexes = keys %$images;
    # время начала предыдущего запроса
    my $prev_req_start_time = 0;
    # у Аватарницы есть ограничение по rps, поэтому перебираем пачками не более этого значения
    while (my @bulk = splice @indexes, 0, $self->put_rps){
        my %post_request;
        for my $image_id (@bulk) {
            my $url = $self->avatars_url( 'put' ).'/'.$image_id;
            if ($self->prioritypass) {
                $url = Yandex::HTTP::make_url($url, { prioritypass => 1 });
            }

            my $image_ref;
            my $image_info = $images->{ $image_id };
            if (!ref $image_info) {
                $image_ref = \$image_info;
            }
            elsif (ref $image_info eq 'SCALAR') {
                $image_ref = $image_info;
            }
            elsif (ref $image_info eq 'HASH' && $image_info->{url}) {
                $url = Yandex::HTTP::make_url($url, { url => $image_info->{url} });
            }
            elsif (ref $image_info eq 'HASH' && $image_info->{data}) {
                $image_ref = \$image_info->{data};
            }
            else {
                croak 'Bad image data';
            }

            my $file_field_name =
                (ref $image_info eq 'HASH' && $image_info->{file_field_name})
                || $O{file_field_name}
                || 'file';

            # загрузку урлов делаем через POST с пустым телом, чтобы работало в одном http_parallel_request
            my $content = $image_ref
                ? [ $file_field_name => [undef, $image_id, Content => $$image_ref] ]
                : [];

            my $req = HTTP::Request::Common::POST(
                $url,
                Content_Type => 'form-data',
                Content      => $content,
            );
            $post_request{$image_id} = {
                url     => $url,
                body    => \$req->content,
                headers => $req->headers,
            }
        }
        # если мы уже делали запрос
        if ( $prev_req_start_time ){
            my $diff = Time::HiRes::time() - $prev_req_start_time;
            #  в течении последней секунды
            if ($diff < 1.0) {
                # подождём пока пройдёт целая секунда
                Time::HiRes::sleep(1.0 - $diff);
            }
        };
        $prev_req_start_time = Time::HiRes::time();

        my $http_responses = http_parallel_request(
            POST => \%post_request,
            # перезапрашиваем только 5xx ошибки
            retry_on => sub { my ($c, $h) = @_;  $h->{Status} =~ /^5/ },
            num_attempts => $self->num_attempts || 1,
        );

        while(my ($id, $http_res) = each %$http_responses){
            if ($LOG_PUT_REQUESTS) {
                $self->log->out("request: " . $post_request{$id}->{url});
                $self->log->out("response: " . $http_res->{'content'});
            }
            my $res4id = eval { $self->json->decode($http_res->{'content'}) } || {};
            $res4id->{http_code} = $http_res->{'headers'}{'Status'};
            $res4id->{http_message} = $http_res->{'headers'}{'Reason'};

            # для унификации ответа при 403 пробрасываем поля из attrs в корень
            hash_merge $res4id, $res4id->{'attrs'}  if $res4id->{'attrs'};

            if ($http_res->{'headers'}{'Status'} !~ /^200|403$/) {
                utf8::decode($http_res->{'headers'}{'Status'});
                utf8::decode($http_res->{'headers'}{'Reason'});
                $self->log->out(sprintf('put_multi: ERROR %s %s %s', $post_request{$id}{'url'}, $http_res->{'headers'}->{'Status'}, $http_res->{'headers'}->{'Reason'}));
            }

            $res->{ $id } = $res4id;
        }
    }
    return $res;
}

=head2 statistics

https://wiki.yandex-team.ru/mds/avatars/#statistics

=cut

sub statistics
{
    my ($self) = @_;
    return $self->_http_request('statistics', GET => $self->avatars_url('statistics'));
}

=head2 avatars_url

Получить url для данного метода

    my $url = $avatars->avatars_url('put');
    my $url = $avatars->avatars_url('get', short => 1);

Если указано short => 1 - возвращается url без протокола (//avatars/get-$ns/)

=cut

sub avatars_url
{
    my ($self, $method, %opt) = @_;

    my $ns = $self->namespace;
    if ($method =~ /^put|delete|couplelist|getimageinfo|getinfo|statistics$/) {
        return "http://".$self->put_host."/$method-$ns";

    } elsif ($method =~ /^get$/) {
        if ($opt{short}) {
            return "//".$self->get_host."/$method-$ns";
        }
        return "http://".$self->get_host."/$method-$ns";

    } else {
        croak "invalid method '$method'";
    }
}

=head2 _http_request

%opt:
    retry_on => ['410', '5\d+'] -- коды http ошибок, при которых нужно перезапрашивать
    json => 1|0 -- не пытаться декодировать json, ручка отвечает plain text

=cut

sub _http_request
{
    my ($self, $action, $method, $url, %opt) = @_;
    my $profile = Yandex::Trace::new_profile("avatars:$action");

    my $retry_on = delete $opt{retry_on} // ['5\d{2}'];
    my $json = delete $opt{json} // 1;
    my $res = eval {
        retry %RETRY_OPTIONS, sub {
            my $res = http_parallel_request($method, { 1 => { url => $url, %opt, timeout => 60 } })->{1};
            $res->{is_success} //= 0;
            utf8::decode($res->{headers}->{Status});
            utf8::decode($res->{headers}->{Reason});
            $res->{headers}->{Status} //= 599;
            if (any { $res->{headers}->{Status} =~ /^$_$/ } @$retry_on) {
                $self->log->die(sprintf('%s: ERROR %s %s %s', $action, $url, $res->{headers}->{Status}, $res->{headers}->{Reason}));
            }
            return $res;
        };
    };
    if ($@) {
        $self->log->die("$url ERROR: $@");
    }
    
    if ($res->{headers}->{Status} !~ /2\d{2}/) {
        $self->log->die("$url: $res->{headers}->{Status} $res->{headers}->{Reason}\n".($res->{content}//''));
    }

    if (!$json) {
        return $res->{content};
    }
    
    my $decoded = eval { $self->json->decode($res->{content}) };
    if ($@) {
        $self->log->die("$url ERROR in json: '$res->{content}'\n$@");
    }
    $self->log->out("$url: $res->{content}");
    return $decoded;
}

1;

