package Yandex::Balance;

=pod
    $Id$
    Модуль/обёртка вокруг XMLRPC вызовов баланса
    https://wiki.yandex-team.ru/Balance/XmlRpc

=cut

use strict;
use warnings;
use XMLRPC::Lite;
use XML::Simple;
use Data::Dumper;
use List::MoreUtils qw/zip/;
use JSON;

use Yandex::Retry;
use Yandex::SendMail;
use Yandex::XMLRPC::UTF8Serializer;
use Yandex::TimeCommon;
use Yandex::Log;
use Yandex::Trace;
use Yandex::Validate qw/is_valid_int is_valid_date/;
use Yandex::HashUtils;
use Yandex::ScalarUtils;

use base qw/Exporter/;

use utf8;

our $DEBUG;

our @EXPORT = qw/
        balance_create_request
        balance_create_request_2
        balance_find_client
        balance_get_clients
        balance_create_update_orders
        balance_create_client
        balance_create_yamoney_invoice
        balance_create_transfer
        balance_order_info_for_vendor
        balance_merge_clients
        balance_list_client_passports
        balance_get_reps_count_of_uid
        balance_set_main_rep_of_client
        balance_get_info_about_rep
        balance_get_working_calendar

        balance_get_client_credit
        balance_create_person
        balance_get_client_persons
        balance_get_client_contracts
        balance_get_options_by_contract_id
        balance_get_request_choices
        balance_create_fast_payment

        balance_create_invoice
        balance_get_url_for_direct_payment
        balance_get_currency_rate
        balance_get_direct_budget

        is_balance_subclient
        get_client_by_uid
        get_clientid_by_uid
        create_client_id_association
        get_equal_clients
        remove_client_id_association

        balance_get_firm_country_currency
        balance_get_client_currencies

        balance_get_direct_discount
        balance_get_client_nds
        balance_get_completion_history
        balance_get_direct_balance
        balance_get_currency_products

        balance_update_notification_url
        balance_get_orders_info

        balance_add_client_domain

        balance_mass_get_managers_info
        balance_get_managers_info

        balance_update_campaigns

        balance_pay_request
        balance_check_request_payment
/;

our @EXPORT_OK = qw(
    balance_get_all_equal_clients
);

our $BALANCE_XMLRPC;
our $TEST_BALANCE_XMLRPC_URL;
our $BALANCE_FAILS_MAIL;
our $BALANCE_MAX_PHONE_LENGTH ||= 255;

=head2 $DEFAULT_RECALL_POLICY, $RECALL_POLICY

    политики перезапросов для разных значений recall (в формате, поддерживаемом Yandex::Retry)

=cut
our $DEFAULT_RECALL_POLICY ||= {tries => 1};
our $RECALL_POLICY ||= {
    1 => {tries => 2},
    2 => {tries => 3},
};

our %PAYSYS_IDS = (
    'YM' => 1000,
    'LIMIT_UR' => 1003,
    'LIMIT_UA' => 1017,
);

our %BALANCE_CALLS_LOG_SETTINGS;
%BALANCE_CALLS_LOG_SETTINGS = (
    log_file_name => "balance_calls.log",
    date_suf => "%Y%m%d",
) if !%BALANCE_CALLS_LOG_SETTINGS;

# Максимальная длина имени заказа (кампании) в Балансе.
our $BALANCE_ORDER_NAME_MAX_LEN ||= 255;

=head2 balance_call

    вызов баланса 
    Обращаемся всегда по одному и тому же адресу -- $BALANCE_XMLRPC 
    Если потребуется иногда делать обращения к другому Балансу -- можно будет добавить в $opts

    Параметры позиционные 
        $method -- 
        $params -- 
        $opts -- хеш опций {
            recall => 1,    # определить политику перезапросов
            timeout => 5,   # таймаут, по умолчанию -- 60 
            write_log => 1, # записывать в лог параметры вызова и ответ Баланса
            send_alert => 1, # в случае ошибки отсылать письмо 
        }

=cut

sub balance_call {
    my ($method, $params, $opts) = @_;

    my $log = new Yandex::Log(
        %BALANCE_CALLS_LOG_SETTINGS,
    );
    my $profile = Yandex::Trace::new_profile("balance:$method");
    # вызов метода
    my $rpc = XMLRPC::Lite->proxy($opts->{BALANCE_XMLRPC_URL} || $BALANCE_XMLRPC, timeout => $opts->{timeout}||60)
        -> serializer( new Yandex::XMLRPC::UTF8Serializer('utf8') )
        -> deserializer( new Yandex::XMLRPC::UTF8Deserializer('utf8') );

    my $recall_policy = defined $opts->{recall} && $RECALL_POLICY->{$opts->{recall}} || $DEFAULT_RECALL_POLICY;

    my $rpc_res;
    retry %$recall_policy, sub {
        if ($DEBUG || $opts->{write_log}) {
            my $msg_prefix_guard = $log->msg_prefix_guard(sprintf "[pid=%d,reqid=%d,method=%s,data_type=request,try=%d]",
                $$, Yandex::Trace::current_span_id() // 0, $method, $Yandex::Retry::iteration);
            _log_message($log, $params);
        }
        $rpc_res = eval { $rpc->call($method, @$params); };

        # проверка всяческих ошибок
        my $error;
        if ($@) {
            $error = "$method died - $@";
        } elsif (!defined $rpc_res) {
            $error = "$method returned undef";
        } elsif ($rpc_res->fault) {
            $error = "$method fault - ".join(', ', $rpc_res->faultcode, $rpc_res->faultstring, $rpc_res->faultdetail);
        }
        if ($error) {
            if ($opts->{send_alert} && $BALANCE_FAILS_MAIL) {
                send_alert("$error\nParams: ".Dumper($params), "balance-xmlrpc: $method error", $BALANCE_FAILS_MAIL);
            }
            my $msg_prefix_guard = $log->msg_prefix_guard(sprintf "[pid=%d,reqid=%d,method=%s,data_type=request,try=%d]",
                $$, Yandex::Trace::current_span_id() // 0, $method, $Yandex::Retry::iteration);
            unless ($DEBUG || $opts->{write_log}) {
                # если раньше вызов не логировали - логируем сейчас
                _log_message($log, $params);
            }
            $log->die($error);
        }
    }; # /retry

    # получение результата
    my @ret = $rpc_res->paramsall();
    if ($DEBUG || $opts->{write_log}) {
        my $msg_prefix_guard = $log->msg_prefix_guard(sprintf "[pid=%d,reqid=%d,method=%s,data_type=response]",
            $$, Yandex::Trace::current_span_id() // 0, $method);
        _log_message($log, \@ret);
    }

    return @ret;
}

# логгер-обертка, понимающая XMLRPC
sub _log_message {
    my ($log, $data) = @_;
    no warnings 'once';
    local *XMLRPC::Data::TO_JSON = sub {return $_[0]->value};
    $log->out(ref $data ? to_json($data, {allow_unknown => 1, allow_blessed => 1, convert_blessed => 1})
                        : str($data));
}

# пропарсить фаулт
sub balance_parse_fault_xml {
    my ($err_string) = @_;
    my $res = XMLin($err_string);
    # конвертируем parent-codes
    my $pc = $res->{'parent-codes'};
    if ($pc) {
        if (exists $pc->{code}) {
            if (ref($pc->{code}) ne 'ARRAY') {
                $pc->{code} = [$pc->{code}];
            }
        } else {
            $pc->{code} = [];
        }
        $res->{'parent-codes'} = $pc->{code};
    }
    return $res;
}

# создание счёта
sub balance_create_request {
    my @params = @_;

    my ($code, $msg, $url) = balance_call('Balance.CreateRequest', \@params, {write_log => 1});
    if ($code) {
        die "Balance error: Balance.CreateRequest return FAIL '".($code||'')."', '".($msg||'')."'";
    }
    return $url;
}

=head2 balance_create_request_2

    То, что и balance_create_request, отличается форматом возвращаемых значений
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createrequest2
    Возвращает ссылку на массив

=cut

sub balance_create_request_2 {
    my @params = @_;

    return [ balance_call('Balance.CreateRequest2', \@params, {write_log => 1}) ];
}

# создание счёта в балансе и платной ссылки в ЯД
sub balance_create_yamoney_invoice {
    my @params = @_;
    my ($ret) = balance_call('Balance.CreateYaMoneyInvoice', \@params, {write_log => 1});
    my ($code, $msg, $hash) = @$ret;
    if ($code) {
        die "Balance error: Balance.CreateYaMoneyInvoice return FAIL '".($code||'')."', '".($msg||'')."'";
    }
    return $hash;
}

=head2 balance_create_transfer

    Перенос денег с кампаний на кампании
    Параметр $detailed_response - флаг "отдавать ли развернутый ответ".
    Если передан, вернется массив движений средств, в противном случае пустой массив.

=cut

sub balance_create_transfer {
    my ($params, $timeout, $detailed_response) = @_;

    my $result = [];

    eval {
        my ($ret) = balance_call('Balance.CreateTransferMultiple',
                                 [@$params, $detailed_response],
                                 {write_log => 1, send_alert => 1, timeout => $timeout});
        my ($code, $msg) = @$ret;

        if ($code && ref($code) ne 'HASH') {
            die "Balance.CreateTransferMultiple error: $code, $msg";
        }
        
        if ($detailed_response) {
            $result = $ret;
        }
    };
    if (UNIVERSAL::isa($@ => "HASH") && $@->{faultString}) {
        # soap fault
        return balance_parse_fault_xml($@->{faultString})->{code};
    } elsif ($@) {
        if ($@ =~ /(BANNED_AGENCY_TRANSFER|ORDERS_NOT_SYNCHRONIZED|TRANSFER_BETWEEN_LOYAL_GENERAL_CLIENTS|NOT_ENOUGH_FUNDS_FOR_REVERSE|NON_RESIDENT_TRANSFER)/) {
            return $1;
        }
        die {message => Dumper($@), dont_alert => 1};
    }
    return $result;
}

# агрегированная информация по заказу за заданный период.
sub balance_order_info_for_vendor {
    my @params = @_;
    my ($res) = balance_call('Balance.GetOrderInfoForIntel', \@params);
    return $res;
}

=head2 balance_merge_clients

  $result_error = balance_merge_clients($UID, $ClientID1, $ClientID2);
  return 0 on no errors

=cut
sub balance_merge_clients {
    my @params = @_;
    my ($res) = balance_call('Balance.MergeClients', \@params, {write_log => 1});
    return $res;
}

=head2 balance_find_client

    Поиск клиентов по заданным критериям.
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.findclient

    Параметры:
        $condition - hashref с параметрами для поиска, см. вики

    Результат:
        arrayref of hashrefs (как в balance_get_clients)

=cut

sub balance_find_client {
    my @params = @_;
    my ($ret, $ret_str, $clients) = balance_call('Balance.FindClient', \@params);
    if ($ret) {
        die "Balance error: FindClient return '$ret', '$ret_str'";
    }
    return $clients;
}

=head2 balance_get_clients

    Массовое получение данных о клиентах по ClientID.
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getclientbyidbatch

    Возвращает данные только по найденным клиентам.

    Параметры:
        $clientids - arrayref ClientID для получения
    Результат:
        [
            {
                AGENCY_ID       => ..., # агентство (для субклиентов) (число)
                CITY            => ..., # город (строка)
                CLIENT_ID       => ..., # идентификатор клиента (число)
                CLIENT_TYPE_ID  => ..., # идентификатор типа клиента (число)
                EMAIL           => ..., # E-mail клиента (строка)
                FAX             => ..., # факс (строка)
                IS_AGENCY       => ..., # является ли агентством (число)
                NAME            => ..., # имя клиента (строка)
                PHONE           => ..., # телефон клиента (строка)
                REGION_ID       => ..., # страна клиента (число или пустая строка)
                SERVICE_DATA    => ..., # hashref ???
                URL             => ..., # URL клиента (строка)
            },
            ...
        ]

=cut

sub balance_get_clients {
    my (@params) = @_;
    my ($ret, $clients) = balance_call('Balance.GetClientByIdBatch', \@params);
    if ($ret) {
        die "Balance error: GetClientByIdBatch return '$ret'";
    }
    return $clients;
}

=head2 balance_create_update_orders

    Пакетное создание/обновление заказов в Балансе

    Параметры позиционные 
        $UID -- uid оператора
        $orders -- ссылка на массив описаний заказов 
        Каждая кампания -- хеш вида: 
            {
                ServiceID => $product->{EngineID},
                ProductID => $product->{ProductID},
                ServiceOrderID => $cid+0,
                ClientID => $client_id+0,
                NDS => 1,
                Text => $data->{name},
                unmoderated => $pay_method_flag,


                AgencyID => $data->{AgencyID}+0,    # только если у заказа есть агентство
                ManagerUID => $camp_manager_uid+0,  # только если у заказа есть менеджер
            }

    Результат: 
        1 -- порядок
        0 -- непорядок. Либо вызов вообще не прошел, либо Баланс вернул код ошибки по какому-то из заказов

=cut

sub balance_create_update_orders
{
    my ($UID, $orders) = @_;

    for my $order (@$orders) {
        if (defined $order->{Text}) {
            $order->{Text} = substr($order->{Text}, 0, $BALANCE_ORDER_NAME_MAX_LEN);
        }
    }
    my ($balance_res) = eval { balance_call('Balance.CreateOrUpdateOrdersBatch', [$UID, $orders], {write_log => 1}) };

    if ( $@ || ref $balance_res ne 'ARRAY' ) {
        return 0;
    }

    for my $r ( @$balance_res ) {
        if ( $r->[0] ) {
            return 0;
        }
    }
    return 1;
}

=head2 balance_create_client

    Создание клиента в Балансе 
    Описание: https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.createclient

    Параметры позиционные: 
        $UID -- uid оператора
        $data -- хеш про клиента
            { 
                NAME    => ($vars->{name}  ? $vars->{name}  : '-'),
                PHONE   => ($vars->{phone} ? $vars->{phone} : '-'),
                FAX     => ($vars->{fax}   ? $vars->{fax}   : '-'),
                EMAIL   => ($vars->{email} ? $vars->{email} : '-'),
                CITY    => ($vars->{city}  ? $vars->{city}  : '-'),
                URL     => ($vars->{url}   ? $vars->{url}   : '-'), 
                CLIENT_TYPE_ID => 0,
                IS_AGENCY => 0, 
                AGENCY_ID =>,
                REGION_ID  =>, # код страны клиента из Гео-базы (напр. 225-Россия, 187-Украина, 159-Казахстан)
                CURRENCY =>, # ISO код валюты (одно из EUR, KZT, BYR, USD, RUB, UAH)
                MIGRATE_TO_CURRENCY => 0|1,
             }

    Результат -- два значения: 
        $error -- 0 (порядок) или не-0 (ошибка),
        $client_id -- ClientID созданного клиента

    my ($error, $client_id) = balance_create_client($operator_uid, $info);

=cut 

sub balance_create_client
{
    my ($UID, $client_data) = @_; 

    if( defined $client_data->{PHONE} && length($client_data->{PHONE}) > $BALANCE_MAX_PHONE_LENGTH ){
        # Баланс не может хранить слишком длинные телефоны
        $client_data->{PHONE} = substr($client_data->{PHONE}, 0, $BALANCE_MAX_PHONE_LENGTH); 
    }

    my ($res, $msg, $client_id) = eval { balance_call('Balance.CreateClient', [$UID, $client_data], {write_log => 1}) };

    if ( $@ || $res ) {
        return (1, 0);
    }

    return (0, $client_id);
}

# ..................................................................................................

=head2 balance_list_client_passports

  получить всех представителей по ClientID

    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.listclientpassports

  %O = (
    timeout => 5,
  )
  $result = balance_list_client_passports($OPERATOR_UID, $ClientID, %O);
  $result = [
              [
                {
                  'ClientId' => '385179',
                  'IsMain' => '1',
                  'Login' => 'login1',
                  'Name' => 'FIO 1',
                  'Uid' => '15034754'
                },
                {
                  'ClientId' => '385179',
                  'IsMain' => '0',
                  'Login' => 'login2',
                  'Name' => '  FIO 2',
                  'Uid' => '15034751'
                }
              ]
            ];

=cut

sub balance_list_client_passports {
    my ($OPERATOR_UID, $ClientID, %O) = @_;

    my $timeout = $O{timeout} || 5;
    my ($reps, $ret, $ret_str) = balance_call('Balance.ListClientPassports', [$OPERATOR_UID, $ClientID], {timeout => $timeout});
    die "Balance error: Balance.ListClientPassports return '$ret', '$ret_str'" if $ret;

    return ref($reps) eq 'ARRAY' ? $reps : [];
}

# ..................................................................................................

=head2 balance_get_reps_count_of_uid

  получить кол-во представителей в балансе
  $reps_count = balance_get_reps_count_of_uid($OPERATOR_UID, $uid);

=cut

sub balance_get_reps_count_of_uid($$) {
    my ($operator_uid, $uid) = @_;

    my $balance_reps_count = 0;

    my $balance_client_id = get_clientid_by_uid($uid);
    if ($balance_client_id) {
        my $balance_reps = balance_list_client_passports($operator_uid, $balance_client_id);
        $balance_reps_count = scalar(@$balance_reps);
    }

    return $balance_reps_count;
}

# ..................................................................................................

=head2 balance_set_main_rep_of_client

  передать признак представителя клиента "главный" в баланс
  $result = balance_set_main_rep_of_client($OPERATOR_UID, $main_rep_uid);

=cut

sub balance_set_main_rep_of_client {
    my ($operator_uid, $main_uid) = @_;

    # получаем ClientID
    my $balance_info = balance_get_info_about_rep($operator_uid, $main_uid);
    die "Balance error: balance_set_main_rep_of_client(), ClientID not found for uid == $main_uid" unless ref($balance_info) eq 'HASH' && $balance_info->{ClientId};

    # получаем всех представителей по ClientID
    my ($res, $ret, $ret_str) = balance_call('Balance.ListClientPassports', [$operator_uid, $balance_info->{ClientId}]);
    die "Balance error: Balance.ListClientPassports return '$ret', '$ret_str'" if $ret;

    # устанавливаем признак "главный", если нужно
    for my $row (@$res) {
        if ($row->{Uid} == $main_uid && ! $row->{IsMain}) {
            ($res, $ret, $ret_str) = balance_call('Balance.EditPassport', [$operator_uid, $row->{Uid}, {IsMain => 1}], {write_log => 1});
            die "Balance error: Balance.EditPassport return '$ret', '$ret_str'" if $ret;
        } elsif ($row->{Uid} != $main_uid && $row->{IsMain}) {
            ($res, $ret, $ret_str) = balance_call('Balance.EditPassport', [$operator_uid, $row->{Uid}, {IsMain => 0}], {write_log => 1});
            die "Balance error: Balance.EditPassport return '$ret', '$ret_str'" if $ret;
        }
    }

    return 1;
}

# ..................................................................................................

=head2 balance_get_info_about_rep

  получить информацию о представителе в балансе
  $result = balance_get_info_about_rep($OPERATOR_UID, $main_rep_uid, $relations);

  $relations - ссылка на хеш отображения дополнительных связей между паспортом и клиентом
               подробности в документации на метод Баланса GetPassportByUid
               https://wiki.yandex-team.ru/Balance/XMLRPC/#balance.getpassportbyuid

  return hash:
  {
    'ClientId' => '454261',
    'IsMain' => '1',
    'Login' => 'passport_login',
    'Name' => 'F I O in balance',
    'Uid' => '47339493'
  }

=cut

sub balance_get_info_about_rep {
    my ($operator_uid, $uid, $relations) = @_;

    my ($res, $ret, $ret_str) = balance_call('Balance.GetPassportByUid', [$operator_uid, $uid, $relations]);
    return ref($res) eq 'HASH' ? $res : {};
}

=head2 balance_get_working_calendar

  получить информацию о праздничных днях за указанный год
  $result = balance_get_working_calendar($year);

  возвращает массив хэшей:
          [
            {
              'calendar_day' => '1',            # для календарного праздника 0, для не-праздника 1
              'day_descr' => "Предпраздничный", # текстовое описание типа дня
              'dt' => '2010-12-31',             # дата дня в календаре
              'five_day' => '1',                # признак рабочего дня в пятидневной неделе
              'six_day' => '1',                 # признак рабочего дня в шестидневной неделе 
              'year' => '2010'                  # год
            },
            ...
          ]

=cut

sub balance_get_working_calendar {
    my ($year) = @_;

    my ($res, $ret, $ret_str) = balance_call('Balance.GetWorkingCalendar', [$year], {timeout => 5});
    return $res;
}

=head2 balance_get_client_credit

    Получить кредитный лимит клиента.

    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getclientcreditlimits

    Параметры позиционные:
        client_id
        product_id

    $res = balance_get_client_credit($client_id, $product_id);
    $res = {
        CURRENCY => 'RUB',
        LIMITS => [
            {
                CONTRACT_EID => '24240/13',
                CONTRACT_ID => 178678,
                LIMIT_SPENT => 78132,
                LIMIT_TOTAL => 100000,
                ISO_CURRENCY => 'RUB',
            },
            ...
        ],
        PRODUCT_PRICE => 1,
    };

=cut

sub balance_get_client_credit {

    my ($client_id, $product_id) = @_;

    my $response;

    eval {
        my ($res, $ret, $ret_str) = balance_call('Balance.GetClientCreditLimits', [ $client_id, $product_id ], {write_log => 1});
        if ($res) {
            $response = $res;
            if ($response && $response->{LIMITS} && @{$response->{LIMITS}}) {
                $response->{CURRENCY} = $response->{LIMITS}->[0]->{ISO_CURRENCY};
            }
        }
    };

    return $response if $response;

    if (UNIVERSAL::isa($@ => "HASH") && $@->{faultString}) {
        # soap fault
        return balance_parse_fault_xml($@->{faultString})->{code};
    } elsif ($@) {
        if ($@ =~ /(CLIENT_NOT_FOUND)/) {
            return $1;
        }
        die {message => Dumper($@), dont_alert => 1};
    }

    return '';
}

=head2 balance_create_person

    Создает плательщика. Апи Баланса принимает только строковые параметры, поэтому оборачиваем их в XMLRPC::Data->type.
    Оборачиваем только поля, необходимые для создания плательщика с типом 'ph'
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createperson

=cut

sub balance_create_person {
    my $uid = shift;
    my $params = shift;

    foreach my $key (('client_id', 'lname', 'fname', 'mname', 'phone', 'email')) {
        if ($params->{$key}) {
            my $new_value = XMLRPC::Data->type('string')->value($params->{$key});
            $params->{$key} = $new_value;
        }
    }

    return balance_call('Balance.CreatePerson', [$uid, $params]);
}

=head2 balance_get_client_persons

    Получить список плательщиков клиента
    Параметры:
        ClientID

    Возвращает массив хэшей:
        {
            name => 
            id => 
        }

=cut

sub balance_get_client_persons {
    my $client_id = shift;

    my ($res, $ret, $ret_str) = balance_call('Balance.GetClientPersons', [ $client_id ], {write_log => 1, timeout => 120});

    my @result;
    foreach (@$res) {
        push @result, {name => $_->{NAME}, id => $_->{ID}, type => $_->{TYPE}, hidden => ($_->{HIDDEN} // 0)};
    }

    return \@result;
}

sub balance_get_client_contracts {
    my ($client_id, $person_id, $date) = @_;

    my ($res, $ret, $ret_str) = balance_call('Balance.GetClientContracts', [ $client_id, $person_id, $date ], {write_log => 1});
    return $res;
}

sub balance_get_options_by_contract_id {

    my ($client_id, $contract_id, $product_id) = @_;

    my $result = {};

    my $credit_limits = balance_get_client_credit($client_id, $product_id);
    my $contract2limits = { map {$_->{CONTRACT_EID} => $_} @{$credit_limits->{LIMITS}} };

    my $persons = balance_get_client_persons($client_id);

    foreach my $person (@$persons) {

        my $contracts = balance_get_client_contracts($client_id, $person->{id}, today());

        foreach my $ct (@$contracts) {

            if ($ct->{EXTERNAL_ID} eq $contract_id) {

                if (!$contract2limits->{$contract_id}) {
                    return "NO_CREDIT_LIMITS";
                }

                $result->{contract_id} = $ct->{ID};
                $result->{person_id} = $person->{id};

                if ($person->{type} eq 'ur' && $credit_limits->{CURRENCY} eq 'RUR') {
                    $result->{paysys_id} = $PAYSYS_IDS{LIMIT_UR};
                } elsif ($person->{type} eq 'ua' && $credit_limits->{CURRENCY} eq 'UAU') {
                    $result->{paysys_id} = $PAYSYS_IDS{LIMIT_UA};
                } else {
                    return "INVALID_CONTRACT";
                }

                last;
            }
        }
        last if $result->{contract_id};
    }

    return $result;
}

=head2 balance_get_request_choices

    Возвращает данные о возможных способах оплаты клиентом.
    Параметры:
        operator_uid
        request_id
        contract_eid - опционально, если указан, возвращается информация об оплате только этим договором

=cut

sub balance_get_request_choices($$;$) {
    my ($operator_uid, $request_id, $contract_eid) = @_;

    my $params = {OperatorUid => $operator_uid, RequestID => $request_id};

    if ($contract_eid) {
        $params->{ContractExternalID} = $contract_eid;
    }

    my $result;

    eval {
        my ($res, $ret, $ret_str) = balance_call('Balance.GetRequestChoices', [ $params ], {write_log => 1});
        if ($res) {
            $result = $res;
        }
    };

    if ($@) {
        if ($@ =~ /(CONTRACT_NOT_FOUND|ILLEGAL_REQUEST_CONTRACT)/) {
            return $1;
        }
        die {message => Dumper($@), dont_alert => 1};
    }

    return $result;
}

=head2 balance_create_fast_payment

    https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.createfastpayment

    Метод используется для проведения платежей через он-лайн способы оплаты (пока только ЯД) в один клик.
    Для этого метод получает всю необходимую информацию, в том числе и данные для авторизации, если требуется.
    Метод может также включать заказы в овердрафт (отложенный платеж).

    Параметры:
        passport_id     целое число     uid в Яндекс.Паспорте
        login   строка  логин в Яндекс.Паспорте, один из параметров passport_id и login должен присутствовать
        client_id       целое число     клиент-владелец счета, не должен быть агентством
        paysys_id       целое число     способ оплаты, или погашения в случае овердрафта; сейчас поддерживается 1000=ЯД
        items   список хешей, формат ниже       список строчек счета
        overdraft       целое число     0 или 1 – включить счет в овердрафт
        payment_password        строка  платежный пароль, если требуется – см. коды ответов
        payment_token   строка  платежный токен, для схемы с хранением токенов вне Баланса 
        qty_is_amount     целое число    0 или 1 - признак того, что в строчках счета вместо количества товара (qty) передается сумма денег (amount)

    Формат хеша, описывающего строчки счета.
        service_id      целое число     ID сервиса в Балансе, Директ=7
        service_order_id        целое число     ID заказа на сервисе
        qty     десятичное число с точкой       единицы товара для зачисления на заказ или сумма денег

=cut

sub balance_create_fast_payment {
    my ($params, %O) = @_;

    my $result;

    eval {
        my ($res, $ret, $ret_str) = balance_call('Balance.CreateFastPayment',
                                 [$params],
                                 {write_log => 1, send_alert => 1, timeout => $O{timeout}});
        if ($res) {
            $result = $res;
        }
    };

    if ($@) {
        die {message => Dumper($@), dont_alert => 1};
    }

    if ($result->{status_code} eq "technical_error") {
        return 'PAYMENT_PROC_ERROR';
    } elsif ($result->{status_code} == 3) {
        if ($result->{overdraft} && ref($result->{overdraft}) eq 'HASH' && $result->{overdraft}{available_sum} > 0) {
            return 'OVERDRAFT_LIMIT_EXCEEDED';
        }
        return 'OVERDRAFT_NOT_AVAILABLE';
    } elsif ($result->{status_code} == 2) {
        return 'AMOUNT_TOO_BIG';
    } elsif ($result->{status_code} == 10) {
        return 'REFRESH_TOKEN';
    } elsif ($result->{status_code} == 20) {
        return 'YM_AUTH_API_ERROR';
    } elsif ($result->{status_code} == 21) {
        return 'YM_PAYMENT_API_ERROR';
    } elsif ($result->{status_code} == 30) {
        return 'PAYMENT_REG_ERROR';
    } elsif ($result->{status_code} == 31) {
        if ($result->{ym_error} && $result->{ym_error} eq 'limit_exceeded') {
            return 'LIMIT_EXCEEDED';
        } elsif ($result->{ym_error} && $result->{ym_error} eq 'not_enough_funds') {
            return 'NOT_ENOUGH_FUNDS';
        } else {
            return 'PAYMENT_PROC_ERROR';
        }
    } elsif ($result->{status_code} == 0) {
        return $result;
    }
}

=head2 balance_create_invoice

    Параметры:
    1.  Паспортный uid оператора (operator_uid).
    2. Хэш параметров счета
        RequestID        целое   Идентификатор недовыставленного счёта
        PaysysID        целое   Идентификатор платёжной системы
        PersonID        целое   Идентификатор плательщика
        Credit  булево значение         Выставлять счет в кредит
        ContractID      целое   Идентификатор договора 

=cut

sub balance_create_invoice {

    my ($params, %O) = @_;

    my $response;

    eval {
        my ($res, $ret, $ret_str) = balance_call('Balance.CreateInvoice', $params, {write_log => 1, timeout => $O{timeout}});
        if ($res) {
            $response = $res;
        }
    };

    return $response if $response;

    if (UNIVERSAL::isa($@ => "HASH") && $@->{faultString}) {
        # soap fault
        return balance_parse_fault_xml($@->{faultString})->{code};
    } elsif ($@) {
        if ($@ =~ /(CREDIT_LIMIT_EXEEDED|ILLEGAL_CONTRACT)/) {
            return $1;
        }
        die {message => Dumper($@), dont_alert => 1};
    }
    return '';
}

=head2 balance_get_url_for_direct_payment

    Получает данные, необходимые для прямого платежа за директ (DIRECT-12610).
    Работает с методом Balance.GetOrdersDirectPaymentNumber: https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.getordersdirectpaymentnumber

    Принимает именованные параметры:
        operator_uid — паспортный UID оператора (человека, который инициирует действие; в большинстве случаев это $UID), обязательный параметр
        cids         — ссылка на список с CIDами кампаний, за которые собираются делать прямой платёж, обязательный параметр
        ServiceID    — идентификатор сервиса в Балансе, не обязательный параметр; если не указан будет использоваться значение 7, соответствующее Директу

    Возвращает URL страницы биллинга с информацией о дальнейших действиях по совершению прямого платежа. При ошибках — умирает.

    $direct_payment_url = balance_get_url_for_direct_payment( operator_uid => $UID, cids => [$cid1, $cid2, 1234567] );
    $direct_payment_url => 'http://balance-test.yandex.ru/direct-payment.xml?payment_number=5007263';

=cut

sub balance_get_url_for_direct_payment {
    my (%O) = @_;

    die 'operator_uid не указан или указан неверно: ' . Dumper(\%O) unless $O{operator_uid} && is_valid_int($O{operator_uid}, 0);
    die 'cids не указан или указан неверно' . Dumper(\%O) unless $O{cids} && ref($O{cids}) eq 'ARRAY' && @{$O{cids}};
    die 'ServiceID указан неверно: ' . Dumper(\%O) if $O{ServiceID} && !is_valid_int($O{ServiceID}, 0);

    my $params = [ $O{operator_uid}, [ map { { ServiceID => $O{ServiceID} || 7, ServiceOrderID => $_ } } @{$O{cids}} ] ];
    # в случае ошибки balance_call возмущается в ppc-admin@ и умирает
    my ($response) = balance_call('Balance.GetOrdersDirectPaymentNumber', $params, {send_alert => 1});
    if ($response && ref($response) eq 'HASH' && $response->{Url}) {
        return $response->{Url};
    } else {
        die 'получен странный результат вызова Balance.GetOrdersDirectPaymentNumber: ' . Dumper($response);
    }
}

=head2 balance_get_currency_rate

    Получить по балансовому коду валюты и дате ее курс в рублях, если дата пустая - возвращается последний.
    Работает с методом Balance.GetCurrencyRate: https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.getcurrencyrate

    Принимает позиционные параметры:
        currency_code — код валюты ('USD', 'UAH', 'KZT', 'EUR', 'BYR', ...); обязательный параметр
        date          — дата, на которую нужен курс в формате YYYY-MM-DD или YYYYMMDD; необязательный параметр
    Возвращает значение курса. Если курс на указанную дату не найден, возвращает 0. В случае каких-либо ошибок умирает.

    $rate = balance_get_currency_rate('USD');
    $rate = balance_get_currency_rate('USD', '2011-09-25');

=cut

sub balance_get_currency_rate {
    my ($currency_code, $date) = @_;

    die 'currency_code не указан или указан неверно: ' . Dumper($currency_code) unless $currency_code && !ref($currency_code);
    die 'date не указан или указан неверно: ' . Dumper($date) unless !defined $date || ($date && is_valid_date($date));

    $date = '' unless defined $date;
    my $params = [$currency_code, $date];

    my ($response) = balance_call('Balance.GetCurrencyRate', $params, {send_alert => 1});
    if ($response && ref($response) eq 'ARRAY' && @$response) {
        my ($response_code, $status, $result) = @$response;
        if ($response_code == 0 && $status eq 'SUCCESS' && $result && ref($result) eq 'HASH'
        && $result->{currency} && $result->{currency} eq $currency_code
        && $result->{date} && $result->{date} =~ /^(\d{4}-?\d{2}-?\d{2})/ # баланс возвращает дату в формате ISO-8601: YYYYMMDDTHH:MM:DD
        && $result->{rate} && $result->{rate} > 0) {
            # если у баланса нет курса на запрошенную дату, возвращается последний имеющийся курс с указанием соответствующей даты
            my $balance_date = $1;
            if (!$date || mysql2unix($balance_date) == mysql2unix($date)) {
                return $result->{rate};
            } else {
                # курс не на ту дату, что просили
                return 0;
            }
        } elsif ($response_code == 1 && $status eq 'NOT_FOUND') {
            return 0;
        }
    }
    die 'получен странный результат вызова Balance.GetCurrencyRate: ' . Dumper($response);
}

=head2 balance_get_direct_budget

    Получить из биллинга информацию о бюджете пользователя за последние 12 месяцев,
    текущей скидке и предыдущей/следующей скидках.
    Для того, чтобы уменьшить объём ответа, есть два параметра - mod и rem.
    Выдаётся информация только по тем ClientID, у ClientID % mod == rem

    Параметры позиционные:
    mod - модуль, по которому делится ClientID
    rem - остаток от деления

    Возвращает ссылку на массив из хэшей с полями:
        CLIENT_ID
        DISCOUNT_PCT - текущая скидка, в процентах
        BUDGET - бюджет за 12 месяцев
        PREV_DISCOUNT_PCT - предыдущая скидка
        PREV_DISCOUNT_QTY - бюджет, при котором скидка откатится на предыдущее значение
        NEXT_DISCOUNT_QTY - следующая скидка
        NEXT_DISCOUNT_PCT - бюджет, при котором скидка увеличится

=cut
sub balance_get_direct_budget {
    my ($mod, $rem) = @_;

    my ($response) = balance_call('Balance.GetDirectBudget',
                                  [{Mod => $mod, Rem => $rem}],
                                  {send_alert => 1, timeout => 300, recall => 2}
        );
    my $ret = _parse_tsv($response);
    $ret = [map {$_->{BUDGET} ||= 0; $_->{DISCOUNT_PCT} ||= 0; $_} @$ret];
    return $ret;
}

=head2

    Отправляет открутки по заказу
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.updatecampaigns

    Принимает массив хешей:
    [
        {
            'ServiceID':7,
            'ServiceOrderID': 7081025,
            'Bucks': '47368.562971',
            'Clicks': '21694.000000',
            'Days': '552.000000',
            'dt': '2015-03-10',
            'Money': '0.000000',
            'Shows': '246600.000000',
            'Stop': 0,
        },
        {
            'ServiceID':7,
            'ServiceOrderID': 7081026,
            ...
        },
        {
            'ServiceID':7,
            'ServiceOrderID': 7081027,
            ...
        }
    ]

    Возвращает хеш с ServiceId, ServiceOrderID и флагом 1 (если успех) или 0 (если ошибка):
    {'7': {'7081025': 1, '7081026': 1, '7081027': 0}}

=cut

sub balance_update_campaigns {
    my ($campaigns) = @_;

    die 'Invalid data format - array of hashes requred' unless ref $campaigns eq 'ARRAY';
    return balance_call('Balance.UpdateCampaigns', [$campaigns], {write_log => 1});
}

# ..................................................................................................
# ..................................................................................................
# ........... Старый код из Common'а (тег 'XMLRPC').................................................
# ..................................................................................................
# ..................................................................................................
### XMLRPC
# TODO
#   * переименовать функции, чтобы у всех был префикс balance_

=head2 is_balance_subclient

 if (is_balance_subclient($UID)) {
    print STDERR "$UID is subclient in balance\n";
 }

=cut

sub is_balance_subclient
{
    my $uid = shift;
    my $client_data;
    eval {
        $client_data = get_client_by_uid($uid, 3);
    };
    if ($@) {
        print STDERR "is_balance_subclient eval error: $@\n";
        return 0;
    }
    return 0 if ! defined $client_data || ref($client_data) ne 'HASH';
    return((! $client_data->{IS_AGENCY} && $client_data->{AGENCY_ID} && $client_data->{AGENCY_ID} != $client_data->{CLIENT_ID}) ? 1 : 0);
}


=head2 get_client_by_uid

=cut

# TODO Хорошо бы внутри использовать balance_find_client
# Препятствия: настраиваемый таймаут (используется в is_balance_subclient), параметр recall для balance_call

sub get_client_by_uid {
    my $uid = shift;
    my $timeout = shift || 3;

    my $hash = { AgencySelectPolicy => 1,
                PassportID => $uid,
                PrimaryClients => 1
            };

    my @params = ($hash);
    my @res = eval { balance_call('Balance.FindClient', \@params, {timeout => $timeout, recall => 1}) };
    
    if ( $@ || $res[0] ) {
        die "get_client_by_uid: XMLRPC: FindClient() error: $@  $res[0] $res[1]\n";
    }

    my $client = $res[2]->[0] || {}; 

    return $client;
}


=head2 get_clientid_by_uid

 $clientid = get_clientid_by_uid($uid);

 return ClientID - if found
 return 0 - if not found
 return undef - if error

=cut

sub get_clientid_by_uid {
    my $uid = shift;

    my $res = get_client_by_uid( $uid );
    if (ref($res) eq 'HASH') {
        return $res->{CLIENT_ID} || 0;
    } else {
        return undef;
    }
}

# ..................................................................................................

=head2 create_client_id_association

 Создание в балансе связки между uid и ClientID
 если в итоге представитель единственный, то передаем в биллинг признак "Главный"

 my $success = create_client_id_association($uid, $client_id, $operator_uid);

 return 1 если представителя успешно привязали

=cut

sub create_client_id_association($$$) {
    my ($uid, $client_id, $operator_uid) = @_;
    $operator_uid = $uid unless $operator_uid;

    my @params = ($operator_uid, $client_id, $uid);
    my ($code) = eval {
        my ($balance_call_res) = balance_call('Balance.CreateUserClientAssociation', \@params, {write_log => 1});

        # если у клиента представитель один, то передаем в баланс признак "Главный"
        my $balance_reps = balance_list_client_passports($operator_uid, $client_id);

        if (! $balance_call_res && 1 == scalar(@$balance_reps)) {
            my ($res, $ret, $ret_str) = balance_call('Balance.EditPassport', [$operator_uid, $uid, {IsMain => 1}], {write_log => 1});
            die "Balance error in create_client_id_association(): Balance.EditPassport return '$ret', '$ret_str'" if $ret;
        }

        # 04006 -- Если указанный пользователь уже ассоциирован с указанным клиентом
        if ($balance_call_res == 4006) {
            warn "указанный пользователь уже ассоциирован с указанным клиентом: ".to_json(\@params);
            $balance_call_res = 0;
        }
        $balance_call_res;
    };

    if ( $@ || $code ) {
        return 0;
    }

    return 1;
}

=head2 get_equal_clients

  @equal_client_ids = get_equal_clients($Client_ID);
  return (0, $arrref_client_ids) on no errors

  Проверять можно так (параметры -- ClientID): 
  perl -Iprotected -ME -e 'for (@ARGV) {p get_equal_clients($_)}' 1290967

=cut

sub get_equal_clients
{
    my $ClientID = shift;

    my @params = ($ClientID);
    my @res = eval { balance_call('Balance.GetEqualClients', \@params) };
    if ( $@ || $res[0] ) {
        return 1;
    }

    if ($res[0] == 0) { # no errors
        return 0, [map {$_->{CLIENT_ID}} @{$res[2]}];
    }

    return undef;
}




# Удаление в балансе и в директе связки между uid и ClientID
sub remove_client_id_association
{
    my ($operator_uid, $uid, $client_id) = @_;

    my @params = ($operator_uid, $client_id, $uid);
    my ($code) = eval { balance_call('Balance.RemoveUserClientAssociation', \@params, {write_log => 1}) };
    if ( $@ || $code ) {
        return 0;
    }

    return 1;
}


### /XMLRPC
# ..................................................................................................
# ..................................................................................................
# ........... Окончание блока старого кода из Common'а (тег 'XMLRPC')...............................
# ..................................................................................................
# ..................................................................................................



=head2 balance_get_firm_country_currency

    Метод возвращает для клиента, по его привязке к определённой стране, саму страну,
    фирму обслуживания и разрешённые для него валюты с указанием флажка "нерезидент".
    Если клиенту не была присвоена страна (см. Balance.CreateClient: region_id),
    то берется список фирм обслуживания, для которых у клиента были заведены
    плательщики.

    Если клиент не передан, возвращается список всех стран, для каждой из которых
    указывается фирма обслуживания, список валют и признаки "нерезидент" и "агентство".

    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getfirmcountrycurrency

    $firm_country_currencies = Yandex::Balance::balance_get_firm_country_currency();
    $client_firm_country_currencies = Yandex::Balance::balance_get_firm_country_currency($client_id, AgencyID => $agency_clientid, ServiceID => 7, service_filter => 1, currency_filter => 1, timeout => 60);
    $firm_country_currencies = $client_firm_country_currencies = [
        {
            agency => 0|1, # признак агентства BALANCE-21954
            region_id => 225, # ID страны по Геобазе (225=Россия, 187=Украина, 159=Казахстан, 84=США, 126=Швейцария, 983=Турция, ...)
            region_name => , # Название страны по Геобазе на русском
            region_name_en => , # Название страны по Геобазе на английском
            currency => 'RUB'|'UAH'|'KZT'|'USD'|..., # код валюты ISO (RUB, UAH, KZT, USD, EUR, CHF, TRY, ...)
            resident => 0|1, # признак того, что клиенты из данной страны платят как нерезиденты
            firm_id => 1 # ID фирмы по метабазе (1="ООО Яндекс", 2="ООО Яндекс.Украина", 3="КазНет Медиа", 4="Yandex Inc", 7="Yandex Europe AG", 8="Yandex Turkey", ...)
            # может отсутствовать
            convert_type_modify => 1, # клиента можно конвертировать в эту страну/валюту без копирования, DIRECT-41304
        },
        [...]
    ];

=cut

sub balance_get_firm_country_currency {
    my ($client_id, %O) = @_;

    my ($balance_ret) = balance_call('Balance.GetFirmCountryCurrency', [{
        client_id => $client_id,
        agency_id => $O{AgencyID},
        service_id => ($O{ServiceID} || 7),
        service_filter => ($O{service_filter} ? 1 : 0),
        currency_filter => ($O{currency_filter} ? 1 : 0),
    }], {send_alert => 1, timeout => $O{timeout}});
    die 'Strange return value from Balance.GetFirmCountryCurrency' unless $balance_ret && ref($balance_ret) eq 'ARRAY';
    my ($code, $msg, $firm_country_currencies) = @$balance_ret;
    if ($code) {
        die "Balance error: Balance.GetFirmCountryCurrency return FAIL '".($code||'')."', '".($msg||'')."'";
    }
    return $firm_country_currencies;
}

=head2 balance_get_client_currencies

    Получить по ClientID список использованных им валют, вычисляется по оплаченным
    счетам клиента.

    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getclientcurrencies

    $currencies = balance_get_client_currencies($client_id);
    $currencies = ['RUB','USD','TRY'];

=cut

sub balance_get_client_currencies {
    my ($client_id) = @_;

    my ($balance_ret) = balance_call('Balance.GetClientCurrencies', [$client_id], {send_alert => 1});
    die 'Strange return value from Balance.GetClientCurrencies' unless $balance_ret && ref($balance_ret) eq 'ARRAY';
    my ($code, $msg, $currencies) = @$balance_ret;
    if ($code) {
        die "Balance error: Balance.GetClientCurrencies return FAIL '".($code||'')."', '".($msg||'')."'";
    }
    return $currencies;
}

=head2 balance_get_direct_discount

    Отдает по всем клиентам информацию о рассчитанных скидках на Директ.

    Пераметры передаются в виде одного параметра-хеша с двумя ключами: Mod и Rem.
    Этими ключами можно регулировать размер пачки данных на выходе.
    Возвращаются данные по всем клиентам, для которых (client_id % Mod) == Rem.

    Для клиента может быть несколько записей с разными периодами, это и есть график.

    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getdirectdiscount

    $discounts_data = balance_get_direct_discount(mod => $mod, rem => $rem, timeout => 60);
    $discounts_data => [
        {
            CLIENT_ID => '12345', # ID клиента
            START_DT => '2012-11-04', # дата начала действия скидки (включительно)
            END_DT => '2012-11-15', # дата окончания действия скидки (исключительно)
            DISCOUNT => '0', # процент скидки
            UPDATE_DT => '2012-11-14T01:32:12', # дата и время обновления информации
        },
        ...
    ];

=cut

sub balance_get_direct_discount {
    my (%O) = @_;

    die "invalid mod and rem values given: mod=$O{mod}; rem=$O{rem}" unless (is_valid_int($O{mod}, 1) && is_valid_int($O{rem}, 0)) || (!defined $O{mod} && !defined $O{rem});
    my $mod = (defined $O{mod}) ? $O{mod} : 1;
    my $rem = (defined $O{rem}) ? $O{rem} : 0;

    my ($discounts_data_str) = balance_call('Balance.GetDirectDiscount', [{Mod => $mod, Rem => $rem}], {send_alert => 1, timeout => $O{timeout}});
    die 'Strange return value from Balance.GetDirectDiscount' unless $discounts_data_str && ref($discounts_data_str) eq '';

    return _parse_tsv($discounts_data_str);
}

=head2 balance_get_client_nds

    Отдает по всем клиентам, у которых установлен region_id, информацию о 
    действующей ставке НДС с указанием периода времени.

    Параметры передаются в виде одного параметра-хеша с двумя ключами: Mod и Rem.
    Этими ключами можно регулировать размер пачки данных на выходе.
    Возвращаются данные по всем клиентам, для которых (client_id % Mod) == Rem.

    https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.getclientnds

    $nds_data = balance_get_direct_nds(mod => $mod, rem => $rem, timeout => 60);
    $nds_data => [
        {
            CLIENT_ID => '12345', # ID клиента
            DT => '2004-01-01', # дата начала действия ставки НДС (включительно)
            NDS_PCT => '0', # процент ставки налога
        },
        ...
    ];

=cut

sub balance_get_client_nds {
    my (%O) = @_;

    die "invalid mod and rem values given: mod=$O{mod}; rem=$O{rem}" unless (is_valid_int($O{mod}, 1) && is_valid_int($O{rem}, 0)) || (!defined $O{mod} && !defined $O{rem});
    my $mod = (defined $O{mod}) ? $O{mod} : 1;
    my $rem = (defined $O{rem}) ? $O{rem} : 0;
    $mod = XMLRPC::Data->type('string')->value($mod);
    $rem = XMLRPC::Data->type('string')->value($rem);

    my ($nds_data_str) = balance_call('Balance.GetClientNDS', [{Mod => $mod, Rem => $rem, ServiceID => 7}], {send_alert => 1, timeout => $O{timeout}});
    die 'Strange return value from Balance.GetDirectNDS' unless $nds_data_str && ref($nds_data_str) eq '';

    return _parse_tsv($nds_data_str);
}

=head2 balance_get_completion_history

    Метод запрашивает историю откруток.
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getcompletionhistory

    $completion_history = balance_get_completion_history(ServiceID => 7, cids => [$cid1, $cid2, ...], start_date => '20120101', end_date => '20121201);
    $completion_history => [
        {
            'QTY' => '8.8', # потрачено фишек
            'START_DT' => '2012-09-17',
            'ORDER_ID' => $cid1,
            'SUM' => '227.04', # потрачено реальных денег
        },
        ...
    ];

=cut

sub balance_get_completion_history {
    my (%O) = @_;

    die "invalid ServiceID or no ServiceID given" unless is_valid_int($O{ServiceID}, 0);
    die 'no cids' unless $O{cids} && ref($O{cids}) eq 'ARRAY' && @{$O{cids}};

    my %args = (Orders => $O{cids});
    $args{StartDT} = $O{start_date} if $O{start_date};
    $args{EndDT} = $O{end_date} if $O{end_date};
    my ($completion_str) = balance_call('Balance.GetCompletionHistory', [$O{ServiceID}, \%args], {send_alert => 1, timeout => $O{timeout}});
    die 'Strange return value from Balance.GetCompletionHistory' unless $completion_str && ref($completion_str) eq '';

    return _parse_tsv($completion_str);
}

=head2 balance_get_direct_balance

    Метод возвращает баланс по заказам клиента в указанной валюте, на которых есть неоткрученные средства.
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getdirectbalance

    $iso_4217_currency_code = 'RUB' | 'USD' | 'UAH' | ...;
    my $orders_data = balance_get_direct_balance(ClientID => $client_id, CurrencyCode => $iso_4217_currency_code);
    $orders_data => [
        {
            'CURRENT_SUM' => '6263.42', # зачислено на заказы (в новой валюте)
            'COMPLETION_SUM' => '5874.83', # откручено (в новой валюте)
            'ORDER_ID' => $cid1, # ID заказа на сервисе
        },
        ...
    ];

=cut

sub balance_get_direct_balance {
    my (%O) = @_;

    die "invalid ClientID or no ClientID given" unless is_valid_int($O{ClientID}, 0);
    die "invalid CurrencyCode or no CurrencyCode given" unless $O{CurrencyCode};

    my ($orders_str) = balance_call('Balance.GetDirectBalance', [{ClientID => $O{ClientID}, CurrencyCode => $O{CurrencyCode}}], {send_alert => 1, timeout => $O{timeout}});
    return _parse_tsv($orders_str);
}

=head2 balance_test_get_max_orderserviceid

    Тестовый метод возвращает максимальный ServiceOrderID (для Директа это cid) для сервиса согласно ServiceID.
    Требуется на ppcdev и ТС Директа после копирования базы из продакшена.
    Сервис в Балансе поднят на отдельном от основного проекта инстансе - требуется указать его адрес. 

    my $max_balance_cid = Yandex::Balance::balance_test_get_max_orderserviceid(7, BALANCE_XMLRC_URL => "...");

=cut

sub balance_test_get_max_orderserviceid
{
    my (%O) = @_;

    return balance_call('TestBalance.GetMaxServiceOrderID', [$O{ServiceID}], {BALANCE_XMLRPC_URL => $O{BALANCE_XMLRPC_URL}, timeout => $O{timeout}});
}

=head2 balance_test_get_max_clientid

    Тестовый метод возвращает максимальный ClientID.
    Требуется на ppcdev и ТС Директа после копирования базы из продакшена.
    Сервис в Балансе поднят на отдельном от основного проекта инстансе - требуется указать его адрес.

    my ($max_balance_client_id) = Yandex::Balance::balance_test_get_max_clientid(BALANCE_XMLRPC_URL => "...");

=cut

sub balance_test_get_max_clientid {
    my %O = @_;
    return balance_call('TestBalance.GetMaxClientID', [], {BALANCE_XMLRPC_URL => $O{BALANCE_XMLRPC_URL}, timeout => $O{timeout}});
}

=head2 balance_test_update_client_id

    Тестовый метод устанавливает следующее значение для сиквенса ClientID.
    Требуется на ppcdev и ТС Директа после копирования базы из продакшена.
    Сервис в Балансе поднят на отдельном от основного проекта инстансе - требуется указать его адрес.

    my ($new_max_balance_client_id) = Yandex::Balance::balance_test_update_client_id(NewMaxClientID => 12345, BALANCE_XMLRPC_URL => "...");

=cut

sub balance_test_update_client_id {
    my %O = @_;
    return balance_call('TestBalance.UpdateClientID', [$O{NewMaxClientID}], {BALANCE_XMLRPC_URL => $O{BALANCE_XMLRPC_URL}, timeout => $O{timeout}});
}

=head2 balance_get_currency_products

    Возвращает словарь, в котором ключи - ISO коды валют, а значения - id валютных продуктов директа в биллинге
    На вход принимает ID сервиса в Баланса.

    $currency2product = balance_get_currency_products( $Settings::SERVICEID{direct} );
    $currency2product => {
        'RUB' => 503162,
        'USD' => 503163,
        ...
    };

    https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.getcurrencyproducts

=cut

sub balance_get_currency_products {
    my ($serviceid) = @_;

    my ($products) = balance_call('Balance.GetCurrencyProducts', [$serviceid], {send_alert => 1, write_log => 1});
    return $products;
}

# --------------------------------------------------------------------

=head2 balance_update_notification_url

Меняем урл для нотификаций из баллинга (работает только на тестовом биллинге)
https://wiki.yandex-team.ru/balance/xmlrpc/#balance.updatenotificationurl

    perl -ME -E 'p balance_update_notification_url(7, "https://8777.beta-precise.direct.yandex.ru/xmlrpc")'
    perl -ME -E 'p balance_update_notification_url(7, "https://test-direct.yandex.ru/xmlrpc")'

=cut

sub balance_update_notification_url($$) {
    my ($serviceid, $url) = @_;

    return balance_call('Balance.UpdateNotificationUrl', [$serviceid, $url], {write_log => 1});
}

=head2 balance_get_orders_info

    Возвращает информацию о заказах по данным Баланса

    $orders_info = balance_get_orders_info(\@cids);
    $orders_info = [
        {
            completion_qty => $sum_spent,
            consume_qty => $sum,
            GroupServiceOrderID => $wallet_cid, # код главного заказа внутри сервиса в едином счете, которому будет привязан заказ
            id => ???
            product_id => $product_id,
            ServiceID => 7,
            ServiceOrderID => $cid,
        }
    ];

    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getordersinfo

=cut

sub balance_get_orders_info {
    my ($cids) = @_;

    my ($orders) = balance_call('Balance.GetOrdersInfo', [[map {{ServiceID => 7, ServiceOrderID => $_}} @$cids]]);
    return $orders;
}


=head2 balance_add_client_domain

    Метод используется для добавления и изменения доменов пользователей.
    https://wiki.yandex-team.ru/Balance/XmlRpc#balance.addclientdomain

    Параметры позиционные:
    raw_data - ссылка на массив хешей следующего формата, описывающего информацию о доменах клиента:
    {
        clientId    - ID клиента
        syncId      - идентификатор посылки от Директа
                        (используется балансом для версионирования записей)
        logDate     - дата посылки от Директа
        removed     - признак удаления домена
        domain      - домен
        type        - источник домена
        recordid    - ???
    }
    Параметры именованные:
        timeout     - таймаут запроса, в секундах

    Результат: 
        1 -- порядок
        0 -- непорядок.

=cut
# хеш соответствия имени ключа XML-типу
my %type_by_key = (
    syncId      => 'string',
    removed     => 'boolean',
    recordid    => 'int',
    clientId    => 'int',
    domain      => 'string',
    logDate     => 'dateTime',
    type        => 'string',
);
# значение по-умолчанию last_accepted_sync_id (в балансе не используется)
my $default_sync_id = 0;
# значение по-умолчанию last_accepted_sync_log_date (в балансе не используется)
my $default_logtime = '20000101T00:00:00';
sub balance_add_client_domain {
    my ($raw_data, %O) = @_;

    my $params = [
        XMLRPC::Data->type($type_by_key{syncId} => $default_sync_id),
        XMLRPC::Data->type(dateTime => $default_logtime),
        [
            map {
                hash_kv_map {
                    my ($k, $v) = @_;
                    exists $type_by_key{$k} ? XMLRPC::Data->type( $type_by_key{$k} => $v ) : $v;
                } $_
            } @$raw_data
        ],
    ];

    my ($ret) = balance_call('Balance.AddClientDomain', $params, {
        recall      => 1,
        timeout     => $O{timeout},
        write_log   => 0,
    });

    # формально проверяем ответ, что он похож на ожидаемый
    unless ($ret && ref $ret eq 'ARRAY' && $ret->[1]) {
        return 0;
    }
    return 1;
}

=head2 balance_get_nds_info

    Возвращает данные о НДС во всех странах за сегодня
    https://wiki.yandex-team.ru/balance/xmlrpc#balance.getndsinfo

    Принимает позиционные параметры:
        ServiceID
    И необязательные именованные параметры:
        timeout

    $nds_info = Yandex::Balance::balance_get_nds_info($service_id, timeout => 2);
    $nds_info => [
        AGENCY =>
        REGION_ID =>
        FIRM_ID =>
        RESIDENT =>
        NDS_PCT =>
        NSP_PCT =>
    ];

=cut

sub balance_get_nds_info {
    my ($service_id, %O) = @_;

    my ($response) = balance_call('Balance.GetNDSInfo', [$service_id], {send_alert => 1, timeout => $O{timeout}});
    return _parse_tsv($response);

}

=head2 balance_get_direct_brand

    Отдает по всем клиентам информацию о брендах Директа
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getdirectbrand

    Принимает обязательные позиционные параметры:
        - mod — делитель ClientID
        - rem — остаток от деления ClientID на mod
    и опциональные именованные:
        timeout — таймаут запроса в секундах

    Возвращает массив хешей:
    [
        {
            CONTRACT_ID => # ID договора-бренда в Биллинге
            DT => # Дата, на которую запрошено состояние бренда
            CLIENT_ID => # ID клиента, по которому найден бренд
            BRAND_CLIENT_ID => # ID клиента, который состоит в бренде с CLIENT_ID
            MAIN_CLIENT_CLASS_ID => # ID главного клиента
            BRAND_TYPE => # Тип бренда: 7 - техническая связка Директа, 77 - "бумажный" бренд Директа
        },
        ...
    ]

=cut

sub balance_get_direct_brand {
    my ($mod, $rem, %O) = @_;

    my ($response) = balance_call('Balance.GetDirectBrand',
        [{Mod => $mod, Rem => $rem}], {
            timeout => $O{timeout},
            recall => 0,
        },
    );
    return _parse_tsv($response);
}

=head2 balance_get_all_equal_clients

    Отдает всех клиентов, у которых есть эквиваленты
    https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.getallequalclients

    Не принимает параметры

    Возвращает массив хешей:
    [
        {
            CLIENT_ID => # ID клиента
            CLASS_ID => # ID главного клиента
        },
        ...
    ]

=cut

sub balance_get_all_equal_clients {
    my (%O) = @_;

    my ($response) = balance_call('Balance.GetAllEqualClients',
        undef, {
            timeout => $O{timeout},
            recall => 0,
        },
    );
    return _parse_tsv($response);
}

=head3 _parse_tsv

    Разбирает TSV (tab-separated) текст, полученный от Баланса.
    В первой строке он должен содержать названия полей.
    Возвращает ссылку на массив хешей.

    $data = _parse_tsv($balance_response);
    $data = [
        {FIELD1NAME => ..., FIELD2NAME => },
        ...
    ];

=cut

sub _parse_tsv {
    my ($tsv_string) = @_;

    my @lines = split /\n/, $tsv_string;
    # первая строка - имена полей
    my @fields = split /\t/, shift(@lines);
    my @ret = map {my @vals = split /\t/, $_; +{zip(@fields, @vals)}} @lines;

    return \@ret;
}

=head2 balance_mass_get_managers_info

    Возвращает хеш вида uid => manager_info, где manager_info - hashref с параметрами менеджера в балансе или undef,
    если менеджера в балансе нет.
    https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.getmanagersinfo

    Принимает позиционные параметры:
        manager_uids - arrayref со списком uid менеджеров, по которым нужно получить информацию
    И необязательные именованные параметры:
        timeout - по умолчанию 5 cекунд

    $muid2info = Yandex::Balance::balance_mass_get_managers_info([12345, 67890], timeout => 2);
    $muid2info => {
        12345 => {...},
        67890 => undef,
    };

=cut


sub balance_mass_get_managers_info {
    my ($manager_uids, %O) = @_;

    $O{timeout} ||= 5;
    my ($response) = balance_call('Balance.GetManagersInfo', [$manager_uids], {send_alert => 1, timeout => $O{timeout}});
    return $response;
}

=head2 balance_get_managers_info

    Возвращает хеш с информацией о менеджере с заданным uid или undef, если менеджера в балансе нет
    https://wiki.yandex-team.ru/Balance/xmlrpc/#balance.getmanagersinfo

    Принимает позиционные параметры:
        manager_uid - uid менеджера, по которому получаем информацию
    И необязательные именованные параметры:
        timeout - по умолчанию 5 cекунд

    $manager_info = Yandex::Balance::balance_get_nds_info($service_id, timeout => 2);
    $manager_info => {...};

=cut

sub balance_get_managers_info {
    my ($manager_uid, %O) = @_;

    return balance_mass_get_managers_info([$manager_uid], %O)->{$manager_uid};
}

=head2 balance_pay_request

    Инициирует процесс оплаты
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.payrequest
    Возвращает ссылку на массив

=cut

sub balance_pay_request {
    my $uid = shift;
    my $params = shift;

    return [ balance_call('Balance.PayRequest', [$uid, $params], { write_log => 1} ) ];
}

=head2 balance_check_request_payment

    Проверяет состояние оплаты
    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.checkrequestpayment
    Возвращает массив

=cut

sub balance_check_request_payment {
    my $uid = shift;
    my $request_id = shift;
    my $service_id = shift;

    return [ balance_call('Balance.CheckRequestPayment', [$uid, { RequestID => $request_id, ServiceID => $service_id }]) ];
}

1;
