package Yandex::Balance::Simple;

# $Id$

use strict;
use warnings;

use utf8;

use parent qw/Exporter/;

use Yandex::Balance qw();

our @EXPORT_OK = qw/
    balance_simple_create_basket
    balance_simple_pay_basket
    balance_simple_check_basket
    balance_simple_list_payment_methods
    balance_simple_create_binding
    balance_simple_do_binding
    balance_simple_unbind_card
/;

=pod

=head1 NAME

    Yandex::Balance::Simple

=head1 SYNOPSIS

    use Yandex::Balance::Simple qw/
        balance_simple_create_basket
    /;

    my $create_basket_options = {
        token => 'very_long_token_string',
        paymethod_id => 'card-x12345',
        user_ip => '127.0.0.1',
        currency => 'RUB',
        orders => [
            { service_order_id => ..., price => ..., },
            ...
        ],
        payment_timeout => 900,
        wait_for_cvn => 1,z
    };

    my $create_basket_responce;
    eval {
        $create_basket_responce = balance_simple_create_basket( $create_basket_options, timeout => 900 );
    };

=head1 DESCRIPTION

    Модуль/обёртка вокруг XMLRPC вызовов BalanceSimple API.

    https://wiki.yandex-team.ru/Balance/Simple/XMLRPC/

=head1 METHODS

=cut

=head2 BALANCE_SIMPLE_XMLRPC

    Адрес BalanceSimple API.

    https://wiki.yandex-team.ru/balance/simple/#sredyidostupy1

=cut

our $BALANCE_SIMPLE_XMLRPC;

=head2 DIRECT_SERVICE_TOKEN

    Идентификатор подключенного к Балансу сервиса, выдается Балансом, всегда передается
    первым позиционным параметром при вызове методов BalanceSimple.

    https://wiki.yandex-team.ru/Balance/Simple/XMLRPC/#formatzaprosaiotveta

=cut

our $DIRECT_SERVICE_TOKEN;

sub _call {
    my ($method, $params, %O) = @_;

    my $opts = {
        write_log => 1,
        send_alert => 1,
        BALANCE_XMLRPC_URL => $BALANCE_SIMPLE_XMLRPC,
        SERVICE_TOKEN => $DIRECT_SERVICE_TOKEN,
        %O
    };

    my ($res) = Yandex::Balance::balance_call($method, [ $opts->{SERVICE_TOKEN}, $params ], $opts);

    return $res;
}

=head2 balance_simple_create_basket()

    Создает корзину для оплаты с позициями из переданных заказов.

    https://beta.wiki.yandex-team.ru/Balance/Simple/xmlrpc/#balancesimple.createbasket

=cut

sub balance_simple_create_basket {
    my ($params, %opts) = @_;

    return _call('BalanceSimple.CreateBasket', $params, %opts);
}

=head2 balance_simple_pay_basket()

    Запускает оплату ранее созданной корзины.

    https://beta.wiki.yandex-team.ru/Balance/Simple/xmlrpc/#balancesimple.paybasket

=cut

sub balance_simple_pay_basket {
    my ($params, %opts) = @_;

    return _call('BalanceSimple.PayBasket', $params, %opts);
}

=head2 balance_simple_check_basket()

    Проверяет статус платежа по ранее созданной корзине.

    https://beta.wiki.yandex-team.ru/Balance/Simple/xmlrpc/#balancesimple.checkbasket

=cut

sub balance_simple_check_basket {
    my ($params, %opts) = @_;

    return _call('BalanceSimple.CheckBasket', $params, %opts);
}

=head2 balance_simple_list_payment_methods()

    Возвращает список доступных способов оплаты, доступных пользователю.
    В числе способов оплаты могут присутствовать привязанные кредитные карты,
    мобильные телефоны пользователя, кошелек Яндекс.Денег.

    https://wiki.yandex-team.ru/Balance/Simple/XMLRPC/#balancesimple.listpaymentmethods

=cut

sub balance_simple_list_payment_methods {
    my ($params, %opts) = @_;

    return _call('BalanceSimple.ListPaymentMethods', $params, %opts);
}

=head2 balance_simple_create_binding()

    Метод для создания заявки на привязку банковской карты к паспортному uid

    https://wiki.yandex-team.ru/Balance/Simple/XMLRPC/#balancesimple.createbinding

=cut

sub balance_simple_create_binding {
    my ($params, %opts) = @_;

    return _call('BalanceSimple.CreateBinding', $params, %opts);
}

=head2 balance_simple_do_binding()

    Возвращает по заявке url на WEB-страницу привязки.

    https://wiki.yandex-team.ru/Balance/Simple/XMLRPC/#balancesimple.dobinding

=cut

sub balance_simple_do_binding {
    my ($params, %opts) = @_;

    return _call('BalanceSimple.DoBinding', $params, %opts);
}


=head2 balance_simple_unbind_card()

    Отвязывает карту клиента.

    https://wiki.yandex-team.ru/Balance/Simple/XMLRPC/#balancesimple.unbindcard

=cut

sub balance_simple_unbind_card {
    my ($params, %opts) = @_;

    return _call('BalanceSimple.UnbindCard', $params, %opts);
}


1;
