package Yandex::Blackbox;

=pod
    $Id$
    Модуль для работы с blackbox.yandex.net
    https://doc.yandex-team.ru/blackbox/concepts/about.xml
=cut

use Direct::Modern;

use Storable qw/dclone/;
use Yandex::HTTP qw/http_fetch/;
use Yandex::Log;
use Yandex::Trace;
use Yandex::TVM2;

use XML::Simple;
use Data::Dumper;
use Time::HiRes;
use POSIX qw/strftime/;
use List::MoreUtils qw/any/;

use base qw/Exporter/;
our @EXPORT = qw/
  bb_set_log
  bb_sessionid bb_userinfo bb_oauth_token
  bb_method
  $BB_LOGIN $BB_EMAIL $BB_FIO $BB_NICKNAME $BB_REG_DATE
  $BB_SUID_DIRECT $BB_SUID_METRIKA
  $BB_SUID_STRONGPWD
  $BB_SUID_WEBMASTER
  $BB_SUID_FOTKI
  $BB_YANDEX_DOMAIN_LOGIN
  $NOAUTH_TIMEOUT
  $BLACKBOX_USE_TVM_CHECKER
  $USE_MULTISESSION_IN_SESSIONID
  /;


=head2 @Yandex::Blackbox::MASTER_DOMAINS

    DEPRECATED - МДА отрывают, все домены будут "главными". Не нужно использовать этот массив

    Домены, которые являются "главными" с точки зрения МДА,
    между ними не пробрасывается сессия.

=cut
our @MASTER_DOMAINS = (
    'yandex.ru', 
    'yandex.com',
    'yandex.com.tr',
    );

# адрес "настоящего" Чёрного Ящика, на который отправляются некоторые запросы, не эмулирующиеся в FakeBlackbox Песочницы
# соответственно, должен использоваться только внутри Sandbox::FakeBlackbox
# может переопределяться, например, для тестовой Песочницы
our $BLACKBOX_URL_BASE ||= 'http://blackbox.yandex.net/blackbox/';

our $BLACKBOX_URL ||= $BLACKBOX_URL_BASE;

our $BLACKBOX_TVM2_ID ||= '222';
our $BLACKBOX_USE_TVM_CHECKER = sub { return 0;};

our $BB_LOGIN        = 'accounts.login.uid';
our $BB_EMAIL        = 'account_info.email.uid';
our $BB_FIO          = 'account_info.fio.uid';
our $BB_NICKNAME     = 'account_info.nickname.uid';
our $BB_REG_DATE     = 'userinfo.reg_date.uid';
our $BB_SUID_DIRECT  = 'subscription.suid.14';
our $BB_SUID_STRONGPWD = 'subscription.suid.67';
our $BB_SUID_METRIKA = 'subscription.suid.48';
our $BB_SUID_WEBMASTER = 'subscription.suid.47';
our $BB_SUID_FOTKI = 'subscription.suid.5';
our $BB_YANDEX_DOMAIN_LOGIN = 'subscription.login.669';

our $DEBUG ||= 0;
our @TIMEOUTS = (0.1, 0.25, 0.5);
our $NOAUTH_TIMEOUT = 2*60*60;    #   Срок годности куки Session_id типа noauth:\d{10} в секундах

# при $LOG_ALL_REQUESTS = 1 в лог будут писаться все запросы и ответы, иначе - только при ошибке
our $LOG_ALL_REQUESTS ||= 0;
our $LOG_USE_SYSLOG ||= 1;

# если переменная LOG_FILE определена, файл открывается и в него пишется информация о ошибках
our ($LOG_FILE, $LOG_PROJECT);

# При включении будут запрашиваться все аккаунты для куки sessionid. Без параметра запрашивается только дефолтный.
our $USE_MULTISESSION_IN_SESSIONID ||= 0;

=head2 bb_set_log

    установка лога

=cut

sub bb_set_log {
    ($LOG_FILE, $LOG_PROJECT) = @_;
}

=head2 bb_sessionid

    проверка сессии, получение аттрибутов пользователя

=cut

sub bb_sessionid {
    my ($session_id, $user_ip, $host, $dbfields, $emails, $session_id2, $aliases) = @_;

    my $params = {sessionid => $session_id, emails => $emails};
    $params->{sslsessionid} = $session_id2 if $session_id2;

    return bb_method('sessionid', $params, $user_ip, $host, $dbfields, $aliases);
}

=head2 bb_oauth_token

    Проверка oauth токена, получение атрибутов пользователя
    $reqid - идентификатор запроса, нужен для провязки наших клиентских запросов с запросами в ББ,
    для возможных разбирательств по проблемам авторизации
    Параметр не обязательный 

=cut

sub bb_oauth_token {
    my ($oauth_token, $user_ip, $host, $dbfields, $emails, $reqid) = @_;
    return bb_method('oauth', {oauth_token => $oauth_token, emails => $emails,  reqid => $reqid},
                     $user_ip, $host, $dbfields
    );
}

=head2 bb_userinfo

    проверка сессии, получение аттрибутов пользователя

=cut

sub bb_userinfo {
    my ($params, $user_ip, $host, $dbfields, $emails) = @_;

    # если params не ссылка - то uid
    if (!ref $params) {
        $params = { uid => $params };
    }

    $params->{emails} ||= $emails;

    return bb_method('userinfo', $params, $user_ip, $host, $dbfields);
}

sub _params_copy_and_hide_token {
    my ($params) = @_;
    my $fixed_params = dclone($_[0]);
    if (exists $fixed_params->{oauth_token}) {
        my $token = $fixed_params->{oauth_token};
        $fixed_params->{oauth_token} = substr($token,0,4) . '..' . substr($token,-4);
    }
    return $fixed_params;
}

=head3 _get_log

    $log = _get_log();

=cut

sub _get_log {
    state $log;
    $log ||= Yandex::Log->new(
        log_file_name => 'blackbox.log',
        date_suf => '%Y%m%d',
        use_syslog => $LOG_USE_SYSLOG,
        no_log => ($LOG_USE_SYSLOG ? 1 : 0),
        syslog_prefix => 'external_service'
    );
    return $log;
}

=head2 bb_log

    bb_log(%param)

    вывести %param в лог-файл, в виде json

=cut

sub bb_log {
    my (%param) = @_;
    my $log = _get_log();
    $log->out(\%param);
}

=head2 bb_die

    bb_die(%params, error => "error message");

    записать параметры в лог и умереть с сообщением из error

=cut

sub bb_die {
    my (%param) = @_;
    bb_log(%param);
    die $param{error} // "BB: Unknown error!";
}

# выполнение метода ЧЯ
sub bb_method {
    my ($method, $params, $user_ip, $host, $dbfields, $aliases) = @_;
    my %log_request = ( method => $method, params => _params_copy_and_hide_token($params), user_ip => $user_ip, host => $host, dbfields => $dbfields, aliases => $aliases);
    if ($LOG_ALL_REQUESTS) {
        bb_log(%log_request);
    }

    my $profile = Yandex::Trace::new_profile("blackbox:$method");

    bb_die(%log_request, error => "Incorrect params") if !defined $user_ip || !defined $host;

    # если нужно, джойним dbfields
    if (ref($dbfields) eq 'ARRAY') {
        $dbfields = join ',', @$dbfields;
    }

    # создаём часть запроса с методом
    my %request_params = (
        method => $method,
        host => $host,
        regname => 'yes',
        userip => $user_ip,
    );
    if ($method && $method eq 'sessionid') {
        bb_die(%log_request, error => "No oauth_token") if !defined $params->{sessionid} || !$params->{sessionid};
        $request_params{sessionid} = $params->{sessionid};
        $request_params{sslsessionid} = $params->{sslsessionid} if exists $params->{sslsessionid};
        if ($USE_MULTISESSION_IN_SESSIONID) {
            $request_params{multisession} = 'yes';
        }
    } elsif ($method && $method eq 'userinfo') {
        # добавляем LOGIN для проверки существования uid
        if (!ref($dbfields) || !grep {$_ eq $BB_LOGIN} @$dbfields) {
            $dbfields = $dbfields ? "$dbfields,$BB_LOGIN" : $BB_LOGIN;
        }
        if ($params->{uid}) {
            $request_params{uid} = $params->{uid};
        } elsif ($params->{login}) {
            $request_params{login} = $params->{login};
            if ($params->{sid}) {
                $request_params{sid} = $params->{sid};
            }
        } else {
            bb_die(%log_request, error => "No login or uid for userinfo");
        }
        if ($params->{getphone}) {
            $request_params{getphones} = 'bound';
            $request_params{phone_attributes} = '103,107';
        }
    } elsif ($method && $method eq 'oauth') {
        bb_die(%log_request, error => "No sessionid") if !defined $params->{oauth_token} || !$params->{oauth_token};
        $request_params{oauth_token} = $params->{oauth_token};
        $request_params{direct_request_id} = $params->{reqid} if defined $params->{reqid};
    } else {
        bb_die(%log_request, error => "Unknown method: '$method'");
    }

    # http://doc.yandex-team.ru/blackbox/reference/MethodUserInfo.xml#emails-detailed
    # http://doc.yandex-team.ru/blackbox/reference/method-oauth.xml#emails-detailed
    # http://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml#emails-detailed
    if ($params->{emails}) {
        $request_params{emails} = (any {$_ eq $params->{emails}} qw/getall getyandex getdefault/) ? $params->{emails} : 'getall';
    }

    my $headers = {
        'User-Agent' => "Yandex::Blackbox/perl/1.00",
    };

    my $use_tvm = $BLACKBOX_USE_TVM_CHECKER->();
    my $tvm_ticket;
    if ($use_tvm) {
        $tvm_ticket = eval{Yandex::TVM2::get_ticket($BLACKBOX_TVM2_ID)}
            or bb_die(msg => "Cannot get ticket for $$BLACKBOX_TVM2_ID: $@");
    }

    $request_params{dbfields} = $dbfields||'';
    if ($aliases) {
        $request_params{aliases} = $aliases;
    }

    # пытаемся послать запрос несколько раз с разными таймаутами
    my $iter = 0;
    for my $timeout (@TIMEOUTS) {
        my $start_ts = Time::HiRes::time();
        $iter++;
        my $res;
        eval {
            if ($tvm_ticket) {
                $headers->{'X-Ya-Service-Ticket'} = $tvm_ticket;
            }
            my $content = http_fetch(GET => $BLACKBOX_URL, \%request_params,
                headers => $headers,
                timeout => $timeout,
                handle_params => {keepalive => 1},
                ipv6_prefer => 1,
                (($DEBUG) ? (log => _get_log()) : ()),
            );

            # парсим полученный XML
            my $simple = XML::Simple->new();
            $res = eval {
                XML::Simple::XMLin($content, ForceArray => [qw/dbfield address user/], ForceContent => 1, SuppressEmpty => 1);
            };

            print STDERR Dumper($res) if $DEBUG;

            # проверяем валидность ответа
            if ($method eq 'sessionid') {
                if ($@ || !defined $res || !defined $res->{status}) {
                    bb_die(%log_request, error => "Blackbox: xml parse error", error_desc => $@, response => $content);
                }

                # При включенном параметре multisession меняется формат ответа от blackbox
                if ($USE_MULTISESSION_IN_SESSIONID && defined $res->{default_uid} && defined $res->{user}) {
                    # Текущий выбранный пользователь в мультисессии
                    my $default_uid = $res->{default_uid}->{content};
                    my $user = $res->{user}->{$default_uid};

                    # Копируем поля текущего пользователя в корень res для обратной совместимости,
                    # когда запросы были без параметра multisession=yes
                    # https://docs.yandex-team.ru/blackbox/methods/sessionid#response_format
                    for my $field (keys %{$user}) {
                        $res->{$field} = $user->{$field};
                    }
                }

                # Признак пользователя почты для доменов: 1 - ПДД, 0 - обычный
                $res->{hosted} = $res->{uid}->{hosted};
                # Прописываем uid
                $res->{uid} = $res->{uid}->{content};
                # NB! в документации ЧЯ это поле отсутствует, но пользователи такие еще есть (лайт-аккаунты)
                $res->{liteuid} = $res->{liteuid}->{content};
                $res->{age} = $res->{age}->{content};

                # перерабатываем статусы
                $res->{valid} = $res->{status}->{id} == 0 || $res->{status}->{id} == 1;
                $res->{renew} = $res->{status}->{id} == 1;

                unless ($res->{status}{content} =~ /^(?:NOAUTH|EXPIRED|NEED_RESET|VALID|DISABLED|INVALID)$/i) {
                    bb_die(%log_request, error => "Invalid status returned: ".$res->{status}{content}." id:".$res->{status}{id});
                }
            } elsif ($method eq 'userinfo') {
                if ($res->{exception}) {
                    bb_die(%log_request, error => $res->{error}->{content});
                }

                $res->{hosted} = $res->{uid}->{hosted};
                # Прописываем uid
                $res->{uid} = $res->{uid}->{content};
                # NB! в документации ЧЯ это поле отсутствует, но пользователи такие еще есть (лайт-аккаунты)
                $res->{liteuid} = $res->{liteuid}->{content};

                if (!$res->{uid} || $res->{uid} !~ /^\d+$/ || !$res->{dbfield}->{$BB_LOGIN}->{content}) {
                    $res = undef;
                    return;
                }
            } elsif ($method eq 'oauth') {
                if ($@ || !defined $res || !defined $res->{status}) {
                    bb_die(%log_request, error => "Blackbox: xml parse error", error_desc => $@, response => $content);
                }

                unless ($res->{status}{content} =~ /^(?:VALID|DISABLED|INVALID)$/i) {
                    bb_die(%log_request, error => "Invalid status returned: ".$res->{status}{content}." id:".$res->{status}{id});
                }

                $res->{status} = $res->{status}->{content};
                $res->{error} = $res->{error}->{content};

                $res->{hosted} = $res->{uid}->{hosted};
                $res->{uid} = $res->{OAuth}->{uid}->{content};
                $res->{login} = $res->{OAuth}->{login}->{content};
                $res->{scope} = { map {$_ => 1} split ' ', ($res->{OAuth}->{scope}->{content}//'')};
                $res->{client_id} = $res->{OAuth}->{client_id}->{content};
            } else {
                bb_die(%log_request, error => "Unknown method $method");
            }

            # переделываем аттрибуты для удобства
            if (my $dbf = $res->{dbfield}) {
                for my $field (keys %{$dbf}) {
                    $dbf->{$field} = $dbf->{$field}->{content};
                }
            }
            if ($res->{display_name}) {
                $res->{display_name}->{name} = $res->{display_name}->{name}->{content};
                if (my $soc = $res->{display_name}->{social}) {
                    for my $field (keys %{$soc}) {
                        $soc->{$field} = $soc->{$field}->{content};
                    }
                }
            }
            if ($res->{have_password}) {
                $res->{have_password} = $res->{have_password}->{content};
            }

            #Разбераем телефонные номера. Нам нужен номер, у которого есть атрибут 107 - дефолтность
            my $phone = $res->{phones}->{phone};
            if ($phone) {
                # BB отдает немного разную структуру в случае если у пользователя 1 номер телефона, или если несколько.
                if (exists $phone->{attribute} && exists $phone->{id}) {
                    # Значит представлен только один номер телефона. Переструктурируем хеш, под формат как в случае нескольких номеров телефона.
                    $phone->{delete $phone->{id}}->{attribute} = delete $phone->{attribute};
                }
                foreach my $phone_id (keys %$phone) {
                    my ($is_defauld_number, $number) = (0, 0);
                    foreach my $attr (@{$phone->{$phone_id}->{attribute}}) {
                        $is_defauld_number = 1 if ($attr->{type} == 107 && $attr->{content} == 1);
                        $number = $attr->{content} if ($attr->{type} ==  103);
                    }
                    if ($is_defauld_number && $number) {
                        $res->{phone_number} = $number;
                        last;
                    }
                }
                delete $res->{phones};
            }

            # список емейлов
            if (my $emails = $res->{'address-list'}{'address'}) {
                $emails = [] unless ref($emails) eq 'ARRAY';
                $res->{'address-list'}{'address-list'} = [
                    map { $_->{content} }
                    sort { ($a->{native}||0) <=> ($b->{native}||0) }
                    @$emails
                ];
                if (my ($default_email_rec) = grep {$_->{default}} @$emails) {
                    $res->{default_email} = $default_email_rec->{content};
                }
            }
        };
        # проверяем успешность
        my $elapsed = Time::HiRes::time() - $start_ts;
        if (!$@) {
            log_access(0, $elapsed);
            return $res;
        }
        log_access(1, $elapsed);
        warn "BB error: $@\n" if $DEBUG;
        # делаем паузу, если нужно
        if ($iter < @TIMEOUTS) {
            my $to_sleep = $timeout - $elapsed;
            Time::HiRes::sleep($to_sleep) if $to_sleep > 0;
        }
    }
    bb_die(%log_request, error => "BB fatal error: $@");
}

# форматирование timestamp для логов с милисекундами
sub format_ts {
    my $ts = shift;
    return strftime("%Y-%m-%d %H:%M:%S", localtime $ts).".".sprintf( "%03d", 1000*($ts-int($ts)) );
}

# вывод в лог информации о обращении к чя
sub log_access {
    my ($res_code, $answer_time) = @_;
    return if !$LOG_FILE;
    open(my $log_fh, ">>", $LOG_FILE) || warn "Can't open '$LOG_FILE': $!";
    printf $log_fh "%d\t%s\t%d\t%d\n", time(), ($LOG_PROJECT||'-'), $res_code, 1000*$answer_time;
    close $log_fh;
}

1;
