##################################################
#
#  Direct.Yandex.ru
#
#  Captcha
#
#  authors:
#    Maxim Kuzmin <kuzmin@yandex-team.ru>
#    Sergey Mudrik <msa@yandex-team.ru>
#    Sergey Zhuravlev <zhur@yandex-team.ru>
#
#  $Id$
#
#  (c) 2004 - 2007 Yandex
#
##################################################

=head1 NAME

Captcha

=head1 DESCRIPTION

  Captcha tools

=cut

package Yandex::Captcha;

require Exporter;

our $VERSION = '0.01';
our @ISA = qw(Exporter);
our @EXPORT = qw(
    get_captcha_id
    check_captcha
);

use warnings;
use strict;

use Yandex::HTTP;
use Yandex::Trace;
use Yandex::HashUtils qw/hash_merge hash_cut/;
use XML::LibXML;

# path to captcha server
my $CAPTCHA_SERVER = "http://api.captcha.yandex.net";
our $TIMEOUT ||= 5;
our $CAPTCHA_TYPE ||= 'std';

=about INFO

    https://wiki.yandex-team.ru/passport/api-captcha
    http://doc.yandex-team.ru/Passport/captcha/concepts/about.xml

=head1 FUNCTIONS

=cut

#======================================================================

=head2 get_captcha_id

    my ($cap_id, $cap_url) = get_captcha_id($type, %opt);

http://doc.yandex-team.ru/Passport/captcha/concepts/api_generate.xml#api_generate

Если указан тип std|lite|rus, используем его. По умолчанию -- $CAPTCHA_TYPE

Опции:

    https - on|any
    checks - 1-9

=cut

sub get_captcha_id
{
    my $type = @_ % 2 ? shift : undef; # some magic
    my (%opt) = @_;
    my $profile = Yandex::Trace::new_profile('captcha:generate');

    $type ||= $CAPTCHA_TYPE;
    die "unknown type '$type'" unless $type =~ /^[en]?(?:std|lite|rus|lat)[ulm]?$|^nbg$/;

    my $call_url = Yandex::HTTP::make_url(
        "$CAPTCHA_SERVER/generate",
        hash_merge {type => $type}, hash_cut(\%opt, qw/https checks/)
    );

    my ($url, $capid);
    my $capid_raw = http_get($call_url, timeout => $TIMEOUT);
    
    my $xres = XML::LibXML->new()->parse_string($capid_raw);
    
    if (my $numbers = $xres->findnodes("//number")) {
        my $number = $numbers->[0];
        $capid = $number->textContent;
        
        # получаем значение атрибута url
        foreach my $a ($number->attributes()) {
            if ($a->name eq 'url') {
                $url = $a->getValue;
            }
        }
    }
    
    return $capid, $url;
}

#======================================================================

=head2 check_captcha

  if (check_captcha($capid, $capcode)) {
      print STDERR "ok\n";
  } else {
      print STDERR "error\n";
  }

=cut

sub check_captcha
{
    my ($capid, $capcode) = @_;
    my $profile = Yandex::Trace::new_profile('captcha:check');

    my $result = '';
    my $cap_res = http_get("$CAPTCHA_SERVER/check?key=$capid&rep=$capcode", timeout => $TIMEOUT);
    
    if ($cap_res) {
        my $xres = XML::LibXML->new()->parse_string($cap_res);
        
        if (my $numbers = $xres->findnodes("//image_check")) {
            $result = $numbers->[0]->textContent;
        }
        
        return $result eq 'ok' ? 1 : 0;
    }
    
    return 0;
}

1;

=head1 AUTHORS

in alphabetical order:

  Maxim Kuzmin C<kuzmin@yandex-team.ru>
  Sergey Mudrik C<msa@yandex-team.ru>
  Sergey Zhuravlev C<zhur@yandex-team.ru>

=head1 COPYRIGHT

Copyright (c) 2004 - 2007 Yandex. All rights reserved.

=cut
