package Yandex::Catalogia;


# $Id$

=head1 NAME

    Yandex::Catalogia

=head1 DESCRIPTION

    Работа с классификатором Catalogia.

    Умеет обращаться в Catalogia в несколько параллельных запросов.

=cut

use strict;
use warnings;
use utf8;
use base qw/Exporter/;
use Encode;
use URI::Escape::XS qw/uri_unescape/;
use Yandex::Log;
use Yandex::HTTP qw( http_parallel_request );
use Data::Dumper;
use JSON;
use Yandex::Trace;

our @EXPORT_OK = qw/banner_classify/;

=head2 $CATALOGIA_URL

    Классификатор Catalogia. Набор URL'ов, где отвечает Catalogia.
=cut

our $CATALOGIA_URL ||= [ 'http://catalogia-mod-rt.yandex.net/cgi-bin/get_categs.pl' ];

=head2 $DEFAULT_TIMEOUT

    Тайм-аут по умолчанию.
=cut

my $DEFAULT_TIMEOUT = 5;

=head2 $CATALOGIA_PARALLEL_REQUESTS

    Кол-во параллеьных запросов.
=cut

our $CATALOGIA_PARALLEL_REQUESTS = 1;

=head2 $CATALOGIA_CHUNK_MAX_SIZE

    Размер максимального чанка.
    Например, если было получено 1000 объектов, то будет выполнено 10 HTTP запросов при размере CATALOGIA_CHUNK_MAX_SIZE = 100.
=cut

our $CATALOGIA_CHUNK_MAX_SIZE = 100;


=head2 LOG_FILE

    Лог файл для ошибок
=cut

our %LOG_FILE;
%LOG_FILE = (
    log_file_name => "Catalogia.log",
    date_suf => "%Y%m%d",
) if !%LOG_FILE;

=head2 %MANDATORY_FIELDS

Обязательные поля которые должны отправляться в каталогию хотя бы пустыми. Если их не будет, то они ответят ошибкой.

=cut

my %MANDATORY_FIELDS = (
    title => 1, 
    body  => 1, 
    phrases => 1,
);

=head2 CATALOGIA_ERRORS

    Типы ошибок (хеш 'код' => 'описание')
=cut

use constant {
	E_TIMEOUT 		=> 1,
	E_ERROR_SIZE	=> 2,
	E_ERROR_HTTP	=> 3,
	E_ERROR_NO_CNT	=> 4,
};

our %CATALOGIA_ERRORS = (
    &E_TIMEOUT   	=> 'TIMEOUT',
    &E_ERROR_SIZE  	=> 'ERROR_SIZE',
    &E_ERROR_HTTP 	=> 'ERROR_HTTP',
    &E_ERROR_NO_CNT	=> 'ERROR_NO_CONTENT',
);

=head2 %CATALOGIA_REL_PHRASES

    Результаты проверки фраз на релевантность
    
    Все константы CRP_ экспортируются тегом :catalogia_rel_phrases

=cut

use constant {
    CRP_OK      => 'CRP_OK',
    CRP_PORNO   => 'CRP_PORNO',
    CRP_BANNED  => 'CRP_BANNED',
    CRP_WIDE    => 'CRP_WIDE',
    CRP_SHORT   => 'CRP_SHORT',
    CRP_DIFF    => 'CRP_DIFF',
    CRP_BHOMO   => 'CRP_BHOMO',
    CRP_BVEND   => 'CRP_BVEND',
    CRP_AGRPH   => 'CRP_AGRPH',
    CRP_WIDETP  => 'CRP_WIDETP',
    CRP_BADWRD  => 'CRP_BADWRD',
    CRP_CELEBR  => 'CRP_CELEBR',
    CRP_BADMIN1 => 'CRP_BADMIN1',
    CRP_BADMIN2 => 'CRP_BADMIN2',
    CRP_ADVQ    => 'CRP_ADVQ',
    CRP_VERYBAD => 'CRP_VERYBAD',
    CRP_NOMINI  => 'CRP_NOMINI',
    CRP_UNKNOWN => 'CRP_UNKNOWN', # псевдокатегория - нераспознанный ответ Каталогии
};

our %CATALOGIA_REL_PHRASES = (
        # релевантна, нет ошибок
        &CRP_OK      => '0',
        # porno -- порно
        &CRP_PORNO   => 'porno',
        # banned -- забанено модерацией Директа
        &CRP_BANNED  => 'banned',
        # wide -- широкая фраза
        &CRP_WIDE    => 'wide',
        # shotword -- короткое одиночное слово
        &CRP_SHORT   => 'shotword',
        # diffphrs -- фраза из несовместимых слов
        &CRP_DIFF    => 'diffphrs',
        # badhomonym -- омонимичное слово с разным значением в баннере и во фразе
        &CRP_BHOMO   => 'badhomonym',
        # badvendor -- если в тексте баннера явно указан вендор, то запрещаем фразы с другими вендорами
        &CRP_BVEND   => 'badvendor',
        # antigraph -- фильтрация по антиграфу
        &CRP_AGRPH   => 'antigraph',
        # widemdltype -- широкие типы товаров
        &CRP_WIDETP  => 'widemdltype',
        # badwordsfilter -- слова, нетипичные для рекламы
        &CRP_BADWRD  => 'badwordsfilter',
        # celebr -- во фразе упоминается имя знаменитости, а в баннере нет
        &CRP_CELEBR  => 'celebr',
        # badminictg, badminictg2 -- категории фразы противоречат категориям баннера
        &CRP_BADMIN1 => 'badminictg',
        &CRP_BADMIN2 => 'badminictg2',
        # advqctg6 -- слишком широкая по смыслу фраза
        &CRP_ADVQ    => 'advqctg6',
        # verybad -- широкая однословная фраза
        &CRP_VERYBAD => 'verybad',
        # некатегоризуемая однословная фраза
        &CRP_NOMINI  => 'nominictg',
);

my %CATALOGIA_TO_CONST = map { $CATALOGIA_REL_PHRASES{$_} => $_ } keys %CATALOGIA_REL_PHRASES;
                    
# особый регэксп для разбора ответа diffphrs
my $DIFFPHRS = $CATALOGIA_REL_PHRASES{&CRP_DIFF};
my $REL_PHRASES_RE = qr!^$DIFFPHRS:([^:]*):(.*)!;

our %EXPORT_TAGS = (
    'catalogia_rel_phrases' => [ keys %CATALOGIA_REL_PHRASES, &CRP_UNKNOWN ],
);

Exporter::export_ok_tags(keys %EXPORT_TAGS);

=head2 __prepare_data_for_requests

    Собирает хеш с двумя ключами requests и sizes.
    В ключе requests находится ссылка на хеш с запросами для http_parallel_request.
    А в ключе sizes находится ссылка на хеш с размерами http-запросов.

    Входные параметры:
        $data - ссылка на массив с хешами (обязательный параметр).
        $options - сслыка на хеш с параметрами запроса (необязательный параметр).

    Пример:
        $data = [ { 'bid' => .., 'title' => .., 'body' =>  .., 'phrases' => [ .. ], 'site_rubrics' => [ .. ] }, { .. } ];
        $options = { 'timeout' => 10, 'use_multi_urls' => 0, 'chunk_size' => 100 };
            timeout - тайм-аут
            chunk_size - размер чанка (столько баннеров будет проверено за один HTTP запрос)
            use_multi_urls - стараться ли каждому запросу обращаться на отдельный URL
            check_phrases - отправлять параметр context<n>, по которому проверяется релевантность фраз

    Возвращает ссылку на хеш.
    {
        'requests' => { 'request_id_1' => {}, 'request_id_2' => .., .., 'request_id_N' => {} },
        'sizes' => { 'request_id_1' => 100, 'request_id_2' => 100, .., 'request_id_N' => 12 }
    }
=cut

sub __prepare_data_for_requests ($;$) {

    my ( $data, $options ) = @_;
    my $max_chunk_size = int( $options->{ 'chunk_size' } || 0 ) || $CATALOGIA_CHUNK_MAX_SIZE;
    my $request_uri    = $options->{ 'use_multi_urls' } ? undef : get_random_url();
    my ( $request_counter, $chunk_counter ) = ( 0, 0 );
    my %requests = ();
    my %request_sizes = ();

    my $create_request = sub {
        my ($params) = @_;

        my $url = $request_uri;
        if ( $options->{ 'use_multi_urls' } ) { $url = get_random_url(); }

        return( { 'url' => $url, 'body' => $params } );
    };

    my @fields = qw/bid phrases pctgs title body lang/;
    push( @fields, 'context' ) if ($options->{check_phrases});
    push( @fields, 'domain' ) if ($options->{send_domain});

    my $request_chunk = [];

    # разбивка данных на фрагменты и создание объектов для http запросов
    foreach my $record ( @{ $data } ) {

        ++$chunk_counter;

        # замена поля "rubrics" на малопонятное "pctgs"
        if ( exists( $record->{ 'site_rubrics' } ) ) {
            $record->{ 'pctgs' } = delete( $record->{ 'site_rubrics' } );
        }

        foreach my $field ( @fields ) {
            my $value;

            # сериализуем массивы
            if ( ( $field eq 'phrases' or $field eq 'pctgs' ) and ( defined( $record->{ $field } ) && ref( $record->{ $field } ) eq 'ARRAY' ) ) {
                $value = join( ',', grep {!ref $_} @{ $record->{ $field } } );
            } elsif ( $field eq 'context' ) {
                $value = 1;
            } else {
                $value = $record->{ $field };
            }
            
            if (defined $value || defined $MANDATORY_FIELDS{$field}) {
                push @$request_chunk, ($field . $chunk_counter, $value);
            }
        }

        # создадим объект для http реквеста
        if ( $chunk_counter >= $max_chunk_size ) {
            push @$request_chunk, ('timeout', $options->{ 'timeout' });
            $requests{ ++$request_counter } =  $create_request->($request_chunk);
            $request_sizes{ $request_counter } = $chunk_counter;

            $request_chunk= [];
            $chunk_counter = 0;
        }
    }

    # возможно, что у нас есть данные не попавшие в запрос (последний чанк меньше $max_chunk_size)
    if ( scalar( @$request_chunk ) ) {
        push @$request_chunk, ('timeout', $options->{ 'timeout' });
        $requests{ ++$request_counter } = &{ $create_request }($request_chunk);
        $request_sizes{ $request_counter } = $chunk_counter;
    }

    return( { 'requests' => \%requests, 'sizes' => \%request_sizes } );
}

=head2 banner_classify( array_ref, hash_ref )

    Классификация фраз и выставление флагов текстам баннера с помощью нового кластеризатора

    Входные параметры:
        $data - ссылка на массив с хешами (обязательный параметр).
        $options - сслыка на хеш с параметрами запроса (необязательный параметр).

    Пример:
        $data = [ { 'title' => .., 'body' =>  .., 'phrases' => [ .. ], 'site_rubrics' => [ .. ] }, { .. } ];
        $options = { 'timeout' => 10, 'requests' => 20, 'use_multi_urls' => 0 };

            timeout - тайм-аут
            retry_factor - множитель тайм-аута для следующего перезапроса (при retry_factor=2 первый timeout 10 с, второй 20 с, третий - 40 с.)
            requests - кол-во максимальных параллельных HTTP запросов
            chunk_size - размер чанка (столько баннеров будет проверено за один HTTP запрос)
            use_multi_urls - стараться ли каждому запросу обращаться на отдельный URL
            tries - кол-во попыток запроса. По умолчанию равно колв-ву URL каталогии. Не может быть меньше 1.
            log - экземпляр Yandex::Log
            check_phrases - отправлять параметр context<n>, по которому проверяется релевантность фраз
            debug_requests - сохранять в log "сырые" запросы
        
    Возвращает ссылку на массив следующего вида:
        [
            {cat => ['категория 1'], flag => ['medicine']},
            {cat => ['категория 2', 'категория 3'], flag => ['tobacco']}
        ]
=cut

sub banner_classify ($;$) {
    my ( $data, $options ) = @_;

    return( [] ) if ( ! scalar( @{ $data } ) );

    $options ||= {};
    $options->{ 'timeout' } ||= $DEFAULT_TIMEOUT;
    $options->{ 'log' }     ||= new Yandex::Log ( %LOG_FILE );
    $options->{ 'debug_requests' } ||= 0;

    my $log            = $options->{ 'log' };
    my $prepared_data  = __prepare_data_for_requests( $data, $options );
    my $requests       = $prepared_data->{ 'requests' };
    my $request_sizes  = $prepared_data->{ 'sizes' };
    
#    warn Dumper({req => $requests});

    my $successful_responses = __execute_requests( $requests, $request_sizes, $options ) || {};
    my @result = ();

    # обработка ответов в порядке переданных данных
    foreach my $request_id ( sort { $a <=> $b } ( keys( %$successful_responses ) ) ) {
        my $processed_response = __process_response(
                $request_id, $successful_responses->{ $request_id }, $request_sizes->{ $request_id }, $options->{ 'log' }
            );

        push( @result, @{ $processed_response } );
    }

    $log->out( { 'data_size' => scalar( @{ $data } ), %{ $options } } ) if ( ! scalar( @result ) );

    return( \@result );
}

=head2 __execute_requests

    Выполняет параллельные запросы.
    Если какие-то запросы не прошли, то может сделать перезапрос.

    Принимает параметры:
        $requests - ссылка на хеш с запросами для http_parallel_request.
        $request_sizes - ссылка на хеш с размерами запрашиваемых фрагментов (ключ request_id, значение - размер фрагмента).
        $options - ссылкна на хеш с параметрами запроса.
            timeout - тайм-аут
            retry_factor - множитель тайм-аута для следующего перезапроса (при retry_factor=2 первый timeout 10 с, второй 20 с, третий - 40 с.)
            requests - кол-во максимальных параллельных HTTP запросов
            log - экземпляр Yandex::Log
            tries -кол-во попыток запроса. По умолчанию равно колв-ву URL каталогии. Не может быть меньше 1.

    Вовзращает ссылку на хеш с HTTP ответами в формате http_parallel_request.
=cut

sub __execute_requests ($$$) {

    my ( $requests, $request_sizes, $options ) = @_;
    my $request_tries = $options->{ 'tries' } || 1;
    my $log = $options->{ 'log' };
    my %processed_responses = ();
    my $timeout = $options->{ 'timeout' } || $DEFAULT_TIMEOUT;
    my $retry_factor = $options->{ 'retry_factor' } || 1;

    while ( $request_tries-- && scalar( keys( %{ $requests } ) ) ) {

        $timeout *= $retry_factor;

        if ( $options->{ 'debug_requests' } ) {
            while (my ($reqno, $req) = each %{ $requests || {} }) {
                $log->out( 'URL>' . $req->{url} . '<URL' ) if ( exists $req->{url} );
                $log->out( 'BODY>' . JSON::to_json($req->{body}) . '<BODY' ) if ( exists $req->{body} );
            }
        }

        # делаем параллельные запросы        
        my $_profile = Yandex::Trace::new_profile('catalogia:categs', obj_num => scalar(keys %$requests));
        
        my $responses = http_parallel_request(
            'POST'    => $requests,
            'max_req' => $options->{ 'requests' } || $CATALOGIA_PARALLEL_REQUESTS,
            'timeout' => $timeout,
            'headers' => {},
            'log'     => $log,
        );

        undef $_profile;

        foreach my $request_id ( keys %{ $responses } ) {
            my $response = $responses->{ $request_id };
            my $bid = $requests->{$request_id} && $requests->{$request_id}->{bid};

            my ($response_error, $error_type) = __check_response_for_errors( 
                        $request_id, $responses->{ $request_id }, $request_sizes->{ $request_id } );

            if ( !$error_type ) {
                # если кончились попытки перезапросов или запрос прошёл успешно, то добавим ответ в %processed_responses
                $response->{error_type} = $error_type;
                $response->{error_msg} = $response_error;
                $processed_responses{ $request_id } = $response;

                delete( $requests->{ $request_id } );
            }
            else {
                warn "response_error => $response_error, error_type => $error_type, tries left: $request_tries";
            }

        }
    }

    return( \%processed_responses );
}

=head2 get_random_url

    Возвращает случайный URL для запроса в Catalogia из массива @{ $Yandex::Catalogia::CATALOGIA_URL }
=cut

sub get_random_url {

    my $random_index = 0;

    if ( scalar( @{ $CATALOGIA_URL } ) > 1 ) {
        $random_index = int( rand( scalar( @{ $CATALOGIA_URL } ) * 10 ) % ( scalar( @{ $CATALOGIA_URL } ) ) );
    }

    return( $CATALOGIA_URL->[ $random_index ] );
}

=head2 __check_response_for_errors

    Внутренний метод для получения текста ошибки для не получившегося запроса.

    Принимает параметры:
        $request_id - ID HTTP запроса.
        $response - ссылкна на хеш с HTTP ответом (формат http_parallel_request).
        $request_size - размер запрашиваемого фрагмента (не более чем C<chunk_size> - см. метод banner_classify).

    Возвращает строку с текстом ошибки и код ошибки из %CATALOGIA_ERRORS. Если ошибки нет, то вернёт undef.
=cut

sub __check_response_for_errors ($$$) {

    my ( $request_id, $response, $request_size ) = @_;
    my $errorText = undef;
    my $errorType = undef;

    if ( $response->{ 'is_success' } ) {
        my @lines = split( /[\r\n]+/, $response->{ 'content' } );

        if ( $lines[0] =~ m/TIMEOUT/ ) {
            $errorText = 'Catalogia timed out';
            $errorType = E_TIMEOUT;
        } elsif ( $request_size != scalar( @lines ) ) {
            $errorText = 'Unexpected output ' .$request_size.' != '. scalar( @lines);
            $errorType = E_ERROR_SIZE;
        }
    } else {
        $errorText = $response->{ 'headers' }->{ 'Status' };
        $errorType = E_ERROR_HTTP;
    }

    return( $errorText, $errorType );
}

=head2 __process_response

    Внутренний метод для обработки HTTP ответа от Catalogia.

    Принимает параметры:
        $request_id - ID HTTP запроса.
        $response - ссылка на хеш с HTTP ответом (формат http_parallel_request).
        $request_size - размер запрашиваемого фрагмента (не более чем C<chunk_size> - см. метод banner_classify).
        $log - экземпляр Yandex::Log

    Возвращает массив с ссылками на хеши.
    
    Если запрос не удалось выполнить, то в хеш будет добавлен ключ 'error' со значением из %CATALOGIA_ERRORS 
    При этом в 'error_msg' вернётся текстовое сообщение об ошибке

    Если запрос выполнен, но категорий и флагов не присвоено, в хеше будет ключ 'is_empty' со значением 1
=cut

sub __process_response ($$$$) {

    my ( $request_id, $response, $request_size, $log ) = @_;
    my $request_uri = $response->{ 'headers' }->{ 'URL' };
    my @result = ();

    my @lines = split( /[\r\n]+/, $response->{ 'content' } || '' );

    my $result_size = 0;
    unless ($response->{error_type}) {
        foreach my $line ( @lines ) {
            $line = decode_utf8( uri_unescape( $line ) );

            my ( $flag, $cat, $cat_ids, $relevant_phrases ) = split( "\t", $line );
                    
            my $result = {
                    'cat'           => [ split( '/', $cat ) ],
                    'flag'          => [ split( '/', $flag ) ],
                    'cat_ids'       => [ split( ',', $cat_ids ) ],
                    'is_relevant'   => undef,
                    'is_empty'      => (($cat eq '') && ($flag eq '')),
                };

            # релевантность фразы - возвращается только если попросили
            if (defined $relevant_phrases) {
                # одна фраза - один ответ, фраз может быть несколько, разделитель ','
                foreach my $phrase_responses (split ',', $relevant_phrases) {
                    # в одном ответе может быть несколько ошибок, разделитель '+'
                    my $decoded_responses;

                    foreach my $response (split '\+', $phrase_responses) {
                        if ( exists $CATALOGIA_TO_CONST{$response} ) {
                            $decoded_responses->{ $CATALOGIA_TO_CONST{$response} } = undef;
                        } elsif ( $response =~ $REL_PHRASES_RE ) {
                            # особый случай: возвращает строку вида "diffphrs:москва:спб", соответственно - слово из баннера и из фразы
                            $decoded_responses->{ &CRP_DIFF } = {
                                    w1  =>  $1,
                                    w2  =>  $2,
                                };
                        } else {
                            $decoded_responses->{ &CRP_UNKNOWN } = undef;
                            $log->out({ tag => '__process_response', catalogia_response => $line, bad_part => $response });
                        }
                    }

                    push @{ $result->{relevant_phrases} }, $decoded_responses;
                }
            }

            push( @result, $result );
            $result_size++;
        }
    }
    
    if ($result_size != $request_size ) {
        @result = ();

        for ( 1 .. $request_size ) {
            push( @result, {
                'cat'           => [],
                'flag'          => [],
                'cat_ids'       => [],
                'is_relevant'   => 0,
                'error'         => $response->{error_type} || E_ERROR_NO_CNT,
                'error_msg'     => $response->{error_msg} || '',
            } );
        }
    }

    return( \@result );
}

1;

=head1 AUTHOR

    Alexey Ziyangirov <metallic@yandex-team.ru>
    Denis  Govorkov   <oxid@yandex-team.ru>
    Dmitriy Chernenko <trojn@yandex-team.ru>
    Andrei Lukovenko  <aluck@yandex-team.ru>

    Copyright (c) 2014 Яндекс. Все права защищены.

=cut
