package Yandex::Clickhouse;

use strict;
use warnings;
use feature 'state';

use AnyEvent;
use AnyEvent::HTTP qw(http_post);
use MIME::Base64 qw(encode_base64);
use Yandex::Clickhouse::Result;
use Yandex::Trace;
use Yandex::XsUtils qw/clh_quote clh_quote_tsv_row/;

our $VERSION = '0.18';
our $DEFAULT_HOST = 'localhost';
our $DEFAULT_PORT = 8123;
our $DEFAULT_USER = undef;
our $DEFAULT_PASSWORD = undef;
our $DEFAULT_PERSISTENT = undef;

our $QUERIES_LOG;

=head2 new(%options)

    Создание экземпляра Yandex::Clickhouse

    Параметры:
        host => имя хоста (по-умолчанию localhost)
        port => порт (по-умолчанию 8123)
        timeout => таймаут (по-умолчанию 600 секунд)
        user => имя пользователя (для аутентификации)
        password => пароль пользователя (для аутентификации)
        persistent => 1 (использовать постоянные соединения с сервером)
        use_https => 0 (подключаться по https вместо http)
        settings => { key => value, ... } настройки ( https://clickhouse.yandex-team.ru/#Настройки )
=cut

sub new {
    my $this = shift;
    my $class = ref($this) || $this;
    my $self = {@_};
    $self->{host} //= $DEFAULT_HOST;
    $self->{port} //= $DEFAULT_PORT;
    $self->{use_https} //= 0;
    $self->{timeout} //= 600;
    $self->{user} //= $DEFAULT_USER;
    $self->{password} //= $DEFAULT_PASSWORD;
    $self->{persistent} //= $DEFAULT_PERSISTENT;
    $self->{settings} //= {};
    bless $self, $class;
    return $self;
}

=head2 get_one_field

    my $max_value = $clh->get_one_field(['select max(some_value)', from => $table]);
    
    query must be in TabSeparated format (which is default)

=cut

sub get_one_field
{
    my $self = shift;
    my $query = shift;
    my $res = $self->query($query);
    return $res->tsv->[0][0];
}

=head2 insert

    $clh->insert($table_name, \@values, %options)
    $clh->insert("table", [ [ val00, val01 ], [ val10, val11], ... ], names => 1);

    Именованные параметры:
     names => 1 - первая строка в \@values содержит имена столбцов. Вставка в формате TabSeparatedWithNames

=cut

sub insert
{
    my ($self, $table, $values, %opt) = @_;
    my ($query_start);
    if ($opt{names}) {
        $query_start = "insert into $table (".join(', ', @{$values->[0]}).") FORMAT TabSeparatedWithNames";
    } else {
        $query_start = "insert into $table FORMAT TabSeparated";
    }
    my $query = $query_start."\n".(join '', map { clh_quote_tsv_row($_) } @$values);
    $self->query($query);
}

=head2 query_async($query, $cb->($result, $error))

    Выполняет асинхронный запрос к базе данных
    По завершению вызывает $cb с параметрами:
    $result: Yandex::Clickhouse::Result или undef
    $error: текст с ошибкой (если была ошибка)

=cut

sub query_async {
    my $self = shift;
    my $cb = pop;
    my $query = _encode_query([@_]);
    if ($self->{_query_format}) {
        $query .= " FORMAT $self->{_query_format}";
    }
    _log_query($query);
    utf8::encode($query) if utf8::is_utf8($query);
    my $proto = $self->{use_https} ? 'https' : 'http';
    my $url = "$proto://$self->{host}:$self->{port}/";
    my @params;
    while (my ($param, $value) = each %{$self->{settings}}) {
        push @params, "$param=$value";
    }
    $url .= '?' . join('&', @params) if @params;

    my %options = (
        headers => {
            "User-Agent" => "Yandex::Clickhouse v$VERSION",
        },
        timeout => $self->{timeout},
        persistent => $self->{persistent},
    );
    if (defined $self->{user}) {
        $self->{password} //= '';
        $options{headers}->{"Authorization"} = "Basic " . encode_base64("$self->{user}:$self->{password}", '');
    }
    if ($self->{_output_file}) {
        open $self->{_FH}, '>', $self->{_output_file} or return $cb->(undef, "$self->{_output_file}: $!");
        $options{on_body} = sub { print { $self->{_FH} } $_[0]; }
    }
    my $profile = Yandex::Trace::new_profile('clickhouse:query');
    http_post $url, $query, %options, sub {
        undef $profile;
        my ($content, $headers) = @_;
        $content //= '';
        unless ($headers->{Status} =~ /^2/) {
            $cb->(undef, "Clickhouse: $headers->{Status} $headers->{Reason}\n$content");
            return;
        }
        if ($self->{_output_file}) {
            close $self->{_FH};
            return $cb->($self->{_output_file}, undef);
        }
        if ($headers->{Status} eq '500') {
            $cb->(undef, "Clickhouse: $content");
            return;
        }
        $cb->(Yandex::Clickhouse::Result->new($content), undef);
        return;
    };
    return;
}

=head2 query($query)

    Выполняет синхронный запрос к базе данных
    В случае успеха возвращает Yandex::Clickhouse::Result

=cut

sub query {
    my $cv = AnyEvent->condvar;
    push @_, sub {
        if (defined $_[1]) {
            $cv->croak($_[1]);
            return;
        }
        $cv->send($_[0]);
    };
    &query_async;
    return $cv->wait;
}

=head2 format($fragments)

    Выполняет формирование запроса в ClickHouse
    Например:

    format(["SELECT * FROM table WHERE ", {a__eq => "hello", b__ne => "world"}])

    Вернёт отформатированные запрос:

    "SELECT * FROM table WHERE a = 'hello' AND b <> 'world'"

=cut

sub format {
    shift if @_ > 1; # skip $self
    my ($fragments) = @_;
    return _encode_query($fragments);
}

=head2 quote_tsv($value)

    Квотирует строку в формат TabSeparated

=cut

sub quote_tsv {
    shift if @_ > 1; # skip $self
    if (ref $_[0] eq 'ARRAY') {
        return clh_quote_tsv_row($_[0]);
    } else {
        return clh_quote($_[0]);
    }
}

=head2 quote_str($text)

    Квотирует строку в литерал SQL строки

=cut

sub quote_str {
    shift if @_ > 1; # skip $self
    my ($text) = @_;
    return "'" . clh_quote($text) . "'";
}

=head2 output_file

    $clickhouse->output_file("my_file.txt");
    my $file = $clickhouse->output_file;

    Задает файл, в который будет писаться результат запроса

=cut

sub output_file
{
    my $self = shift;
    if (@_) {
        return $self->{_output_file} = shift;
    }
    return $self->{_output_file};
}

=head2 settings

Установить/прочитать параметр, который будет передаваться в url

    $clickhouse->url_param(profile => 'web');
    my $p = $clickhouse->url_param('profile');

=cut

sub settings
{
    my $self = shift;
    my $param = shift;
    if (@_) {
        return $self->{settings}->{$param} = shift;
    }
    return $self->{settings}->{$param};
}

=head2 query_format

=cut

sub query_format
{
    my $self = shift;
    if (@_) {
        return $self->{_query_format} = shift;
    }
    return $self->{_query_format};
}

# --- private ---

=head2 _quote

Квотирует единичное значение

Если значение -- массив, сериализует

=cut
    sub _quote {
        my ($text) = shift;
        if ( ref $text eq 'ARRAY' ){
            return '['.join(',', map { clh_quote($_) } @$text).']';
        } else {
            return clh_quote($text);
        }
    }

    sub _quote_str {
        return "'" . clh_quote($_[0]) . "'";
    }

{
    my %modifiers = (
        'sql' => { sql => 1 },
        'int' => { convert => 'int' },
        'date' => { convert => 'date' },
        'datetime' => { convert => 'datetime' },
        'lt' => { op => '<' },
        'gt' => { op => '>' },
        'le' => { op => '<=' },
        'ge' => { op => '>=' },
        'ne' => { op => '<>' },
        'eq' => { op => '=' },
        'rlike' => { op => 'match', func => 1 },
        'in' => { op => 'in'},
        'not_in' => { op => 'not in'},
        'has' => { op => 'has' },
    );

    sub _parse_field {
        my ($name) = @_;
        my %field;
        ($field{name}, my @tokens) = split '__', $name;
        for my $token (@tokens) {
            my $props = $modifiers{$token} || die "ClickHouse: unknown field modifier $token in '$name'";
            if (exists $field{sql} && exists $props->{sql}) {
                die "ClickHouse: multiple sql modifiers in '$name'";
            } elsif (exists $field{convert} && exists $props->{convert}) {
                die "ClickHouse: multiple convert modifiers in '$name'";
            } elsif (exists $field{op} && exists $props->{op}) {
                die "ClickHouse: multiple operations in '$name'";
            }
            @field{keys %$props} = values %$props;
        }
        $field{op} ||= '=';
        return \%field;
    }
}

sub _convert_value {
    my ($convert, $value, $skip_quote) = @_;

    if ($convert eq 'int') {
        die "ClickHouse: value is not an integer: $value" unless $value =~ /^-?[0-9]+$/;
    } elsif ($convert eq 'date') {
        if (!$skip_quote) {
            die "ClickHouse: value is not a date: $value" unless $value =~ /^[0-9]{4}-[0-9]{2}-[0-9]{2}$/;
            $value = _quote_str($value);
        }
        $value = "toDate($value)";
    } elsif ($convert eq 'datetime') {
        if (!$skip_quote) {
            die "ClickHouse: value is not a date/time: $value" unless $value =~ /^[0-9]{4}-[0-9]{2}-[0-9]{2}[ T][0-9]{2}:[0-9]{2}:[0-9]{2}$/;
            $value = _quote_str($value);
        }
        $value = "toDateTime($value)";
    } else {
        die "ClickHouse: unhandled conversion $convert";
    }

    return $value;                
}

sub _encode_query {
    my ($frag, $joiner) = @_;
    if (ref($frag) eq "") {
        return $frag;
    }
    if (ref($frag) eq "ARRAY") {
        # nested fragments
        return join($joiner // " ", map {_encode_query($_, $joiner)} @$frag);
    }
    if (ref($frag) eq "HASH") {
        # operators (where clause)
        my @conditions;
        while (my ($name, $value) = each(%$frag)) {
            my $field = _parse_field($name);
            if (ref($value) eq "CODE") {
                $value = $value->();
            }
            if ($field->{name} eq "NOT") {
                push(@conditions, "NOT (" . _encode_query($value, $joiner) . ")");
                next;
            }
            if ($field->{name} eq "AND") {
                push(@conditions, "(" . _encode_query($value, " AND ") . ")");
                next;
            }
            if ($field->{name} eq "OR") {
                push(@conditions, "(" . _encode_query($value, " OR ") . ")");
                next;
            }
            if ($field->{op} eq "has") {
                my $convert = delete $field->{convert};
                my @vals = map { defined $convert ? _convert_value($convert, $_, $field->{sql}) : _quote_str($_) } @{ ref $value eq "ARRAY" ? $value : [$value]};
                push(@conditions, sprintf "(%s)", join " OR ", map { sprintf "has(%s, %s)", $field->{name}, $_ } @vals);
                next;
            }
            if (ref($value) eq "ARRAY") {
                my $convert = delete $field->{convert};
                my @vals = map { defined $convert ? _convert_value($convert, $_, $field->{sql}) : _quote_str($_) } @$value;
                $value = "(" . join(', ', @vals) .")";
                $field->{sql} = 1;
            }
            die "ClickHouse: unsupported value in '$name' => '$value'" unless ref($value) eq "";
            if (exists $field->{convert}) {
                $value = _convert_value($field->{convert}, $value, $field->{sql});
                $field->{sql} = 1;
            }
            $value = _quote_str($value) unless $field->{sql};
            if ($field->{func}) {
                push(@conditions, "$field->{op}($field->{name}, $value)");
            } else {
                push(@conditions, "$field->{name} $field->{op} $value");
            }
        }
        return join($joiner // " AND ", @conditions);
    }
    die "ClickHouse: unexpected query fragment '$frag'";
}

sub _log_query
{
    return unless $QUERIES_LOG;

    require Yandex::Log;
    state $log = Yandex::Log->new(
        log_file_name => $QUERIES_LOG,
        date_suf => "%Y%m%d",
    );
    $log->out(@_);
}

1;
