package Yandex::Clickhouse::Result;

use strict;
use warnings;

use JSON ();

=head2 new($data)

    Создаёт результаты на основе полученные данных

=cut

sub new {
    my $this = shift;
    my $class = ref($this) || $this;
    my $self = {data => shift};
    bless $self, $class;
    return $self;
}

=head2 json()

    Возвращает результат как декодированный JSON

    См. http://clickhouse.yandex-team.ru/#JSON

    Внимание: сами данные находятся по ключа data.

=cut

sub json {
    my ($self) = @_;
    return JSON->new->utf8(1)->decode($self->{data});
}

=head2 tsv(%options)

    Возвращает результат как декодированный TabSeparated

    См. http://clickhouse.yandex-team.ru/#TabSeparated

    Дополнительные опции:

    names => 1 (парсит формат TabSeparatedWithNames)
    columns => список колонок (возвращает именованные hashref вместо arrayref)

=cut

sub tsv {
    my ($self, %options) = @_;
    my @rows = split(/\n/, $self->{data});
    if (exists $options{names} && $options{names}) {
        die "ClickHouse: result has no rows (column names required)" unless @rows;
        my $names = _tsv_unquote_row(shift @rows);
        return [map {_tsv_unquote_named_row($_, $names)} @rows];
    } elsif (exists $options{columns}) {
        my $names = $options{columns};
        die "ClickHouse: columns must be a reference to array" unless ref $names eq "ARRAY";
        return [map {_tsv_unquote_named_row($_, $names)} @rows];
    } else {
        return [map {_tsv_unquote_row($_)} @rows];
    }
}

# --- private ---

sub _tsv_unquote_row {
    my ($text) = @_;
    return [map {_tsv_unquote_value($_)} split(/\t/, $text, -1)];
}

sub _tsv_unquote_named_row {
    my ($text, $names) = @_;
    my %row;
    @row{@$names} = map {_tsv_unquote_value($_)} split(/\t/, $text, -1);
    return \%row;
}

{
    # См. http://clickhouse.yandex-team.ru/#TabSeparated
    my %unquote = (
        "\\0" => "\x00",
        "\\b" => "\b",
        "\\f" => "\f",
        "\\r" => "\r",
        "\\n" => "\n",
        "\\t" => "\t",
        "\\'" => "'",
        "\\\\" => "\\",
    );

    sub _tsv_unquote_value {
        my ($text) = @_;
        # FIXME: does not handle \uXXXX unicode escapes yet!
        # NOTE: on the other hand clickhouse never produces those
        $text =~ s{\\(.)}{$unquote{$&}//$1}geo;
        return $text;
    }
}

1;
