#!/usr/bin/perl -w
use strict;
use warnings;
use utf8;
use Test::More;
use Tie::IxHash;
use Yandex::Clickhouse;

my $c = Yandex::Clickhouse->new(host => 'testhost');
is($c->{host}, 'testhost');
is($c->{port}, 8123);
is($c->{timeout}, 600);

is(Yandex::Clickhouse::quote_tsv("hello\tworld"), "hello\\tworld");
is(Yandex::Clickhouse->quote_tsv("hello\tworld"), "hello\\tworld");
is($c->quote_tsv("hello\tworld"), "hello\\tworld");

is($c->quote_tsv("api\nunits"), "api\\nunits", "quote newline");
is($c->quote_tsv("api\runits"), "api\\runits", "quote newline");

is(Yandex::Clickhouse::quote_tsv(["hello\tworld", "42"]), "hello\\tworld\t42\n");
is(Yandex::Clickhouse->quote_tsv(["hello\tworld", "42"]), "hello\\tworld\t42\n");
is($c->quote_tsv(["hello\tworld", "42"]), "hello\\tworld\t42\n");

is(Yandex::Clickhouse::quote_str("hello\tworld"), "'hello\\tworld'");
is(Yandex::Clickhouse->quote_str("hello\tworld"), "'hello\\tworld'");
is($c->quote_str("hello\tworld"), "'hello\\tworld'");

is(Yandex::Clickhouse::format({"hello" => "world"}), "hello = 'world'");
is(Yandex::Clickhouse->format({"hello" => "world"}), "hello = 'world'");
is($c->format({"hello" => "world"}), "hello = 'world'");

is(Yandex::Clickhouse::format({"column" => "hello\tworld"}), "column = 'hello\\tworld'");
is(Yandex::Clickhouse::format({"column__eq" => "hello\tworld"}), "column = 'hello\\tworld'");
is(Yandex::Clickhouse::format({"column__lt" => "hello\tworld"}), "column < 'hello\\tworld'");
is(Yandex::Clickhouse::format({"column__gt" => "hello\tworld"}), "column > 'hello\\tworld'");
is(Yandex::Clickhouse::format({"column__le" => "hello\tworld"}), "column <= 'hello\\tworld'");
is(Yandex::Clickhouse::format({"column__ge" => "hello\tworld"}), "column >= 'hello\\tworld'");
is(Yandex::Clickhouse::format({"column__ne" => "hello\tworld"}), "column <> 'hello\\tworld'");
is(Yandex::Clickhouse::format({"column__rlike" => "^hello\tworld"}), "match(column, '^hello\\tworld')");
is(Yandex::Clickhouse::format({"column__in__int" => [qw/1 2 3/]}), "column in (1, 2, 3)");
is(Yandex::Clickhouse::format({"column__in" => [qw/one two three/]}), "column in ('one', 'two', 'three')");
is(Yandex::Clickhouse::format({"column__not_in__int" => [qw/1 2 3/]}), "column not in (1, 2, 3)");
is(Yandex::Clickhouse::format({"column__not_in" => [qw/one two three/]}), "column not in ('one', 'two', 'three')");

is(Yandex::Clickhouse::format({"column__has__int" => [qw/1 2 3/]}), "(has(column, 1) OR has(column, 2) OR has(column, 3))");
is(Yandex::Clickhouse::format({"column__has" => [qw/one two three/]}), "(has(column, 'one') OR has(column, 'two') OR has(column, 'three'))");

is(Yandex::Clickhouse::format({"column__int" => 42}), "column = 42");
is(Yandex::Clickhouse::format({"column__sql" => "someFunction()"}), "column = someFunction()");
is(Yandex::Clickhouse::format({"column__date" => '2014-06-09'}), "column = toDate('2014-06-09')");
is(Yandex::Clickhouse::format({"column__datetime" => "2014-06-09 15:51:55"}), "column = toDateTime('2014-06-09 15:51:55')");

is(Yandex::Clickhouse::format({"NOT" => "hello"}), "NOT (hello)");
is(Yandex::Clickhouse::format({"AND" => ["hello", "world"]}), "(hello AND world)");
is(Yandex::Clickhouse::format({"OR" => ["hello", "world"]}), "(hello OR world)");

sub ordered_hash {
    tie my %hash, 'Tie::IxHash', @_;
    return \%hash;
}

# need to preserve WHERE order in a string
my $where = ordered_hash(
    "a__lt__int" => 42,
    "a__le__int" => 43,
    "b__gt__int" => 51,
    "b__ge__int" => 52,
    "c__eq" => "hello world",
    "c__ne" => "test",
    "d__int" => 115,
    "OR" => ["true", "false", ordered_hash("e__int" => 1, "f__int" => 2)],
    "NOT" => ordered_hash("g__int" => 3, "h__int" => 4));
is(
    Yandex::Clickhouse::format([
        "SELECT",
        ["name,", "value"],
        "FROM table",
        WHERE => $where]),
    "SELECT name, value FROM table WHERE a < 42 AND a <= 43 AND b > 51 AND b >= 52".
    " AND c = 'hello world' AND c <> 'test' AND d = 115 AND".
    " (true OR false OR e = 1 OR f = 2) AND".
    " NOT (g = 3 AND h = 4)");

eval { Yandex::Clickhouse::format({"column__int" => '୫'}) };
like($@, qr/value is not an integer/);
eval { Yandex::Clickhouse::format({"column__date" => '୫୫୫୫-୫୫-୫୫'}) };
like($@, qr/value is not a date/);
eval { Yandex::Clickhouse::format({"column__datetime" => '୫୫୫୫-୫୫-୫୫ ୫୫:୫୫:୫୫'}) };
like($@, qr/value is not a date\/time/);

done_testing();
