#!/usr/bin/perl -w
use strict;
use warnings;
use Test::More;
use Yandex::Clickhouse;
BEGIN {
    plan skip_all => 'AnyEvent::HTTPD is not available' unless eval 'use AnyEvent::HTTPD; 1';;
    plan tests => 20;
    AnyEvent::detect();
}

{
    my @requests;
    my @responses;
    my $httpd = AnyEvent::HTTPD->new;
    $httpd->reg_cb(
        '' => sub {
            my ($httpd, $req) = @_;
            push(@requests, $req);
            my $response = shift @responses;
            if (defined $response) {
                $req->respond([200, 'OK', {}, $response]);
            } else {
                $req->respond([500, 'Internal Server Error', {}, 'Code 42: You Win']);
            }
            $httpd->stop_request;
        },
    );

    sub client(%) {
        die "All requests should have been consumed" if @requests;
        die "All responses should have been returned" if @responses;
        my %options = (host => $httpd->host, port => $httpd->port, @_);
        my $response = delete $options{response};
        if (defined $response) {
            push(@responses, $response);
        }
        my $responses = delete $options{responses};
        if (defined $responses) {
            push(@responses, @$responses);
        }
        return Yandex::Clickhouse->new(%options);
    }

    sub requests() {
        my @copy = @requests;
        @requests = ();
        return \@copy;
    }

    sub responses() {
        my @copy = @responses;
        @responses = ();
        return \@copy;
    }

    sub next_request() {
        return shift @requests;
    }

    sub stop_server() {
        $httpd->stop;
    }
}

{
    my $c = client(response => "hello\tworld\nanother\tline\n");
    my $result = $c->query("SELECT * FROM TestTable");
    my $request = next_request;
    is(ref($result), "Yandex::Clickhouse::Result");
    is($result->{data}, "hello\tworld\nanother\tline\n");
    is_deeply($result->tsv, [["hello", "world"], ["another", "line"]]);
    is($request->method, 'POST');
    is($request->url->path_query, '/?max_query_size=16407');
    is($request->content, 'SELECT * FROM TestTable');
}

{
    my $c = client(response => "{\"data\": [{\"column1\": \"hello\", \"column2\": 42}, {\"column1\": \"world\", \"column2\": 51}]}");
    my $result = $c->query("SELECT * FROM TestTable FORMAT JSON");
    my $request = next_request;
    is(ref($result), "Yandex::Clickhouse::Result");
    is($result->{data}, "{\"data\": [{\"column1\": \"hello\", \"column2\": 42}, {\"column1\": \"world\", \"column2\": 51}]}");
    is_deeply($result->json, {data => [{column1 => "hello", column2 => 42}, {column1 => "world", column2 => 51}]});
    is($request->method, 'POST');
    is($request->url->path_query, '/?max_query_size=16419');
    is($request->content, 'SELECT * FROM TestTable FORMAT JSON');
}

{
    my $c = client();
    my $result = eval { $c->query("SELECT 1") };
    my $request = next_request;
    is($result, undef);
    like($@, qr/^Clickhouse: Code 42: You Win/);
    is($request->content, 'SELECT 1');
}

{
    my $c = client(response => "1\n");
    $c->query("SELECT 1");
    my $request = next_request;
    like($request->headers->{"user-agent"}, qr/^Yandex::Clickhouse v/);
    is($request->headers->{"authorization"}, undef);
}

{
    my $c = client(user => "hello", password => "world", response => "1\n");
    $c->query("SELECT 1");
    my $request = next_request;
    is($request->headers->{"authorization"}, "Basic aGVsbG86d29ybGQ=");
}

is_deeply(requests, [], "All requests should have been consumed");
is_deeply(responses, [], "All responses should have been returned");
stop_server;

done_testing();
