package Cocaine::App;

use 5.010;
use strict;
use warnings;
use utf8;

use Mouse;

use Carp;
use Attribute::Handlers;
use Cocaine;

use List::MoreUtils qw/all/;
use File::Slurp;
use YAML;


use Cocaine::App::Method;
use Cocaine::App::Method::JSON;
use Cocaine::App::Method::HTTP;

# Throwable is not compatible with Mouse
use Cocaine::App::Exception;
use Exception::Class ( CocaineAppException => {isa =>'Cocaine::App::Exception'} );


has cocaine => ( is => 'ro', isa => 'Cocaine' );
has worker => ( is => 'ro', isa => 'Cocaine::Worker' );


=head1 DESCRIPTION

Base class.

Provides framework for creating and running cocaine applications.

=head1 SYNOPSIS

    # create inherited class
    package MyCocaineApp;
    BEGIN { extends 'Cocaine::App; }

    # describe methods
    sub my_method
        :CocaineMethod
    {
        return { some => 'data' };
    }

    1;
=cut


=head1 ATTRIBUTE HANDLERS
=head2 CocaineRawMethod

    sub my_method_sub :CocaineRawMethod(my_method)
    {
        my ($self, $args) = @_;
        return { some => 'data' };
    }

Mark sub as cocaine method, which becomes available from outside.

=cut

sub CocaineRawMethod :ATTR(CODE,BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    my $method = *{$symbol}{NAME};
    my $cocaine_method = $data ? (ref $data ? $data->[0] : $data) : $method;

    _add_cocaine_method($cocaine_method => $method,
        class => 'Cocaine::App::Method',
    );

    return;
}


=head2 CocaineMethod

    =cocaine-method my_method
    title: my cool method
    description: provides some nice results
    params: {}
    result:
      type: string
    =cut

    sub my_method_sub :CocaineMethod(my_method)
    {
        my ($self, $args) = @_;
        return { some => 'data' };
    }

Mark sub as json-based cocaine method, which becomes available from outside.
Formatted documentation (json-schema-like) can be provided in =cocaine-method pod section.

=cut

sub CocaineMethod :ATTR(CODE,BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    my $method = *{$symbol}{NAME};
    my $cocaine_method = $data ? (ref $data ? $data->[0] : $data) : $method;

    _add_cocaine_method($cocaine_method => $method,
        class => 'Cocaine::App::Method::JSON',
        schema => _get_format_desc($cocaine_method, $filename),
    );

    return;
}

=head2 FitToSchema
    
    =cocaine-method my_method
    result:
      type: array
      items:
        type: number
        multipleOf: 0.01
    =cut

    sub my_method_sub
        :CocaineMethod(my_method)
        :FitToSchema
    {
        my ($self, $args) = @_;
        return [7, 8.66666667, "10"];
    }

Adjust method result according to schema

=cut

sub FitToSchema :ATTR(CODE,BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    my $method_name = *{$symbol}{NAME};

    my $method = __PACKAGE__->_get_cocaine_method_by_sub($method_name);
    croak "Method $method_name does not support response schema"  if !$method->can('use_response_schema');

    $method->use_response_schema();

    return;
}


=head2 CocaineHttpMethod

    sub my_method_sub :CocaineHttpMethod(my_method)
    {
        my ($self, $env) = @_;
        
        state $app = Plack::Util::load_app('app.psgi');
        return $app->($env);
    }

Mark sub as http method.

Method receives plack environment and should return plack response.

=cut

sub CocaineHttpMethod :ATTR(CODE,BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    my $method = *{$symbol}{NAME};
    my $cocaine_method = $data ? (ref $data ? $data->[0] : $data) : $method;

    _add_cocaine_method($cocaine_method => $method,
        class => 'Cocaine::App::Method::HTTP',
    );

    return;
}


=head2 AliveCheck($check_id)

    sub my_check
        :AliveCheck(database)
    {
        my ($self) = @_;
        return _is_database_available();
    }

Mark sub to be evaluated during call for 'alive' cocaine method.

=cut

sub AliveCheck :ATTR(CODE,BEGIN) {
    my ($package, $symbol, $referent, $attr, $data) = @_;
    my $method = *{$symbol}{NAME};
    my $check = $data ? (ref $data ? $data->[0] : $data) : $method;
    _add_alive_check($check => $method);
    return;
}

{
my %alive_check;

sub _add_alive_check {
    my ($check, $method) = @_;
    $alive_check{$check} = $method;
    return;
}

sub _get_alive_checks {
    return \%alive_check;
}
}



=head1 SERVICE METHODS

Used from worker script

=head2 get_methods

Returns list of registered methods

=cut

{
my %method;
my %method_by_sub;

sub get_methods {
    return [ sort keys %method ];
}

sub _add_cocaine_method {
    my ($name, $sub, %opt) = @_;

    my $class = delete $opt{class};
#    require ($class =~ s/::/\//gxr).".pm";
    my $method = "$class"->new(method_sub => $sub, %opt);

    $method{$name} = $method;
    $method_by_sub{$sub} = $method;

    return;
}

sub _get_cocaine_method {
    my ($self, $method) = @_;
    return $method{$method};
}

sub _get_cocaine_method_by_sub {
    my ($self, $sub) = @_;
    return $method_by_sub{$sub};
}
}


=head2 invoke

Executes registered method and returns prepared answer

=cut

sub invoke {
    my ($self, $cocaine_method, $raw_request) = @_;

    my $method = $self->_get_cocaine_method($cocaine_method);
    my $method_sub = $method->method_sub();

    my @responses;
    eval {
        my $request = $method->decode_request($raw_request);
        my @results = $self->$method_sub($request);
        @responses = $method->encode_response(@results);
    };
    if ( my $e = Exception::Class->caught() ) {
        @responses = $method->get_error_response($e);
    }

    return @responses;
}


sub _get_format_desc
{
    my ($method, $file) = @_;
    state $cache = {};

    if ( !$cache->{$file} ) {
        # or use some Pod::* parser here
        my $file_cache = $cache->{$file} = {};
        my $data = read_file $file, binmode => ':utf8';
        my @doc_sections = $data =~ / (?: ^ | (?<= [\r\n]) ) =cocaine-method \s+ (.+?) (?= [\r\n]+ =\w ) /gxms;
        for my $doc_section (@doc_sections) {
            my ($name, $doc) = $doc_section =~ /^ (\w+) \s+ (.+) $/xms;
            my $yaml = "---\n" . $doc . "\n";
            $file_cache->{$name} = eval { YAML::Load $yaml }
                or do { carp "Bad method description:\n$yaml\n$@" };
        }
    }

    return $cache->{$file}->{$method};
}



=head2 declare_psgi_method

    __PACKAGE__->declare_psgi_method( http => '/path/to/app.psgi' );

Helper sub to load prepared psgi-file as cocaine method.

=cut

sub declare_psgi_method {
    my ($class, $method_name, $psgi_file) = @_;

    # avoid global plack dependency
    require Plack::Util;

    my $app = Plack::Util::load_psgi($psgi_file);
    my $method = sub {
        my ($self, $env) = @_;
        return $app->($env);
    };
    $class->meta->add_method($method_name => $method);
    _add_cocaine_method($method_name => $method_name,
        class => 'Cocaine::App::Method::HTTP',
    );

    return;
}



=head1 COCAINE METHODS



=head2 ping

Check if app properly starts

=cocaine-method ping

title: ping
description: check if app properly starts

params: {}

result:
  type: string

=cut

sub ping :CocaineMethod {
    my ($self, $param) = @_;
    return 'pong';
}


=head2 methods

Provides list of supported methods.

=cocaine-method methods

title: methods
description: list of supported methods

params: {}

result:
  type: array
  items:
    type: string

=cut

sub methods :CocaineMethod {
    my ($self, $param) = @_;
    return $self->get_methods();
}



=head2 alive

Evaluate registered alive checks, and return their results

=cocaine-method alive

title: alive check
description: evaluate registered alive checks, and return their results

params: {}

result:
  type: object
  properties:
    result:
      type: string
    subchecks:
      type: object

=cut


sub alive :CocaineMethod {
    my ($self, $param) = @_;
    my $checks = _get_alive_checks();

    my %subcheck;
    while ( my ($check, $method) = each %$checks ) {
        $subcheck{$check} = eval { $self->$method() };
    }

    my $result = all {$_} values %subcheck;
    return {
        status => ($result ? 'ok' : 'failed'),
        subchecks => \%subcheck,
    };
}


=head2 help

Provides method call description from =cocaine-method pod section.

=cocaine-method help

title: help
description: basic description for supported methods

params:
  type: object
  properties:
    method:
      type: string
  required: [ method ]

result:
  title: schema
  type: object
=cut

sub method_help :CocaineMethod(help) {
    my ($self, $param) = @_;
    my $method_name = $param->{method};
    my $method = $self->_get_cocaine_method($method_name);

    CocaineAppException->throw(code => 'MethodIsUnknown', error => "Unknown method <$method_name>")  if !$method;
    CocaineAppException->throw(code => 'MethodIsUnupported', error => "Method <$method_name> doesn't provide help")  if !$method->can("schema");
    
    return $method->schema;
}







__PACKAGE__->meta->make_immutable();

1;
