package Cocaine::App::Method::HTTP;

use 5.010;
use strict;
use warnings;

use Mouse;
extends 'Cocaine::App::Method';

use Carp;
use Encode;

use Log::Any '$log';
use Data::MessagePack;
use URI::Escape;
use Scalar::Util qw/reftype/;
use YAML;


override decode_request => sub {
    my ($self, $packed_request) = @_;

    my $request = eval { Data::MessagePack->unpack($packed_request) }
    or do {
        $log->info("Request: invalid")  if $log->is_info();
        CocaineAppException->throw( code => 'InvalidRequest', message => "invalid messagepack" );
    };

    if ( $log->is_info() ) {
        my ($http_method, $url, $http_version, $headers, $body) = @$request;
        $log->info(qq{Request: "$http_method $url HTTP/$http_version"} . ($body ? q{ }.length $body : q{}));
    }

    my $env = $self->_build_plack_env($request);

    return $env;
};


override encode_response => sub {
    my ($self, $response) = @_;

    my $plack_response = $self->_build_plack_response($response);
    my ($status, $p_headers, $p_body) = @$plack_response;

    # convert plain array (plack) to array of pairs (cocaine)
    my @headers = map {[$p_headers->[$_*2], "$p_headers->[$_*2+1]"]} (0 .. $#$p_headers / 2);
    my $body = $self->_build_response_body($p_body);

    $log->info("Response: $status " . length($body))  if $log->is_info();

    return [0+$status, \@headers], $body;
};

override get_error_response => sub {
    my ($self, $error) = @_;

    my $formal_error = CocaineAppException->get_formal_error($error);

    $log->warning("Failure: $formal_error->{message} / $formal_error->{details}")  if $log->is_warn();

    my $response = [
        500,
        ['Content-type' => 'text/plain; charset=UTF-8'],
        [ encode utf8 => join "\n", @{$formal_error}{qw/code message details/} ]
    ];

    return $self->encode_response($response);
};


sub _build_plack_env {
    my ($self, $cocaine_request) = @_;

    my ($http_method, $url, $http_version, $headers, $body) = @$cocaine_request;

    my ($path, $query) = split /\?/, $url, 2;

    my $env = {
        REQUEST_METHOD => $http_method,
        SCRIPT_NAME => "",
        PATH_INFO => URI::Escape::uri_unescape($path),
        REQUEST_URI => $url,
        QUERY_STRING => $query,
        SERVER_NAME => 'localhost',
        SERVER_PORT => '8080',
        SERVER_PROTOCOL => "HTTP/$http_version",
        'psgi.version' => [ 1, 1 ],
        'psgi.url_scheme' => 'http',
        'psgi.nonblocking'  => 1,
        'psgi.streaming'    => 0, # todo: implement
        'psgi.run_once'     => 0,
        'psgi.multithread'  => 0,
        'psgi.multiprocess' => 0,
        'psgi.input' => do { open my $fh, '<', \$body; $fh },
        'psgi.errors' => *STDERR,
        'cocaine.request' => $cocaine_request,
    };

    for my $item (@$headers) {
        my ($key, $val) = @$item;
        $key =~ tr/-/_/;
        $env->{"HTTP_" . uc $key} = $val;
    }

    $env->{CONTENT_TYPE}   = delete $env->{HTTP_CONTENT_TYPE};
    $env->{CONTENT_LENGTH} = delete $env->{HTTP_CONTENT_LENGTH};

    return $env;
}


sub _build_plack_response {
    my ($self, $response) = @_;

    # natural plack response
    return $response  if ref $response eq 'ARRAY' && @$response == 3;

    # data structure
    return [200, ['Content-type' => 'text/x-yaml'], [YAML::Dump($response)]]  if ref $response;

    # text
    return [200, ['Content-type' => 'text/plain; charset=UTF-8'], [encode utf8 => $response]]  if utf8::is_utf8($response);
    return [200, ['Content-type' => 'text/plain'], [$response]]  if $response =~ m/\A [[:ascii:]]* \Z/xms;

    # raw binary
    return [200, ['Content-type' => 'application/octet-stream'], [$response]];
}


sub _build_response_body {
    my ($self, $plack_body) = @_;
    
    my $body;
    if (ref $plack_body eq 'ARRAY') {
        $body = join q{}, @$plack_body;
    }
    else {
        local $/ = \65536 unless ref $/;
        $body = "";
        while (defined(my $line = $plack_body->getline())) {
            $body .= $line;
        }
        $plack_body->close;
    }

    return $body;
}



__PACKAGE__->meta->make_immutable;

1;
