package Cocaine::App::Method::JSON;

use 5.010;
use strict;
use warnings;

use Mouse;
extends 'Cocaine::App::Method';

use Carp;
use JSON;

use Log::Any qw/$log/;
use JSON::Schema::Fit;

has schema => ( is => 'ro' );
has response_schema => ( is => 'rw' );

override decode_request => sub {
    my ($self, $request) = @_;

    if ( $log->is_info() ) {
        my $method = $self->method_sub;
        $log->info("Call: $method $request");
    }

    my $params = eval { decode_json($request || {}) }
    or CocaineAppException->throw( code => 'InvalidRequest', message => "invalid arguments' encoding" );

    return $params->{args} if ref $params eq 'HASH' && $params->{args};
    return $params;
};

override encode_response => sub {
    my ($self, @response_parts) = @_;
    
    CocaineAppException->throw( code => 'InvalidResponse', message => "multi-response not allowed" )  if @response_parts>1;
    my $response = shift @response_parts;

    if ( my $schema = $self->response_schema() ) {
        state $fit = JSON::Schema::Fit->new();
        $response = eval { $fit->get_adjusted($response, $schema) }
        or CocaineAppException->throw( code => 'InvalidResponse', message => "unable to fit response to schema: $@" );
    }

    my $formal_response = { result => $response, error => undef };
    my $response_json = eval { JSON->new->utf8->allow_nonref->encode($formal_response) }
    or CocaineAppException->throw( code => 'InvalidResponse', message => "unable to serialize response: $@" );
    
    $log->info("Succeed: $response_json")  if $log->is_info();

    return $response_json;
};

override get_error_response => sub {
    my ($self, $error) = @_;

    my $formal_error = CocaineAppException->get_formal_error($error);
    my $response = { error => $formal_error };
    my $response_json = JSON->new->utf8->allow_nonref->encode($response);

    $log->info("Failed: $response_json")  if $log->is_info();

    return $response_json;
};


sub use_response_schema {
    my ($self) = @_;

    my $schema = $self->schema()
    or croak $self->method_sub . " does not have schema";

    $self->response_schema($schema->{result});
    return;
}


__PACKAGE__->meta->make_immutable;

1;
