#!/usr/bin/perl

=head1 DESCRIPTION

    Прототип визуализаций Кондукторных групп

    Зависимости скрипта не внесены в зависимости пакета, т.к. скрипт пока экспериментальный и в будущем может отсюда уехать

=head1 EXAMPLES

direct-conductor-viz.pl tree direct_ng_databases_mysql_ppcdata_ppcdata
direct-conductor-viz.pl tree ppcdata

direct-conductor-viz.pl graph group ppcdata9_nonstandby >direct.svg

=head1 TODO

 * выбор api/api-cached
 * полные графы трудночитаемы, слишком мелкие и запутанные. Что делать?
 * переделать в http-сервис?
 * если оставить скрипт -- перейти на Yandex::Conductor
 * если оставить скрипт -- возможно, выделить в отдельный пакет, чтобы не добавлять зависимости к yandex-du-conductor-perl

=cut

use strict;
use warnings; 

use feature qw/state/;

use GraphViz;
use JSON;
use LWP::UserAgent;
use YAML;

my $C_API_URL = "https://c.yandex-team.ru/api-cached";

run() unless caller();

sub run
{
    my $action = $ARGV[0];

    if ($action eq 'tree'){
        my $root_group = $ARGV[1];
        vizualize_tree($root_group);
    } elsif ( 'graph' ){
        my $root_type = $ARGV[1];
        my $root_name = $ARGV[2];
        vizualize_graph($root_type => $root_name);
    } else {
        die "unsupported action '$action', stop\n";
    }
    exit(0);
}


sub vizualize_graph
{
    my ($type, $start_node, %opt) = @_;
    my %UNIVERSUM;
    my $depth = $opt{depth} || 1000;

    my @nodes_to_fetch = ({ type => $type, name => $start_node, depth => $depth });

    my $i = 0;
    while ( @nodes_to_fetch > 0 ){
        die "too large graph, stop" if $i++ > 1000;

        my $n = shift @nodes_to_fetch;
        next if exists $UNIVERSUM{$n->{name}};
        #print "$n->{type}: $n->{name}\n";
        my $node_info;
        if( $n->{type} eq 'group' ){
            $node_info = group_info($n->{name});
        } elsif ( $n->{type} eq 'host' ){
            $node_info = host_info($n->{name});
        } else {
            die;
        }
        $UNIVERSUM{$n->{name}} = $node_info;
        # TODO проверить краевые значения
        if ( $depth > 0 ){
            push @nodes_to_fetch, 
                map { {type => 'host', name => $_, depth => $n->{depth} - 1} } 
                grep { !exists $UNIVERSUM{$_} } 
                @{$node_info->{child_hosts}}
                ;
            push @nodes_to_fetch, 
                map { {type => 'group', name => $_, depth => $n->{depth} - 1} } 
                grep { !exists $UNIVERSUM{$_} } 
                @{$node_info->{child_groups}}, @{$node_info->{parents}};
        } else {
            die;
            $node_info->{truncated}->{child_hosts} = @{$node_info->{child_hosts}} > 0;
            $node_info->{truncated}->{child_groups} = @{$node_info->{child_groups}} > 0;
            $node_info->{truncated}->{parents} = @{$node_info->{parents}} > 0;
        }
    }

    #print YAML::Dump(\%UNIVERSUM);

    my $g = GraphViz->new(directed => 1, ratio => 1);
    for my $n (keys %UNIVERSUM){
        next if $UNIVERSUM{$n}->{type} eq 'host';
        $g->add_node($n);
        # TODO type -> color / shape
    }
    my %drawn;
    for my $n1 (keys %UNIVERSUM){
        next if $UNIVERSUM{$n1}->{type} eq 'host';
        my $node1_info = $UNIVERSUM{$n1};
        my @childs = (
            #  @{$node1_info->{child_hosts}},
            @{$node1_info->{child_groups}}, 
        );
        my @parents = (
            @{$node1_info->{parents}},
        );
        for my $n2 ( @childs ){
            next if $drawn{$n1}->{$n2}++;
            $g->add_edge( $n1, $n2 );
        }
        for my $n2 ( @parents ){
            next if $drawn{$n2}->{$n1}++;
            $g->add_edge( $n2, $n1 );
        }
    }
    print $g->as_svg;
}


sub vizualize_tree
{
    my ($root_group) = @_;
    my %UNIVERSUM;

    my @groups_to_fetch = ($root_group);

    my $i = 0;
    while ( @groups_to_fetch > 0 ){
        die "too large tree, stop" if $i++ > 1000;
        my $gr = shift @groups_to_fetch;
        next if exists $UNIVERSUM{$gr};
        my $group_info = group_info($gr);
        $UNIVERSUM{$gr} = $group_info;
        push @groups_to_fetch, grep { !exists $UNIVERSUM{$_} } @{$group_info->{child_groups}};
    }

    for my $gr ( sort keys %UNIVERSUM ){
        for my $child_host ( @{$UNIVERSUM{$gr}->{child_hosts}} ){
            next if exists $UNIVERSUM{$child_host};
            $UNIVERSUM{$child_host} = {
                type => 'host', 
                name => $child_host, 
                parents => [ $gr],
            };
        }
    }

    #print YAML::Dump(\%UNIVERSUM);
    my_print_tree(\%UNIVERSUM, $root_group);
}

sub my_print_tree
{
    my ($U, $root_name) = @_;
    _my_print_tree($U, $root_name, '', 0, '', 1);
}

=head2 

    $root_name
    $parent_name -- имя родителя, от которого пришли к текущему поддереву
      важно потому, что граф не дерево и "родителей" может быть много
    $level -- на каком уровне вложенности находится текущее поддерево
      в основном важно 0 или не 0
    $branches_prefix -- строка-префикс для изображения незаконченных поддеревьев,
      начавшихся где-то раньше
    $last -- является ли текущее поддерево последним ребенком для своего родителя

=cut
sub _my_print_tree
{
    my ($U, $root_name, $parent_name, $level, $branches_prefix, $last) = @_;
    die "too deep tree, stop" if $level > 1000;
    $level ||= 0;
    my $prefix;
    if ( $level <= 0 ){
        $prefix = '';
    } elsif ( $level > 0 && !$last ){
        $prefix = $branches_prefix."  |-";
    } elsif ( $level > 0 && $last ){
        $prefix = $branches_prefix."  `-";
    } else {
        die;
    }

    my @other_parents = grep { $_ ne $parent_name } @{$U->{$root_name}->{parents}};
    my $parents_comment = '';
    if ( @other_parents > 0 ){
        $parents_comment = ' (other parents: '.(join ", ", @other_parents).')';
    }
    my $type_prefix = $U->{$root_name}->{type} eq 'group' ? "%" : '';

    print $prefix . $type_prefix . $root_name . $parents_comment. "\n";

    my @child_groups = ( sort @{$U->{$root_name}->{child_groups}||[]} );
    my @child_hosts = ( sort @{$U->{$root_name}->{child_hosts}||[]} );
    my @children = (@child_groups, @child_hosts);
    while (my $ch = shift @children){
        my $new_branches_prefix;
        if ( $level <= 0 ){
            $new_branches_prefix = $branches_prefix;
        } elsif ( $level > 0 && !$last ){
            $new_branches_prefix = $branches_prefix."  |";
        } elsif ( $level > 0 && $last ){
            $new_branches_prefix = $branches_prefix."   ";
        } else {
            die;
        }
        
        my $new_last = (@children > 0) ? 0 : 1;
        _my_print_tree($U, $ch, $root_name, $level + 1, $new_branches_prefix, $new_last);
    }
}

sub group_info
{
    my ($group_name) = @_;

    my $group = c_groups([$group_name])->[0];
    my $hosts = c_groups2hosts([$group_name]);

    my $res = {
        type => 'group',
        name => $group->{name},
        child_groups => $group->{children},
        parents => $group->{parents},
        child_hosts => [ map {$_->{fqdn}} @$hosts ],
    };

    return $res;
}

sub host_info
{
    my ($host_name) = @_;
    my $host = c_hosts([$host_name])->[0];
    my $res = {
        type => 'host',
        name => $host->{fqdn},
        parents => [$host->{group}],
        child_groups => [],
        child_hosts => [],
    };
    return $res;
}


sub c_groups2hosts
{
    my ($groups, %opt) = @_;

    state $ua;
    unless ($ua){
        $ua = LWP::UserAgent->new;
        $ua->timeout(10);
    }

    my $groups_str = join(',', @$groups);
    my $param_recursive = $opt{recursive} ? "" : "&recursive=no";
    my $url = $C_API_URL."/groups2hosts/$groups_str?format=json$param_recursive";
    my $response = $ua->get($url);

    unless ($response->is_success) {
        die "can't get c_groups2hosts for '$groups_str', details: ".$response->status_line;
    }
    my $hosts = decode_json($response->content);

    return $hosts;
}

sub c_groups
{
    my ($groups) = @_;

    state $ua;
    unless ($ua){
        $ua = LWP::UserAgent->new;
        $ua->timeout(10);
    }

    my $groups_str = join(',', @$groups);
    my $url = $C_API_URL."/groups/$groups_str?format=json";
    my $response = $ua->get($url);

    unless ($response->is_success) {
        die "can't get group info for '$groups_str', details: ".$response->status_line;
    }
    my $groups_info = decode_json($response->content);

    return $groups_info;
}

sub c_hosts
{
    my ($hosts) = @_;

    state $ua;
    unless ($ua){
        $ua = LWP::UserAgent->new;
        $ua->timeout(10);
    }

    my $hosts_str = join(',', @$hosts);
    my $url = $C_API_URL."/hosts/$hosts_str?format=json";
    my $response = $ua->get($url);

    unless ($response->is_success) {
        die "can't get group info for '$hosts_str', details: ".$response->status_line;
    }
    my $hosts_info = decode_json($response->content);

    return $hosts_info;
}

