use strict;
use warnings;

package Yandex::Conductor;

=head1 DESCRIPTION

Простая некеширующая обертка на http-api Кондуктора

Кондуктор -- админский сервис для много чего, 
и в частности для хранения хостов, групп, и т.п. 

Полезные ссылки:

https://wiki.yandex-team.ru/Conductor
http://c.yandex-team.ru/groups

Все функции экспортируются, имена составляются как conductor_<имя метода api>

=cut

use base qw/Exporter/;

use Encode qw( decode_utf8 );
use Yandex::HTTP qw( http_parallel_request );
use JSON;

our @EXPORT = qw/
    conductor_groups2hosts
    conductor_get_host_tags
    conductor_hosts
/;

our $CONDUCTOR_API_URL ||= "http://c.yandex-team.ru/api";

=head2 conductor_groups2hosts

Разворачивает группы в список хостов.

На входе: 
   либо строка -- имя группы, 
   либо массив строк -- имен групп

На выходе
   массив строк -- имен хостов из всех этих групп

Если запрос к api не прошел успешно -- умирает.

Примеры: 
> perl -MYandex::Conductor -MYAML -le 'print YAML::Dump(conductor_groups2hosts("direct_frontends_sandbox"))' 
---
- ppcsandbox-front01e.yandex.ru
- ppcsandbox-front01f.yandex.ru

> perl -MYandex::Conductor -MYAML -le 'print YAML::Dump(conductor_groups2hosts(["direct_frontends_sandbox", "direct_mysql_sandbox"]))'
---
- ppcsandbox-front01e.yandex.ru
- ppcsandbox-front01f.yandex.ru
- ppcsandbox-mysql01e.yandex.ru
- ppcsandbox-mysql01f.yandex.ru
- ppcstandby04e.yandex.ru

=cut
sub conductor_groups2hosts
{
    my ($groups) = @_;
    die "missing groups" unless $groups;
    $groups = [$groups] unless ref $groups eq "ARRAY";
    return [] if @$groups == 0;

    my $http_result = http_parallel_request(
        GET => { groups => { url => $CONDUCTOR_API_URL."/groups2hosts/".join(",", @$groups) } },
        timeout => 10,
    );

    my $groups_result = $http_result->{groups};

    die $groups_result->{headers}->{Status} unless $groups_result->{is_success};
    return [ split "\n", Encode::decode_utf8( $groups_result->{content} ) ];
}

=head2 conductor_get_host_tags

Запрашивает тэги, заданные для указанного хоста.

    use Yandex::Conductor;
    my $tags = conductor_get_host_tags("ppcdev1.yandex.ru");
    say $_ for @$tags;

=cut
sub conductor_get_host_tags {
    my ($hostname) = @_;

    my $http_result = http_parallel_request(
        GET => { tags => { url => "$CONDUCTOR_API_URL/get_host_tags/$hostname" } },
        timeout => 10,
    );

    my $tags_result = $http_result->{tags};

    die $tags_result->{headers}->{Status} unless $tags_result->{is_success};
    return [ split "\n", Encode::decode_utf8( $tags_result->{content} ) ];
}

=head2 conductor_hosts

Выдает всю информацию о хостах (fqdn, datacenter, group и тд)

Пример:
> perl -MYandex::Conductor -MData::Dumper -le '$hi = conductor_hosts(["ppcdata1-01e.yandex.ru", "ppcdata1-01h.yandex.ru"]); map { print Dumper($_) } @$hi'

=cut
sub conductor_hosts
{
    my ($hosts) = @_;
    die "missing hosts" unless $hosts;
    $hosts = [$hosts] unless ref $hosts eq "ARRAY";
    return {} if @$hosts == 0;

    my $http_result = http_parallel_request(
        GET => { hosts => { url => $CONDUCTOR_API_URL."/hosts/".join(",", @$hosts)."?format=json" } },
        timeout => 10,
    );

    my $hosts_result = $http_result->{hosts};

    die $hosts_result->{headers}->{Status} unless $hosts_result->{is_success};

    return from_json($hosts_result->{content});
}

1;
