package Yandex::CRM;


=head1 NAME

    Yandex::CRM

=head1 DESCRIPTION

    Работа с аттачментами в системе CRM.

    Умеет обращаться к CRM и получать оттуда список аттачментов и скачивать их.

=cut

use Direct::Modern;

use Exporter;
use JSON qw/from_json to_json/;

use Data::Dumper;

use Yandex::HTTP qw/http_fetch/;


our @EXPORT_OK = qw(get_files_list download_file);


=head2 $CRM_URL

    url ручки CRM, есть дефолтное значение,
    которое можно переопределить в настройках.

=cut

our $CRM_URL ||= 'https://tcrm.yandex-team.ru/space/api/';


=head2 $CRM_ATTEMPTS

    Количество попыток запросов в CRM в случае неудачи,
    которое можно переопределить в настройках.

=cut

our $CRM_ATTEMPTS ||= 2;


=head2 get_files_list(ticket_number)

    Получает список файлов в тикете CRM (аттачментов).

    Входной параметр:
        ticket_number -- int, номер тикета в CRM.

    Возвращает структуру:
    {
        files_list => $files_list,
        headers => $headers,
        error_code => $code,
        error_text => $message
    }

    Если операция прошла без ошибок, то по ключам
        error_code и error_text будет лежать undef.

    По ключу files_list будет лежать:

    [
        {
            'fileSize' => 24904,
            'name' => '123-500x333.jpeg',
            'id' => 648
        },
        {
            'fileSize' => 7216,
            'name' => '1OaDsXPFzRQ.jpg',
            'id' => 646
        }
    ]

    Если была прислана ошибка, то по ключу files_list будет лежать undef.

=cut

sub get_files_list {

    my ($ticket_number) = @_;

    my $response = _execute_request('file/list', ticketNum => $ticket_number);

    # Достать ссылку на массив с информацией о файлах
    my $files_list = ${$response->{'content_ref'}}->{'items'};

    # Удалить ненужный ключ content_ref
    delete $response->{'content_ref'};
    # Положить в хеш ссылку на массив с информацией о файлах
    $response->{'files_list'} = $files_list;

    return $response;

}


=head2 download_file(file_id)

    Получает файл (аттачмент) из CRM по его id.

    Входной параметр:
        file_id -- int, id файла в CRM, можно получить с помощью get_files_list.

    Возвращает структуру:
    {
        content_ref => $content_ref,
        headers => $headers,
        error_code => $code,
        error_text => $message
    }

    Если операция прошла без ошибок, то по ключам
        error_code и error_text будет отдаваться undef.
    Если была прислана ошибка, то по ключу content_ref будет отдан undef.
    По ключу content_ref будет лежать ссылка на скаляр -- на контент файла.

=cut

sub download_file {

    my $file_id = shift @_;

    my $path = 'file/download';
    my %params = (id => $file_id);

    return _execute_request($path, %params);

}


=head2 _execute_request(path, param_string)

    Запрашивает данные из CRM по урлу и запросу GET.

    Входные параметры:
        path -- string, путь после основного урла,
        params -- hash, get-параметры.

    Возвращает структуру:
    {
        content_ref => $content_ref,
        headers => $headers,
        error_code => $code,
        error_text => $message
    }

    Если операция прошла без ошибок, то по ключам
        error_code и error_text будет отдаваться undef.
    Если была прислана ошибка, то по ключу content_ref будет отдан undef.
    По ключу content_ref будет лежать ссылка на скаляр -- на контент ответа.

=cut

sub _execute_request {

    my ($path, %params) = @_;

    my $headers;
    my $content = http_fetch("GET", $CRM_URL.$path, \%params,
                             response_headers_ref => \$headers,
                             num_attempts => $CRM_ATTEMPTS,
                             ipv6_prefer => 1,
                             handle_params => {keepalive => 1});
    my $content_type = (split /;/, $headers->{"content-type"})[0];
    my $code = undef;
    my $message = undef;
    my $content_ref = \$content;

    if ($content_type eq 'application/json') {
        $content = from_json($content);
        $content_ref = \$content;

        # В случае ошибки:
        if (exists $content->{'code'} && exists $content->{'message'}) {
            $code = $content->{'code'};
            $message = $content->{'message'};
            $content_ref = undef;
        }
    }
    else {

        if ($path ne 'file/download') {
            my $json_func_params = to_json({
                url => $CRM_URL.$path,
                params => \%params,
                response_headers_ref => \$headers,
                num_attempts => $CRM_ATTEMPTS,
                ipv6_prefer => 1,
                handle_params => {keepalive => 1}});
            die "Expected content-type application/json but \
                $content_type was returned.\n\
                Function arguments are:\n\
                $json_func_params"
        }

    }

    # Во избежание копирования большой строки (содержимого файла),
    # передаем ссылку на $content
    return {
        content_ref => $content_ref,
        headers => $headers,
        error_code => $code,
        error_text => $message
    };

}

1;
