package Yandex::CSVWriter;

=head1 NAME

    Yandex::CSVWriter
    Реализация почанковой записи данных в csv-формате

=cut

use Direct::Modern;

use Text::CSV 1.18;
use Yandex::ScalarUtils;
use Yandex::HashUtils;

=head2 new(%options)

    Создание экземпляра Yandex::ReportsXLSWriter

    Параметры:
    
    $output - хендлер куда писать результат, или SCALARREF, тогда пишем в него, или имя файла, тогда пишем в этот файл (обязательный)
    
    именованные параметры:
        bom_header (0|1)
        header_row - список заголовков столбцов
        binmode
        --опции Text::CSV
        sep_char
        eol
        и другие

=cut

sub new{
    
    my $this = shift;
    my $class = ref($this) || $this;    
    my $output = shift;
    die '$output handler must be defined' unless $output;

    my %options = @_;
    my @main_params = qw/bom_header header_row binmode/;

    my $self = hash_cut \%options, @main_params;
    bless $self, $class;


    $self->{output} = $output;
    $self->{output_file} = $self->{output} unless ref $self->{output};

    open $self->{fh},
        '>' . ($self->{binmode} || ':utf8'),
        ($self->{output_file} ? $self->{output_file} . '.tmp' : $self->{output}) or croak $!;

    if ( $self->{bom_header} ) {
        print {$self->{fh}} "\x{feff}"     or croak $!;
    }

    my $main_params_re = join '|', @main_params;
    my $CSV_opt = hash_merge { binary => 1, eol => "\r\n" }, hash_kgrep {!m/^($main_params_re)$/} \%options;
    $self->{csv} = Text::CSV->new( $CSV_opt );

    if ( $self->{header_row} ) {
        $self->_write($self->{header_row});
    }

    return $self;
}

=head2 add_data

    Дописывает в csv-приемник очередную порцию данных
    Параметры:
        $data - arrayref на список объектов-строк csv-файла
                или ссылка на функцию, возвращающую по одному объектоу-строке за вызов, в виде массива значений столбцов
                в случае arrayref, каждый объекты-строка может быть или arrayref, или hashref (распаковывается с использованием header_row)

=cut

sub add_data {
    my ($self, $data) = @_;
    if ( ref $data eq 'ARRAY' ) {
        for my $row ( @$data ) {
            my $out_row = $row;
            if ( ref $row eq 'HASH' && $self->{header_row} ) {
                $out_row = [ @{$row}{@{$self->{header_row}}} ];
            }
            next unless @$out_row;
            $self->_write($out_row);
        }
    }
    elsif ( ref $data eq 'CODE' ) {
        while ( my @row = $data->() ) {
            $self->_write(\@row);
        }
    }
    else {
        croak 'Invalid data type';
    }
}

=head2 _write
    
    Параметры:
        $row - ссылка на 

=cut

sub _write {
    my ($self, $row) = @_;

    if ($row) {
        $self->{csv}->print( $self->{fh}, $row )     or croak $self->{csv}->error_diag();
    }
}

=head2 close

    Закрывает все открытые дескрипторы, переименовывает временные файлы, если надо - декодирует utf8

=cut

sub close {
    my $self = shift;

    return unless $self->{fh};

    close($self->{fh})  or croak $!;
    delete $self->{fh};

    if ( $self->{output_file} ) {
        rename $self->{output_file}.'.tmp', $self->{output_file}  or croak $!;
    }

    if ( !$self->{binmode} && !$self->{output_file} ) {
        utf8::decode( ${$self->{output}} );
    }
}

=head2 DESTROY

=cut

sub DESTROY {
    my $self = shift;

    $self->close();
}

1;

