package Yandex::HashUtils;

use strict;
use warnings;

use Scalar::Util qw/reftype/;

=pod

=head1 NAME

Yandex::HashUtils - functions for work with hashes - merge, slice, cut.

=head1 REVISION
    $Id$

=head1 SYNOPSIS

    use Yandex::HashUtils;
    
    my $h1 = {a=>"b", c=>"d", e=>"f"};
    my $h2 = {1=>2, 3=>4, g=>"h"};
    
    # for merge hashes
    $h1 = hash_merge $h1, $h2;
    
    # for getting slice of hash
    my $h3 = hash_cut $h1, qw/a e/
    
    # for getting slice of hash
    my $my_keygrep = sub {return /^\d+$/};
    my $h4 = hash_kgrep \&$my_keygrep, $h2;
    
    # for copy values from another hash
    hash_copy \%DEST, \%SRC, @keys;

    # compare two hashes
    hash_diff $h1, $h2;

=head1 METHODS

=cut

use vars qw(@ISA @EXPORT);
require Exporter;

@ISA         = qw(Exporter);
@EXPORT = qw/ 
                            hash_cut
                            hash_merge
                            hash_copy
                            hash_grep
                            hash_kgrep
                            hash_kmap
                            hash_map
                            hash_kv_map
                            hash_diff
                        /;


=head2 hash_cut($HASH_REF, @keys)

    Function make a hash slice
    Returns ref to the slice

=cut

sub hash_cut{
    my $h = shift;
    my $ans = {};

    for my $k (map {((reftype($_) // '') eq 'ARRAY') ? @$_ : $_} @_) {
        next unless defined $k and exists $h->{$k};
        $ans->{$k} = $h->{$k};
    }
    return $ans;
}

=head2 hash_merge($BASE_HASH_REF, @HASH_REFs)

    Function for inline hash merging
    Be sure the first hash given takes attributes of all other hashes
    
    USAGE:
      merge($h1,$h2,$h3..)    i.e. %$h(2,3..) --> %$h1
          if h1 is blessed it left blessed
    
      merge({},$h1,$h2..)
          all %$h(1,2..) dont change theif fields

=cut

sub hash_merge{
    my $h1 = shift;
    for my $h2 (@_) {
        next unless defined $h2 && (reftype($h2) // '') eq 'HASH';

        for my $key (keys %$h2) {
            $h1->{$key} = $h2->{$key};
        }
    }
    return $h1;
}

=head2 hash_copy($DEST_HASH_REF, $SRC_HASH_REF, @keys)

    Function for inline coping values of some hash keys
    Short analog of
    hash_merge $DEST_HASH_REF, hash_cut $SRC_HASH_REF, @keys
    
=cut

sub hash_copy {
    my ($dst, $src) = (shift||{}, shift);
    for my $k (map {((reftype($_) // '')  eq 'ARRAY') ? @$_ : $_} @_) {
        next unless defined $k and exists $src->{$k};
        $dst->{$k} = $src->{$k};
    }
    return $dst;
}

=head2 hash_kmap(&SUB_REF, HASH_REF)

    Function renamed keys from source hash (via SUB_REF)
    Returned reference to new hash
    
=cut

sub hash_kmap(&$) {
    my ($code, $h1) = @_;
    my $h2 = {};
    map { my $a = $_; $h2->{$code->($a)} = $h1->{$_}} keys %$h1;
    return $h2;
}

=head2 hash_map(&SUB_REF, HASH_REF)

    Function apply SUB_REF for each values of source hash
    Returned reference to result hash

=cut

sub hash_map(&$){
    my ($code, $h1) = @_;
    my $h2 = {};
    map { my $k = $_; $_ = $h1->{$_}; $h2->{$k} = $code->($_) } keys %$h1;   # inaffective
#     map { $h2->{$_} = $code->($h1->{$_}) } keys %$h1;               # affective
    return $h2;
}

=head2 hash_kgrep(&SUB_REF, HASH_REF)

    Function make slice of source hash. 
    In result hash copy only keys which satisfy the condition of &sub_ref
    
    Returned reference to new hash

=cut

sub hash_kgrep(&$) {
    my ($code, $h1) = @_;
    my $h2 = {};
    map { my $a = $_; $h2->{$a} = $h1->{$a} if $code->($_)} keys %$h1;
    return $h2;
}

=head2 hash_grep(&SUB_REF, HASH_REF)

    Function make slice of source hash. 
    In result hash copy only values which satisfy the condition of &sub_ref
    
    Returned reference to new hash

=cut

sub hash_grep(&$){
    my ($code, $h1) = @_;
    my $h2 = {};
    map { my $k = $_; $_ = $h1->{$k}; $h2->{$k} = $h1->{$k} if $code->($_) } keys %$h1;
    return $h2;
}

=head hash_diff

    Сравнить 2 хеша, 


    hash_diff {a=>1, b=>2}, {b=>2, c=>3} ==
        { a=>undef, c=>3 }

        
    в идеале хочется следующее поведение
        hash_diff $h2, hash_merge $h1, hash_diff $h1, $h2 == {};
        как этого добится

BUGS & FEATURES:
    не учитываются вложенные объекты
    undef равносильно отсутствию записи в хеше

=cut
sub hash_diff($$){
    my ($p0,$p1) = @_;
    my $dp = {};
    for my $key (keys %$p1) {
        next unless defined $p1->{$key};
        next if ref $p1->{$key};    # skip links
        if (!defined $p0->{$key} or $p0->{$key} ne $p1->{$key}) {
            $dp->{$key} = $p1->{$key};
        }
    }
    for my $key (keys %$p0) {
        next if ref $p0->{$key};
        $dp->{$key} = undef unless defined $p1->{$key};
    }
    return $dp;
}

=head2 hash_kv_map(&SUB_REF, HASH_REF)

    Функция применяет SUB_REF для каждой пары key/value исходного хеша.
    Возвращает ссылку на результирующий хеш:
    {
        key1 => &SUB_REF->(key1, value1),
        key2 => &SUB_REF->(key2, value2),
    }

=cut
sub hash_kv_map(&$){
    my ($code, $h1) = @_;
    my $h2 = {};
    map { my $k = $a = $_; $b = $h1->{$_}; $h2->{$k} = $code->($a,$b) } keys %$h1;   # inaffective
    return $h2;
}


=head1 AUTHOR

    Ivanov Georgy <gerich@yandex-team.ru>
    Zhuravlev Sergey <zhur@yandex-team.ru>
 
=cut


1;
