package Yandex::ListUtils;

=head1 NAME

    Yandex::ListUtils
    Сборник функций для работы с массивами.

=head1 SYNOPSIS

    xuniq {$_->{id}} @list

    # сортировка массива хэшей по name
    xsort {$_->{name}} @list
    # сортировка массива хэшей сначала по name, потом по id
    xsort {($_->{name}, int($_->{id})} @list
    # сортировка массива хэшей сначала по name, потом по id в обратном порядке
    xsort {($_->{name}, \int($_->{id})} @list
    # сортировка массива хэшей сначала по name в обратном порядке, потом по id
    xsort {(\$_->{name}, int($_->{id})} @list

    # вычислить, какие элементы присутствуют в первом массиве, но отсутствуют во втором:
    xdiff( $arr1, xisect($arr1,$arr2) );
    или
    xisect( $arr1, xdiff($arr1, $arr2) );

=cut

use strict;
use warnings;

use Carp qw/croak/;
use List::MoreUtils qw/part/;
use B;

use base qw/Exporter/;
our @EXPORT = qw/
        xuniq 
        xsort nsort 
        xisect xdiff xminus xflatten
        range chunks 
        enumerate_iter
        as_array
        /;
our @EXPORT_OK = qw/
        weighted_shuffle
        weighted_xshuffle
        /;

=head2 xuniq { lc $_ } qw/AA aa Aa aA bB BB/

    Выбрать из списка уникальные элементы
    Параметры: список
        первый элемент списка -- блок, результат выполнения которого используется для определения равенства элементов
        остальное -- элементы, подлежащие уникализации

    Результат: 
        список уникальных значений

    Пример:
    print xuniq { lc $_ } qw/AA aa Aa aA bB BB/
    STDOUT: AAbB

=cut
sub xuniq(&@){
    my $code=shift;
    my %seen=();
    my $undef=0;
    local $_;
    return grep { my $e = $code->($_); !( defined $e ? $seen{$e}++ : $undef++ ) } @_;
}

=head2 xsort BLOCK LIST

=head2 xsort SUBNAME LIST

    "Умная" сортировка, понимающая лексикографическую сортировку по нескольким значениям,
    разный порядок сортировки по разным элементам и правильно сравнивающую строки и числа.

    Переданная функция должна возвращать массив из строковых, числовых скаляров или ссылок 
    на них. Строки сравниваются лексикографически, числа - арифметически. Ссылка меняет
    порядок сортировки данного элемента.

=cut
sub xsort(&@) {
    my $code=shift;
    return () if !@_;

    # первый мэп преобразования Шварца
    my @schwartz_data = map {[$_, $code->($_)]} @_;

    # по первому набору данных будем строить функцию-сортер
    my @first_elem = @{$schwartz_data[0]};
    # если код ничего не возвращал - массив не сортируем
    return @_ if @first_elem == 1;

    # строки - части сортирующей функции
    my @sort_parts;
    for my $i (1..$#first_elem) {
        my $el = $first_elem[$i];
        my ($aa, $bb, $obj);
        if (!ref $el) {
            # если не ссылка - прямой порядок сортировки
            $obj = B::svref_2object(\$el);
            ($aa, $bb) = ("\$a->[$i]", "\$b->[$i]");
        } else {
            # если ссылка - обратный
            $obj = B::svref_2object($el);
            ($aa, $bb) = ("\${\$b->[$i]}", "\${\$a->[$i]}");
        }
        # в зависимости от типа - разный оператор сравнения
        my $op = $obj->FLAGS & (B::SVf_IOK | B::SVf_NOK) ? '<=>' : 'cmp';
        push @sort_parts, "$aa $op $bb";
    }

    # компилируем функцию
    my $sort_sub = eval "sub {".join(" || ", @sort_parts).'}';
    
    # собственно сортировка и второй мэп преобразования Шварца
    return map {$_->[0]} sort $sort_sub @schwartz_data;
}

=head2 nsort LIST

    Числовая сортировка, краткий аналог sort {$a <=> $b}

=cut
sub nsort(@) {
    return sort {$a <=> $b} @_;
}

=head2 @rand = weighted_xshuffle {rand} @arr;

    Случайная пересортировка массива хэшей с учётом веса.
    (Вес - число, вычисляемое функцией, вычисляемой первым аргументом).

=cut
sub weighted_xshuffle(&@) {
    my $code = shift;
    return map { $_->[0] }
        sort { $a->[1] <=> $b->[1] || $a->[2] <=> $b->[2] }
        map { [$_->[0], rand() ** $_->[1], rand() ] }
        map { [$_, $code->()||0] }
        @_;
}

=head2 @rand = weighted_shuffle(@arr);

    Случайная пересортировка массива хэшей с учётом веса.
    (Вес - число, записанное в поле weight).

=cut
sub weighted_shuffle(@) {
    return weighted_xshuffle {$_->{weight} || 0} @_;
}

=head2 range([$start] $stop, [$step])

    Тоже самое, что функция range в Python
    Возвращает массив чисел, которые представляют собой аремфитическую прогрессию.
    
    $start - Начало прогрессии.
    $stop  - Окончание прогрессии.
    $step  - коэфициент
    
=cut
sub range {
    my ($v1, $v2, $v3) = @_;

    my ($start, $stop, $step);
    if (defined($v3)) {
        ($start, $stop, $step) = ($v1, $v2, $v3);
    } elsif (defined($v2)) {
        ($start, $stop, $step) = ($v1, $v2, 1);
    } elsif (defined($v1)) {
        ($start, $stop, $step) = (0, $v1, 1);
    } else {
        # Вернуть пустой массив, если не задан ни один параметр
        return [];
    }
    # Вернуть пустой массив, если хотя бы одно значение не является целым числом
    for ($start, $stop, $step) {
        return [] if $_ !~ /^[\+\-]?\d+$/;
    }

    # Вернуть пустой массив, если достижения $stop при шаге $step невозможно
    return [] if ($start < $stop && $step <=0 || $start > $stop && $step >=0);

    my @values;

    for (my $i = $start; ($start < $stop)?($i < $stop):($i > $stop); $i+=$step) {
        push @values, $i;
    }
    return \@values;
}

=head2 chunks($arr, $n)

    Разбивает массив на куски размера n (последний кусок может получиться меньшего размера)
    Возвращает список из ссылкок.
    Лучше чем вариант с while/splice тем, что не модифицирует исходный список.
    
    $arr - ссылка на исходный массив
    $n   - размер кусков

    Пример:
    for my $chunk (chunks(\@ids, 1_000)) {
        ...
    }
    
=cut
sub chunks($$) {
    my ($arr, $n) = @_;

    croak "Argument n must be positive integer: ".(defined $n ? $n : 'undef') if !$n || $n !~ /^\+?\d+$/;

    my $i = 0;
    return part {int($i++ / $n)} @$arr; # / fix vim highlight
}

=head2 xisect($arr1, $arr2)

    Получает на вход 2 массива (не обязательно с уникальными значениями) вычисляет их пересечение.

    Возвращает ссылку на массив
    
    $arr1 - Ссылка на первый массив
    $arr2  - ссылка на второй массив

    Note: элементы в итоговом массиве не сортируются

    Пример:
    xisect ([1,2,3,4,5], [2,3,4,5,6,7,8]);
    
    (2,3,4,5)
    
    
=cut
sub xisect($$) {
    my ( $arr1, $arr2 ) = @_;

    my %hash1 = map {$_ => undef} @$arr1;
    my @res = xuniq {$_} 
                    grep {exists $hash1{$_}} @$arr2;

    return \@res;

}

=head2 xdiff($arr1,$arr2)

    Получает на вход 2 массива (не обязательно с уникальными значениями) вычисляет их симметрическую разность, то есть те
    элементы, которые присутствуют только в одном из массивов. 

    Возвращает ссылку на массив
    
    $arr1 - Ссылка на первый массив
    $arr2  - ссылка на второй массив

    Note: элементы в итоговом массиве не сортируются

    Пример:
    xdiff ([1,2,3,4,5], [2,3,4,5,6,7,8]);
    
    (1,6,7,8)

=cut
sub xdiff($$) {
    my ( $arr1, $arr2 ) = @_;

    my %hash1 = map {$_ => undef} @{xisect($arr1,$arr2)};
    my @res = xuniq {$_}
                    grep {!exists $hash1{$_}} @$arr1, @$arr2;

    return \@res;
}

=head2 xminus($arr1,$arr2)

    Получает на вход 2 массива (не обязательно с уникальными значениями) вычисляет их разность, то есть те
    элементы, которые присутствуют в первом массиве, но не присутствуют во втором.

    Возвращает ссылку на массив

=cut
sub xminus($$) {
    my ($arr1, $arr2) = @_;

    my %hash2 = map {$_ => undef} @$arr2;

    return [grep {!exists $hash2{$_}} @$arr1];
}


=head2 xflatten($arg1, $arg2, $arg3, ....)

    Преобразовываем параметры в один одноуровневый массив
    (без глубоко просмотра параметров, т.е. содержимое $argN остаётся без изменений)

=cut

sub xflatten(@) {
    return map {ref $_ eq 'ARRAY' ? @$_ : $_} @_;
}

=head2 enumerate_iter($arr1, $arr2, ...)

    Генератор кортежей (индекс, значение1, ...) + zip
    Количество возвращаемых значений - длина первого из массивов

    Пример использования:
    > my $it = enumerate_iter(\@cids)
    > while(my ($idx, $cid) = $it->()) {
    >     print "$idx: $cid\n";
    > }
    0 1
    1 2
    2 3

=cut
sub enumerate_iter {
    my @arrs = @_;
    my $i = 0;
    return sub {
        return () if $i > $#{$arrs[0]};
        my $idx = $i++;
        return ($idx, map {$_->[$idx]} @arrs);
    };
}

=head2 as_array

Возвращает ссылку на массив, если параметр - скаляр или ссылка на массив:
    my $x = 'x';
    my $y = [qw/y z/];
    for my $i (@{as_array($x)}, @{as_array($y)}) ...

=cut

sub as_array
{
    return ref $_[0] eq 'ARRAY' ? $_[0] : [$_[0]];
}

1;
