package Yandex::DateTime;

# $Id$

=head1 NAME
    
    Yandex::DateTime
    Обёртка вокруг DateTime

=head1 SYNOPSIS

    my $d = date("2009-01-01");
    my $d1 = date("20090101");
    my $dt = datetime("2009-01-01 12:23:34");
    my $dt1 = datetime("20090101000000");

    duration(days=>1, months=>-1);
    duration("1d-1m");

    tz_offset("Europe/Paris");

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use DateTime;
use DateTime::Duration;
use DateTime::Format::MySQL;
use DateTime::Format::ISO8601;

use base qw/Exporter/;
our @EXPORT = qw/
                now
                now_utc
                date 
                datetime 
                duration 
                tz_offset 
                iso8601_2_mysql 
                iso8601_2_datetime
                mysql2iso8601
                /;

# наша текущая временная зона на наших серверах, если не передано иного, используем ее
our $YANDEX_LOCAL_TIMEZONE ||= 'Europe/Moscow';

=head2 date()

    Из строки в формате mysql сделать объект DateTime.
    Поддерживаемые форматы; YYYMMDD, YYYY-MM-DD

=cut
sub date {
    my $string = shift;
    $string =~ s/^(\d{4})(\d{2})(\d{2})$/$1-$2-$3/;
    return DateTime::Format::MySQL->parse_date($string, @_);
}

=head2 now()

    Возвращает текущее время в форме объекта DateTime;
    TimeZone берётся из локальных настроек

=cut
sub now {
    my $time_zone = DateTime::TimeZone->new(name=>'local');
    return DateTime->now(time_zone => $time_zone);
}

sub now_utc {
    return DateTime->now();    
}

=head2 datetime()

    Из строки в формате mysql сделать объект DateTime.
    Поддерживаемые форматы; YYYMMDDHHMMSS, YYYY-MM-DD HH:MM:SS

=cut
sub datetime {
    my $string = shift;
    $string =~ s/^(\d{4})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})$/$1-$2-$3 $4:$5:$6/;
    return DateTime::Format::MySQL->parse_datetime($string, @_);
}

=head2 duration()

    Удобный способ зоздания DateTime::Duration
    Можно передавать либо параметры для DateTime::Duration->new(),
    либо строку, с закодированным интервалом "1y1d-1m" - один год и один день без месяца

=cut

sub duration {
    if (@_ == 1) {
        my $d = $_[0];
        my %duration;
        my %translate = ('y' => 'years', 'm' => 'months', 'd' => 'days',
                         'H' => 'hours', 'M' => 'minutes', 'S' => 'seconds',
                         'hour' => 'hours', 'min' => 'minutes', 'sec' => 'seconds',
            );
        my $translate_re = join '|', sort {$b cmp $a} map {"\Q$_\E"} keys %translate;
        while($d =~ s/^([-+]?\d+)($translate_re)//) {
            $duration{$translate{$2}} += $1;
        }
        die "Can't parse duration '$d'" if $d ne '';
        return DateTime::Duration->new(%duration);
    } else {
        return DateTime::Duration->new(@_);
    }
}

=head2 tz_offset($tz_name)

    Получить смещение относительно GMT для тайм-зоны по названию.

=cut

sub tz_offset {
    my ($tz_name) = shift;
    return DateTime->now(time_zone => $tz_name)->offset();
}

=head2 iso8601_2_mysql

    Из строки формата iso8601 получить mysql datetime

=cut

sub iso8601_2_mysql($) {
    my $str = shift;
    my $dt = iso8601_2_datetime($str);
    return $dt->ymd('-').' '.$dt->hms(':');
}

=head2 iso8601_2_datetime

    Из строки формата iso8601 получить объект DateTime

=cut

sub iso8601_2_datetime($) {
    my $str = shift;
    my $dt = DateTime::Format::ISO8601->parse_datetime( $str );

    $dt->set_time_zone($YANDEX_LOCAL_TIMEZONE);
    return $dt;
}

=head2 mysql2iso8601

=cut

sub mysql2iso8601($$) {
    my ($mysql_datetime, $timezone) = @_;

    my $date = datetime($mysql_datetime);

    $date->set_time_zone($timezone);
    $date->set_time_zone('UTC');

    return $date->ymd.'T'.$date->hms.'Z';
}

1;
