package Yandex::TimeCommon;

=pod
    $Id$
    
=cut

use strict;
use warnings;

use Time::Local;
use POSIX qw/strftime/;

require Exporter;
our @ISA = qw/Exporter/;
our @EXPORT = qw/
        ts_round
        ts_round_5min ts_round_hour ts_round_day ts_round_week ts_round_month ts_round_year 
        str_round str_round_day str_round_week str_round_month str_round_quarter str_round_year
        mysql_round_day
        ts_get_borders 
        ts_get_hour_borders ts_get_day_borders ts_get_week_borders ts_get_month_borders ts_get_year_borders
        ts_to_str ts_to_tbl_suffix
        mysql2unix unix2mysql unix2human human_datetime gm_to_local
        check_mysql_date
        normalize_date
        today yesterday tomorrow
        ts_get_distinct_dates get_distinct_dates get_distinct_months
        mysql2human_date human_date
        is_time_yet
        /;

###########################################################
#хеши %num и %month используются в процедуре ts_get_distinct_dates
our %num = map {$_ => sprintf("%02d",$_)} (1..31);
our %month = (
    1  => 31,
    2  => 28,
    3  => 31,
    4  => 30,
    5  => 31,
    6  => 30,
    7  => 31,
    8  => 31,
    9  => 30,
    10 => 31,
    11 => 30,
    12 => 31,
);

# Округления дат

# Округление до года
sub ts_round_year {
    my $ts = shift;
    my ( $ty ) = ( localtime( $ts || time ) )[5];
    return timelocal(0, 0, 0, 1, 0, 1900 + $ty);
}

sub str_round_year {
    my $date = shift;
    $date =~ s/^(\d{4}).*/$1-01-01/;
    return $date;
}

# округление до квартала
sub str_round_quarter {
    my $date = shift;
    if ($date =~ /^(\d{4})\D?(\d{2})/) {
        my $y = $1;
        my $m = $2;
        my $quarter = int(($m - 1) / 3);
        my $m_rounded = 1 + 3*$quarter;
        $m_rounded = "0$m_rounded" if $m_rounded < 10;
        return "$y-$m_rounded-01";
    } else {
        die "invalid date '$date'";
    }
}

# Округление до месяца
sub ts_round_month {
    my $ts = shift;
    my ( $tm, $ty ) = ( localtime( $ts || time ) )[4,5];
    return timelocal(0, 0, 0, 1, $tm, 1900 + $ty);
}

sub str_round_month {
    my $date = shift;
    $date =~ s/^(\d{4})\D?(\d{2}).*/$1-$2-01/;
    return $date;
}

# Округление до недели
sub ts_round_week {
    my $ts = shift || time;
    $ts -= (((localtime($ts))[6]||7)-1)*23*60*60;
    my ($td, $tm, $ty) = ( localtime($ts) )[3,4,5];
    return timelocal(0, 0, 0, $td, $tm, 1900 + $ty);
}

sub str_round_week {
    my $date = shift;
    my $ts = mysql2unix($date);
    $ts = ts_round_week($ts);
    return strftime("%Y-%m-%d", localtime $ts);
}

# Округление до дня
sub ts_round_day {
    my $ts = shift;
    my ( $td, $tm, $ty ) = ( localtime( $ts || time ) )[3,4,5];
    return timelocal(0, 0, 0, $td, $tm, 1900 + $ty);
}

sub str_round_day {
    my $date = shift;
    $date =~ s/^(\d{4})\D?(\d{2})\D?(\d{2}).*/$1-$2-$3/;
    return $date;
}


{
    my %str_round_sub = (
        'year' => \&str_round_year,
        'quarter' => \&str_round_quarter,
        'month' => \&str_round_month,
        'week' => \&str_round_week,
        'day' => \&str_round_day,
    );
    sub str_round {
        my ($str, $dateagg) = @_;
        my $subref = $str_round_sub{$dateagg};
        die "Unknown dateagg: '$dateagg'" if !$subref;
        return $subref->($str);
    }
}

################################################

# Округление до дня
# возвращаем дату в том же формате, что и today - YYYYMMDD
# иногда нужно получать дату в формате через '-', поэтому
# если передан параметр delim, используем его как разделите
sub mysql_round_day {
    my ($dt, %O) = @_;
    my $delim = $O{delim} || '';

    if (defined $dt && $dt =~ /^([1-9]\d{3})-?(\d{2})-?(\d{2})/) {
        return join $delim, ($1, $2, $3);
    } else {
        return undef;
    }
}

# Округление до часа
sub ts_round_hour {
    my $ts = shift;
    my ( $th, $td, $tm, $ty ) = ( localtime( $ts || time ) )[2,3,4,5];
    return timelocal(0, 0, $th, $td, $tm, 1900 + $ty);
}

# округление с точностью 5 минут
sub ts_round_5min {
    my $ts = shift;
    my $remainder = $ts % (60*60);#остаток(минуты и секунды)
    my $minutes = int($remainder / 60);#число полных минут
    my $bound = 5;
    
    my $m = $bound * int( $minutes / $bound);
    return $ts - $remainder + $m*60;
}
        
# Округлить до какого-нибудь уровня аггрегации
{
    my %ts_subs = (
        'year' => \&ts_round_year,
        'month' => \&ts_round_month,
        'week' => \&ts_round_week,
        'day' => \&ts_round_day,
        'hour' => \&ts_round_hour,
        '5min' => \&ts_round_5min,
        'sec' => sub {return $_[0]},
    );
    sub ts_round {
        my ( $ts, $dateagg ) = @_;
        my $subref = $ts_subs{$dateagg};
        if ( $subref ) {
            return $subref->($ts);
        } else {
            die "Unknown dateagg: '$dateagg'";
        }
    }
}

################################################
# нахождение временных границ уровня аггрегации

# для часа
sub ts_get_hour_borders {
    my $ts = shift;
    my $start_time = ts_round_hour($ts);
    return $start_time, $start_time + 60*60 - 1;
}

# для дня
sub ts_get_day_borders {
    my $ts = shift;
    my $start_time = ts_round_day($ts);
    return $start_time, ts_round_day($start_time + 26*60*60)-1;
}

# для недели
sub ts_get_week_borders {
    my $ts = shift;
    my $start_time = ts_round_week($ts);
    return $start_time, ts_round_week($start_time + 8*24*60*60)-1;
}

# для месяца
sub ts_get_month_borders {
    my $ts = shift;
    my $start_time = ts_round_month($ts);
    return $start_time, ts_round_month($start_time + 33*24*60*60)-1;
}

# для года
sub ts_get_year_borders {
    my $ts = shift;
    my $start_time = ts_round_year($ts);
    return $start_time, ts_round_month($start_time + 370*24*60*60)-1;
}

# для любого уровня
{
    my %ts_subs = (
        'year' => \&ts_get_year_borders,
        'month' => \&ts_get_month_borders,
        'week' => \&ts_get_week_borders,
        'day' => \&ts_get_day_borders,
        'hour' => \&ts_get_hour_borders,
    );
    sub ts_get_borders {
        my ( $ts, $dateagg ) = @_;
        my $subref = $ts_subs{$dateagg};
        if ( $subref ) {
            return $subref->($ts);
        } else {
            die "Unknown dateagg: '$dateagg'";
        }
    }
}

# из timestamp получаем строку
{
    my %fmt = (
        hour => '%Y%m%d%H',
        day => '%Y%m%d',
        week => '%Y%m%d',
        month => '%Y%m',
        year => '%Y',
    );
    sub ts_to_str {
        my ( $ts, $agg ) = @_;
        if ( $fmt{$agg||'day'} ) {
            return strftime($fmt{$agg||'day'}, localtime($ts));
        } else {
            die "Unknown aggregate level: $agg";
        }
    }
}

# из timestamp получить суффикс таблицы
{
    my %suf = (
        hour => 'hourly',
        day => 'daily',
        week => 'weekly',
        month => 'monthly',
        year => 'yearly',
    );
    sub ts_to_tbl_suffix {
        my ( $ts, $agg ) = @_;
        $agg ||= 'day';
        if ( defined $suf{$agg} ) {
            return "$suf{$agg}_".ts_to_str($ts, $agg);
        } else {
            die "Unknown aggregate level: $agg";
        }
    }
}

################################################
# Преобразование дат

# Преобразовать число из формата mysql в unix
sub mysql2unix {
    my $mysql = shift;
    if (defined $mysql && $mysql =~ /^0000/) {
        die "Incorrect date: '$mysql'";
    } elsif (defined $mysql && $mysql =~ /^(\d{4})-?(\d{2})-?(\d{2})\s*(\d{2}):?(\d{2}):?(\d{2})$/ ) {
        return timelocal( $6, $5, $4, $3, $2-1, $1 );
    } elsif (defined $mysql && $mysql =~ /^(\d{4})-?(\d{2})-?(\d{2})$/ ) {
        return timelocal( 0, 0, 0, $3, $2-1, $1 );
    } elsif (defined $mysql && $mysql eq '0' ) {
        return 0;
    } else {
        die "Unknown format ".(defined $mysql ? "'$mysql'" : 'undef');
    }
}

# Преобразовать число из формата unix в mysql
sub unix2mysql {
    my $unix = shift;
    return strftime("%Y%m%d%H%M%S", localtime($unix));
}

# Преобразовать число из формата unix в человеко-понятный вид
sub unix2human {
    my $unix = shift;
    my $date_format = shift;
    return strftime($date_format || "%Y-%m-%d %H:%M:%S", localtime($unix));
}
sub human_datetime {
    return unix2human($_[0]||time());
}

sub human_date {
    return unix2human($_[0]||time(), "%d.%m.%Y");
}
sub mysql2human_date {
    if (defined $_[0] && $_[0] =~ /^(\d{4})-?(\d{2})-?(\d{2})/ ) {
	return "$3.$2.$1";
    } else {
	return undef;
    }
}

# Получить список дат, которые лежат в диапазоне двух дат
sub get_distinct_dates {
    return ts_get_distinct_dates(mysql2unix($_[0]), mysql2unix($_[1]));
}

# Получить месяцев в формате YYYYMM, в которые попадает диапазон указанных дат
sub get_distinct_months {
    my ($d1, $d2) = @_;

    # check d2
    my $ts1 = mysql2unix($d1);
    my $ts2 = mysql2unix($d2);
    return () if $ts1 > $ts2;

    my ($m1, $y1) = (localtime $ts1)[4,5];
    $y1 += 1900; $m1++;
    # избавляемся от "-" в дате (2013-12-12)
    $d2 = unix2mysql($ts2);

    my @ret;
    while(1) {
        my $cur = sprintf("%04d%02d", $y1, $m1);
        last if $cur gt $d2;
        push @ret, $cur;
        if ($m1 < 12) {
            $m1++;
        } else {
            $y1++;
            $m1 = 1;
        }
    }
    return @ret;
}

# Получить список дат, которые лежат в диапазоне двух timestamp-ов
sub ts_get_distinct_dates {
    my ( $start_time, $end_time )  = @_;
    return () if $start_time > $end_time;
    
    my ($d,$m,$y) = (localtime($end_time))[3..5];
    $m++;
    $y+= 1900;
    my $end = join('',$y,$num{$m},$num{$d});
    
    ($d,$m,$y) = (localtime($start_time))[3..5];
    $m++;
    $y+= 1900;
    $month{2} = is_leap_year($y) ? 29 : 28;

    my $start = join('',$y,$num{$m},$num{$d});
    my @res;
    while( $start <= $end) {
        push(@res,$start);

        $d++;
        if ($month{$m} < $d) {
            $m++;
            $d=1;
        }
        if ($m>12) {
            $y++;
            $m=1;
            $month{2} = is_leap_year($y) ? 29 : 28;
        }
        $start = join('',$y,$num{$m},$num{$d});
    }
    return @res;
}

sub is_leap_year {
    my ($year) = @_;
    return ($year % 4 ==0) && ($year % 100) || ($year % 400 ==0) ? 1 : 0;
}

# get today date
{
my $t = 0;
my $today;
sub today {
    if (!$t || $t != time) {
	$t = time;
	$today = ts_to_str($t);
    }
    return $today;
}
}
    
# Get yesterday for date
sub yesterday {
    my $date = shift;
    if (!defined $date) {
        return ts_to_str( timelocal( 0, 0, 0, (localtime)[3, 4, 5] ) - 12*60*60 );
    } elsif ($date =~ /^(\d{4})-?(\d{2})-?(\d{2})$/) {
        return ts_to_str( timelocal( 0, 0, 0, $3, $2-1, $1-1900 ) - 12*60*60 );
    } else {
        die "Incorrect date format: '$date'";
    }
}

# Get tomorrow for date
sub tomorrow{
    my $date = shift;
    my ($day, $month, $year);
    if (!defined $date) {
        ($day, $month, $year) = (localtime)[3, 4, 5];
    } elsif ($date =~ /^(\d{4})-?(\d{2})-?(\d{2})$/) {
        $day = $3;
        $month = $2 - 1;
        $year = $1;
    } else {
        die "Incorrect date format: '$date'";
    }
    
    return ts_to_str( timelocal( 0, 0, 0, $day, $month, $year)  + 3*12*60*60 );
}

# перевод времени из GMT в local
sub gm_to_local {
    my $date = shift;
    my $gmtime = mysql2unix($date);
    my @localtime = localtime($gmtime);
    $localtime[5] += 1900;
    return unix2mysql(timegm @localtime);
}

# Проверка строки на валидную дату
sub check_mysql_date {
    return 0 if !defined $_[0] || $_[0] eq '' || $_[0] =~ /^0000/;
    eval {mysql2unix(@_)};
    return $@ ? 0 : 1;
}

# преобразование даты формата yyy-m-d в yyyy-mm-dd, даты в других форматах (например yyyymmdd) остаются как есть
sub normalize_date {
    my $date = shift;
    if (defined $date && $date =~ /^(\d+)-(\d+)-(\d+)$/) {
        return sprintf('%04d-%02d-%02d', $1, $2, $3);
    } 
    return $date;
}

=head2
    sub is_time_yet ($cur_hour, $cur_min, $hour_from, $min_from, $hour_to, $min_to)
    В предположении, что границы интервала не совпадают, возвращает 1, если переданное текущее время попадает в интервал с учётом границы суток.
=cut

sub is_time_yet($$$$$$) {
    my ($cur_hour, $cur_min, $hour_from, $min_from, $hour_to, $min_to) = @_;
    my $cur_time  = $cur_hour  * 60 + $cur_min;
    my $time_from = $hour_from * 60 + $min_from;
    my $time_to   = $hour_to   * 60 + $min_to;
    if (($cur_time >= $time_from && $cur_time <= $time_to) || # промежуток не на границе суток
            ($time_from > $time_to && # промежуток на границе суток
                ($cur_time <= $time_to || # после полуночи
                 $cur_time >= $time_from))) { # до полуночи
        return 1;
    }
    return 0;
}

1;
