#!/usr/bin/perl -w

# $Id$

=head1 NAME

db_schema.pl - генерация и проверка файлов со схемой БД

=head1 DESCRIPTION

    Скрипт генерации и проверки файлов со схемой БД.

    Смотрит струкутру БД в текущей конфигурации (всей или отдельных таблиц), 
    смотрит на сейчас сохраненные файлы в каталоге db_schema, 
    сравнивает одно с другим, 
    так или иначе сообщает о расхождении. 

    Варианты "сообщить о расхождении": 
    db_schema.pl      # печатает сводку несоответсвий файлов и БД
    db_schema.pl -i   # интерактивный режим
    db_schema.pl -w   # обновляет файлы до соответсвия с БД

    db_schema.pl -db ppc
    db_schema.pl -w -db ppclog
    db_schema.pl -i -db ppcdict -t suggest_phrases
    db_schema.pl -tf org_details

    Опции: 

    -h, --help
        вывести справку

    -i, --interactive 
        интерактивный режим работы: про каждое несоответсвие файлов и БД спрашивает, обновлять ли файл, и при положительном ответе (y) обновляет

    -w, --non-interactive-write
        неинтерактивный режим работы: пишет все несоответствия в STDOUT + обновляет файлы

    --compare-init-data
        Сравнивать данные в таблицах с содержимым *.init-data.sql
        Нужно только в случаях, когда появляется новый файл *.init-data.sql и его нужно заполнить из образцовой-очищенной таблицы.

    При обновлении/удалении файла делается и svn add/svn rm

    Если не указано ни -i, ни -w (=умолчальный режим работы)
        пишет в STDOUT все несответсвия файлов и БД. Если несоответствий нет -- молча завершается.

    -db, --database <database> 
        смотреть только на указанную базу данных

    -t, --table <table> 
        смотреть только на указанную таблицу

    -tf, --text-file <table>
        сгенерировать текстовый файл с описанием указанной таблицы

    --show_old_temporals
        генерировать комментарии вида /* 5.5 binary format */ к временным колонкам в старом формате, блокирующим online-альтеры
        работает только начиная с 5.6

    --skip_dir=<db_dir>
    --skip_dir=<another_db_dir>
        пропускать указанные «лишние» каталоги с описаниями

    Человеческие комментарии к таблицам предлагается хранить в файлах c именем таблицы, но с суффиксом .text
    db_schema/database/table.text


=head1 COMMENTS

   TODO 

    (+) конфиг с базами, исключениями и т.п. -- в параметрах принимать путь к файлу
    (+) путь к конфигу для DBTools уметь принимать из параметров
    
    (+) каталог, в котором лежат текстовые файлы -- из параметров

    * игнорировать часть файлов (skip_file)

    * уметь не только печатать сводку, но и возвращать ее как perl-данные 
    * хочется сравнивать между собой реплики (структура одинакова)
    * хочется проверять Директовую Песочницу (таблицы как в продакшене)

    * direct_conf.json переименовать в db_schema.conf и переместить в Директ, в каталог etc рабочей копии (geocontext.conf -- в Геоконтекст)
    * скопировать из Директа checkDirectDBSchema.pl, научить его работать с конфигами из определенного каталога
    * написать конфиги для Директа и Геоконтекста


=cut

use strict;

use File::Find;
use File::Slurp;
use File::Basename; 
use Text::Diff;
use Encode;
use Getopt::Long;
use Cwd qw/abs_path/;

use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::Interactive qw/prompt prompt_yn/;
use Yandex::Shell;
use Yandex::Svn; 

use Data::Dumper;

use utf8;
use open ':std' => ':utf8';

    
# переменная для глобального хранения режима работы
my %MODE;


run() unless caller();

sub run
{
    my %OPTIONS = parse_options();

    print "***\ndb_config: $OPTIONS{conf}->{db_config}\ndir:       $OPTIONS{main_dir}\nconfig:    $OPTIONS{conf_file}\n***\n";
    if ($OPTIONS{conf}{db_config} eq '/etc/yandex-direct/db-config.json') {
        $Yandex::DBTools::DB_USER = sub { $_[0] =~ /^((sharded_)?unit_tests|ppchouse)/ ? undef : 'direct-ro' };
    }

    if ($OPTIONS{text_file}) {
        generate_text_file($OPTIONS{text_file}, 
            conf => $OPTIONS{conf},
            main_dir => $OPTIONS{main_dir},
        );
    } else {
        my $schema_from_db = get_schema_from_db(
            conf => $OPTIONS{conf},
            database => $OPTIONS{database},
            table => $OPTIONS{table},
            show_old_temporals => $OPTIONS{show_old_temporals},
        );
        my $schema_from_files = get_schema_from_files(
            conf => $OPTIONS{conf},
            main_dir => $OPTIONS{main_dir},
            no_svn => $OPTIONS{no_svn},
        );

        compare_schemata(
            files => $schema_from_files,
            db => $schema_from_db,
            main_dir => $OPTIONS{main_dir},
            opt => {
                database    => $OPTIONS{database},
                table       => $OPTIONS{table},
                skip_dir    => $OPTIONS{skip_dir},
            },
        );
    }

    exit; 
}


sub parse_options 
{
    my %O;

    GetOptions( 
        'h|help'                  => \&usage, 
        'c|conf=s'                => \$O{conf_file},
        'db-config=s'             => \$O{db_config},
        'd|dir=s'                 => \$O{main_dir},
        'w|non-interactive-write' => \$O{non_interactive_write}, 
        'i|interactive-write'     => \$O{interactive_write}, 
        'compare-init-data'       => \$O{compare_init_data},
        'db|database=s'           => \$O{database}, 
        't|table=s'               => \$O{table}, 
        'no-svn'                  => \$O{no_svn}, 
        'tf|text-file=s'          => \$O{text_file},
        'show_old_temporals'      => \$O{show_old_temporals},
        'skip_dir=s@'             => \$O{skip_dir},
    ) or die "can't parse options";

    die "--conf </path/to/conf/file> missed" unless $O{conf_file};
    die "--dir </path/to/db_schema/dir> missed" unless $O{main_dir};

    $O{conf_file} = abs_path($O{conf_file});
    $O{main_dir} = abs_path($O{main_dir});
    $O{db_config} = abs_path($O{db_config}) if $O{db_config};

    $O{conf} = JSON::from_json(read_file($O{conf_file}, binmode => ":utf8"));
    $O{conf}->{db_config} = $O{db_config} if $O{db_config};
    $Yandex::DBTools::CONFIG_FILE = $O{conf}->{db_config};

    $O{write_files} = 1 if $O{interactive_write} || $O{non_interactive_write};
    $MODE{$_} = $O{$_} for qw/write_files interactive_write non_interactive_write compare_init_data/;

    return %O;
}


sub usage {
    system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(0);
}


=head2 get_table_list_from_database

    [ {table => logrbac, nick => logrbac}, {table => events_201009, nick => events_nnnnnn}]

=cut

sub get_table_list_from_database
{
    my %O = @_;

    my @tables_as_is = @{ get_one_column_sql($O{db}, "show tables from $O{database}") };

    my %suffix_by_date = (
        6 => 'YYYYMM',
        8 => 'YYYYMMDD',
    );


    my %tables_nicks; 
    for my $t (@tables_as_is){
        $t =~ /_(\d+)$/;
        my $table_suff = defined $1 ? $1 : '';
        my $nick_suff = $O{suffixes_by_date} ? $suffix_by_date{length $table_suff} : '';
        if (!$nick_suff){
            ($nick_suff = $table_suff) =~ s/\d/n/g if !$nick_suff;
            $nick_suff = 'n' x 7 if $O{no_serial_numbers} || length $nick_suff > 2;
        }
        (my $nick = $t) =~ s/_\d+$/_$nick_suff/; 
        push @{$tables_nicks{$nick}}, $t;
    }

    my @tables;
    for my $nick (keys %tables_nicks){
        my $table = [ sort {$b cmp $a} @{$tables_nicks{$nick}} ]->[0];
        push @tables, {nick => $nick, table => $table}
    }

    return \@tables;
}

sub get_procedure_list_from_database {
    my %O = @_;

    my $status = get_all_sql( $O{db}, 'show procedure status' );

    my @procedures;
    for my $status_row (@$status) {
        next unless $status_row->{Db} eq $O{database};
        push @procedures, $status_row->{Name};
    }

    return \@procedures;
}

=head2 get_schema_from_db

=cut

sub get_schema_from_db
{
    my (%O) = @_;
    my $instances = $O{conf}->{instances};
    my $schema;

    for my $instance (keys %$instances) {
        my %skip_db = map {$_ => 1} @{$instances->{$instance}->{skip_db}||[]};
        my %skip_table = map {$_ => 1} @{$instances->{$instance}->{skip_table}||[]};
        my %dump_table = map {$_ => 1} @{$instances->{$instance}->{dump_data}||[]};
        my %init_data = map {$_ => 1} @{$instances->{$instance}->{init_data}||[]};
        # выбраем все базы, исключая системные
        my @databases = eval { grep {/^\w+$/ && !/^(performance_schema|information_schema|mysql|sys)$/} @{get_one_column_sql($instance, "show databases") } };
        if ( $@ ){
            warn "Can't get databases for $instance: $@";
            exit 1 if $MODE{non_interactive_write};
            next;
        }
        if ($O{show_old_temporals}) {
            local $Yandex::DBTools::DONT_SEND_LETTERS = 1;
            local get_dbh($instance)->{PrintError} = 0;
            eval { do_sql($instance, 'SET SESSION show_old_temporals = ON') };
            if ($@) {
                warn "WARNING: error setting show_old_temporals: $@";
            }
        }
        for my $database (@databases) {
            next if $skip_db{$database};
            next if $O{database} && $database ne $O{database};
            $schema->{$database}->{instance} = "$instance\n";
            
            for my $table_hash ( @{ get_table_list_from_database(db => $instance, database => $database, map { $_ => $instances->{$instance}->{$_} || ''} qw/suffixes_by_date no_serial_numbers/)} ){
                my ($nick, $table) = @$table_hash{qw/nick table/};
                next if $O{table} && $table ne $O{table};

                next if $skip_table{"$database.$table"} || $skip_table{"$database.$nick"};
                #print "table $instance.$database.$table => $instance.$database.$nick\n";

                my $create_table_sql = get_one_line_sql($instance, "show create table $database.$table")->{"Create Table"};
                next if !$create_table_sql;
                $create_table_sql =~ s/AUTO_INCREMENT=\d+\s*//;
                $create_table_sql =~ s/\b$table\b/$nick/;
                $create_table_sql =~ s/^(\s*CONSTRAINT\s*`)_+/$1/gm;

                $schema->{$database}->{schema}->{$nick} = $create_table_sql . "\n";
                if ($dump_table{"$database.$table"}){
                    $schema->{$database}->{data}->{$table} = dump_table(
                        instance => $instance, 
                        database => $database, 
                        table => $table,
                    );
                }

                if ( $init_data{"$database.$table"} ) {
                    if ( $MODE{compare_init_data} ) {
                        $schema->{$database}->{init_data}->{$table} = dump_table(
                            instance => $instance, 
                            database => $database, 
                            table => $table,
                        );
                    } else {
                        $schema->{$database}->{init_data}->{$table} = "(non-empty)\n";
                    }
                }
            }

            for my $procedure ( @{ get_procedure_list_from_database( db => $instance, database => $database ) } ) {
                my $create_procedure_sql = get_one_line_sql( $instance, "show create procedure $database.$procedure" )->{'Create Procedure'};
                $schema->{$database}->{'procedure'}->{$procedure} = $create_procedure_sql;
            }
        }

    }

    return $schema;
}


=head2 dump_table

    mysqldump -f --single-transaction --skip-extended-insert --no-create-db --no-create-info --compact --skip-comments --complete-insert --default-character-set=utf8 --user=adiuser --password=utro --host=ppcmaster01b.yandex.ru --port=3311 ppcdict yandex_offices

=cut

sub dump_table
{
    my %O = @_;

    my $cfg = get_db_config($O{instance});

    my @mysqldump_args = (
        "--default-character-set=utf8",
        "--force",
        "--single-transaction",
        "--skip-extended-insert",
        "--no-create-db",
        "--no-create-info",
        "--compact",
        "--skip-comments",
#        "--complete-insert",
        "--user=$cfg->{user}",
        "--password=$cfg->{pass}",
        "--host=$cfg->{host}",
        "--port=$cfg->{port}",
    );
    
    my $data_sql = eval { yash_qx('mysqldump', @mysqldump_args, $O{database}, $O{table}) };
    if ( $@ ){
        die $@ if $MODE{non_interactive_write};
        return '';
    }
    
    $data_sql = Encode::decode_utf8($data_sql);

    return $data_sql;
}

=head2 get_schema_from_files

=cut

sub get_schema_from_files
{
    my (%O) = @_;

    my $instances = $O{conf}->{instances};
    my %skip_db = map {$_ => 1} map {@{$_->{skip_db}||[]}} values %$instances;
    my %skip_table = map {$_ => 1} map {@{$_->{skip_table}||[]}} values %$instances;

    my $schema = {};

    for my $path ( get_schema_files(%O) ) {
        ( my $db = dirname($path) ) =~ s!^\Q$O{main_dir}\E/!!;
        my $file = basename($path);

        my $table_nick = $file =~ s/\..*//r;
        next if $skip_db{$db} || $skip_table{"$db.$table_nick"};

        $schema->{$db}->{$file} = read_file($path, binmode => ':utf8');
        $schema->{$db}->{$file} =~ s/^(\s*CONSTRAINT\s*`)_+/$1/gm;
    } 
    
    return $schema;
}


=head2 get_schema_files

=cut

sub get_schema_files
{
    my (%O) = @_;
    my $dir = $O{main_dir};

    my @files;

    if ( $O{no_svn} ) {
        find(
            sub {
                if ( !/^\./ && $File::Find::name !~ /\.svn/ && -f $File::Find::name) {
                    push @files, $File::Find::name;
                } 
            },
           $dir
        ); 
    } else {
        @files = grep { -f } svn_files($dir);
    }

    return @files;
}



=head2 compare_schemata

    сравнивает два хеша со схемами БД

    schemata -- это можественное число от schema

=cut 

sub compare_schemata
{
    my %O = @_;
    my ( $db_schema, $files_schema, $main_dir, $opt) = 
    @O{qw/db          files          main_dir   opt/};

    my $created = dir_and_db_differ(
                dir => "$main_dir", 
                desc => "Каталог для схемы БД",
                action => 'create',
            );
    return if $MODE{write_files} && !$created;

    for my $database (sort keys %$db_schema){
        next if $opt->{database} && $database ne $opt->{database};
        if ( !exists $files_schema->{$database} ){
            my $created = dir_and_db_differ(
                dir => "$main_dir/$database", 
                desc => "Каталог для базы данных $database",
                action => 'create',
            );
            next if $MODE{write_files} && !$created;
        }

        # сравниваем записанные инстансы
        my $instance_from_db = $db_schema->{$database}->{instance} || '';
        my $instance_from_file = $files_schema->{$database}->{instance} || '';
        if ( $instance_from_db ne $instance_from_file ){
                file_and_db_differ(
                    file => "$main_dir/$database/instance", 
                    title => "Инстанс, в котором живет база $database",
                    sql_from_db => $instance_from_db, 
                    sql_from_file => $instance_from_file, 
                );
        }

        # сравниваем "create table..."
        for my $table (sort keys %{$db_schema->{$database}->{schema}} ){
            next if $opt->{table} && $table ne $opt->{table};
            my $db_create_sql = $db_schema->{$database}->{schema}->{$table} || '';
            my $file_create_sql = $files_schema->{$database}->{"$table.schema.sql"} || '';
            if ( $db_create_sql ne $file_create_sql ){
                file_and_db_differ(
                    file => "$main_dir/$database/$table.schema.sql", 
                    title => "Схема таблицы $database.$table",
                    sql_from_db => $db_create_sql, 
                    sql_from_file => $file_create_sql, 
                );
            }
        }

        # сравниваем "create procedure..."
        for my $procedure (sort keys %{$db_schema->{$database}->{procedure}} ){
            next if $opt->{procedure} && $procedure ne $opt->{procedure};
            my $db_create_sql = $db_schema->{$database}->{procedure}->{$procedure} || '';
            my $file_create_sql = $files_schema->{$database}->{"$procedure.procedure.sql"} || '';
            if ( $db_create_sql ne $file_create_sql ){
                file_and_db_differ(
                    file => "$main_dir/$database/$procedure.procedure.sql", 
                    title => "Процедура $database.$procedure",
                    sql_from_db => $db_create_sql, 
                    sql_from_file => $file_create_sql, 
                );
            }
        }

        # сравниваем "insert into..."
        for my $table (sort keys %{$db_schema->{$database}->{data}} ){
            next if $opt->{table} && $table ne $opt->{table};
            my $db_data_sql = $db_schema->{$database}->{data}->{$table} || '';
            my $file_data_sql = $files_schema->{$database}->{"$table.data.sql"} || '';
            if ( $db_data_sql ne $file_data_sql ){
                file_and_db_differ(
                    file => "$main_dir/$database/$table.data.sql", 
                    title => "Данные в таблице $database.$table",
                    sql_from_db => $db_data_sql, 
                    sql_from_file => $file_data_sql, 
                );
            }
        }

        # сравниваем "insert into..."
        for my $table (sort keys %{$db_schema->{$database}->{init_data}} ){
            next if $opt->{table} && $table ne $opt->{table};
            my ( $db_data_sql, $file_data_sql );
            if ( $MODE{compare_init_data} ) {
                $db_data_sql = $db_schema->{$database}->{init_data}->{$table} || '';
                $file_data_sql = $files_schema->{$database}->{"$table.init-data.sql"} || '';
            } else {
                $db_data_sql = $db_schema->{$database}->{init_data}->{$table} ? "(non-empty)\n" : "(empty)\n";
                $file_data_sql = $files_schema->{$database}->{"$table.init-data.sql"} ? "(non-empty)\n" : "(empty)\n";
            }

            if ( $db_data_sql ne $file_data_sql ){
                file_and_db_differ(
                    file => "$main_dir/$database/$table.init-data.sql", 
                    title => "Данные в таблице $database.$table",
                    sql_from_db => $db_data_sql, 
                    sql_from_file => $file_data_sql, 
                );
            }
        }

        # ищем лишние файлы:
        ## sql'ные...
        for my $file ( sort grep {/\.sql$/} keys %{$files_schema->{$database}} ){
            my ($subj) = ($file =~ /\.(schema|procedure|data|init-data)\.sql$/);
            $subj //= '';
            $subj = 'init_data' if $subj eq 'init-data';

            (my $table = $file) =~ s/\..*//;
            next if $opt->{table} && $table ne $opt->{table};
            if ( ! exists $db_schema->{$database}->{$subj}->{$table} ){
                file_and_db_differ(
                    file => "$main_dir/$database/$file", 
                    title => "Таблица $database.$table",
                    sql_from_db => '', 
                    sql_from_file => $files_schema->{$database}->{$file}, 
                );
            }
        }
        ## ... и все другие
        for my $file ( sort grep {!/^instance$/ && !/\.sql$/} keys %{$files_schema->{$database}} ){
            my ($table) = ($file =~ /^(.*).text$/);
            next if $opt->{table} && $table ne $opt->{table};
            if ( ! $table || ! exists $db_schema->{$database}->{schema}->{$table} ){
                file_and_db_differ(
                    file => "$main_dir/$database/$file", 
                    title => 'Лишний файл',
                    desc => 'Описания таблиц должны находиться в файлах <имя_таблицы>.text',
                    sql_from_db => '', 
                    sql_from_file => $files_schema->{$database}->{$file}, 
                );
            }
        }
    }

    # ищем лишние каталоги
    my %skip;
    if ($opt->{skip_dir}) {
        %skip = map {$_ => undef} @{$opt->{skip_dir}};
    }
    for my $database ( sort keys %$files_schema ){
        next if $opt->{database} && $database ne $opt->{database};
        next if exists $skip{$database};
        if ( ! exists $db_schema->{$database} ){
            dir_and_db_differ(
                dir => "$main_dir/$database", 
                desc => "Каталог для базы данных $database",
                action => 'remove',
            );
        }
    }

    return;
}


=head2 dir_and_db_differ

    Обрабатывает ситуацию "каталог $O{dir} (назначение -- $O{desc}) и БД не соответствуют друг другу"
    Что надо сделать с каталогом (добавить/удалить): $O{action} (create/remove)

=cut

sub dir_and_db_differ
{
    my %O = @_;

    return 1 if $O{action} eq 'create' && -d $O{dir} || $O{action} eq 'remove' && ! -d $O{dir} ; 

    my $diag_rus = $O{action} eq 'create' ? 'Нет такого каталога' : 'Лишний каталог';
    print "\n\n===$O{dir}\n$O{desc}\n$diag_rus\n";

    return 0 if !$MODE{write_files};

    my $update_dir = 1; 
    
    if ( $MODE{interactive_write} ){
        my $action_rus = $O{action} eq 'create' ? 'Создаем' : 'Удаляем';
        my $answer = Yandex::Interactive::prompt("$action_rus каталог? {y|n} ", {"$action_rus? (Please enter y|n): " => qr/^(y|n)$/i});
        $update_dir = 0 if $answer ne 'y';
    }

    return 0 if !$update_dir;

    if ( $O{action} eq 'remove' ){
        yash_system('svn', rm => $O{dir});
    } else {
        mkdir $O{dir};
        yash_system('svn', '-q', '--force', add => $O{dir});
    }

    return 1;
}


=head2 file_and_db_differ

    Обрабатывает ситуацию "файл $O{file} (назначение -- $O{desc}) и БД ($O{sql_from_db}) не соответствуют друг другу" 

=cut

# TODO, мечта: можно когда-нибудь по диффу в схеме таблицы выводить нужный alter
sub file_and_db_differ
{
    my %O = @_;

    my $title = $O{title};
    my $desc = $O{desc} || 'Файл и база данных не соответствуют друг другу';
    print "\n\n===$O{file}\n--- file\n+++ DB\n" 
          . diff(\$O{sql_from_file}, \$O{sql_from_db})
          . "\n----\n$O{file}\n$title\n$desc\n";

    return if !$MODE{write_files};

    if ( $O{file} =~ /\.init-data\.sql$/ && ! $MODE{compare_init_data} ) {
        die "Не буду записывать $O{file} без --compare-init-data";
    }

    my $update_file = 1; 
    
    my $action = $O{sql_from_db} ne '' ? 'update' : 'remove';
    my $action_rus = {update => 'Обновляем', remove => 'Удаляем'}->{$action};

    if ( $MODE{interactive_write} ){
        # Вот здесь спрашивать: обновляем файл? Или сгенерировать вам подходящий alter?
        my $answer = Yandex::Interactive::prompt("$action_rus файл $O{file}? {y|n} ", {"$action_rus? (Please enter y|n): " => qr/^(y|n)$/i});
        $update_file = 0 if $answer ne 'y';
    }

    return if !$update_file;

    if ($action eq 'remove'){
        yash_system('svn', '--force', rm => $O{file});
    } else {
        my $file_exists = -f $O{file};
        write_file($O{file}, {atomic => 1, binmode => ':utf8'}, $O{sql_from_db});
        yash_system('svn', '-q', add => $O{file}) if !$file_exists;
    }

    return;
}


=head2 generate_text_file (table)

    Генерирует файл для текстового описания таблицы
    Params:
      table - название таблицы, для которой сгенерировать файл.

=cut

sub generate_text_file
{
    my ($table, %O) = @_;

    my $db;
    # Вычисляем БД
    for my $instance (keys %{$O{conf}->{instances}}) {
        next if (! Yandex::DBTools::is_table_exists($instance, $table));
        $db = $instance;
    }
    # Проверяем существует ли таблица
    if (! $db) {
        print "Указанная таблица не существует\n\n";
        return;
    }
    my $filename = "$table.text";

    my $subdir = $db;
    # в $db может быть название вида ppc:shard; номер шарда можно просто отрезать, потому что
    # пока что структура каталогов одноуровневая
    $subdir =~ s/:.*//g;

    my $dir = "$O{main_dir}/$subdir/";

    my @svn_files = map {basename($_)} grep {! -d $_ } Yandex::Svn::svn_files($dir);
    my $is_existed_in_svn = scalar(grep {$_ eq $filename} @svn_files);

    if (-f $dir.$filename) {
        if ($is_existed_in_svn) {
            print "Описание для таблицы $table уже присутствует, пожалуйста, используйте уже существующий файл описания:\n$dir$filename\n";
        } else {
            print "Файл с описанием уже создан, но не добавлен в SVN\nВыполните команду svn add чтобы добавить файл в SVN\n\tsvn add $dir$filename\n";
        }
    } else {
        # подобного файла ещё нет, поэтому можно генерировать новый
        my $cols = get_all_sql($db, "SHOW COLUMNS FROM $table");

        #my $podcherk = $table; $podcherk =~ s/./\-/g; 
        (my $podcherk = $table) =~ s/./\-/g;
        my $header = "$table\n$podcherk\n\n";
        my $body   = "<Описание таблицы>\n\n";
        my $columns= "### Столбцы: ###\n\n";
        for my $column (@{$cols}) {
            $columns .= "$column->{Field}\n:   < >\n\n";
        }
        my $text = $header.$body.$columns;

        # Сохраняем файл на диске
        write_file($dir.$filename, {atomic => 1, binmode => ':utf8'}, $text);
        # Добавляем в SVN
        if (!$is_existed_in_svn) {
            yash_system('svn', '-q', add => $dir.$filename); 
        }

        print "Теперь можно открыть файл $dir$filename\n и добавить описание таблицы и полей\n";
    }
}
