package Yandex::DBSchema;

# $Id$

=head1 NAME
    
    DBSchema

=head1 DESCRIPTION

    Модуль для создания таблиц в соответствии с хранимой схемой БД (db_schema).
    Главная функция -- create_table_by_schema

=head1 FUNCTIONS

=cut

use strict;
use warnings;

use Yandex::DBTools;
use Yandex::HashUtils;

use File::Slurp;
use Data::Dumper;
use Params::Validate;

use base qw/Exporter/;
our @EXPORT = qw(
    create_table_by_schema
);

use utf8; 

our $DB_SCHEMA_ROOT; 

our $NO_DB_IN_TABLE_NAME;

=head2 get_create_table_sql

    на входе
        нормализованный хеш с полями db и table

    возвращаемое значение
        строка с sql-запросом для создания указанной таблицы

=cut
sub get_create_table_sql
{
    my (%table_desc) = @_;

    my $schema_file = "$DB_SCHEMA_ROOT/$table_desc{db}/$table_desc{table}.schema.sql";
    my $create_table_sql = read_file($schema_file, binmode => ":utf8");

    return $create_table_sql;
}


=head2 get_table_data_sqls

    на входе
        нормализованный хеш с полями db и table

    возвращаемое значение
        ссылка массив из sql-запросов для заполниния таблицы из .data файла

=cut
sub get_table_data_sqls
{
    my (%table_desc) = @_;

    my $data_file = "$DB_SCHEMA_ROOT/$table_desc{db}/$table_desc{table}.data.sql";
    my @sql = read_file($data_file, binmode => ":utf8");

    return \@sql;
}


=head2 get_table_text_desc

    на входе
        нормализованный хеш с полями db и table

    возвращаемое значение
        строка с текстовым описанием таблицы (файл .text)
        если файла нет - undef

=cut
sub get_table_text_desc
{
    my (%table_desc) = @_;

    my $text_file = "$DB_SCHEMA_ROOT/$table_desc{db}/$table_desc{table}.text";
    return undef unless -f $text_file;

    return scalar read_file($text_file, binmode => ":utf8");
}


=head2 parse_table_text_desc

    на входе
        строка с тектовым описанием таблицы (файл .text)

    на выходе 
        ссылка на хэш
            table_name => Str
            table_desc => Str
            tags? => [Str, ...]
            columns => [
                {
                name => Str
                text => Str
                fk? => {
                     table => Str
                     column => Str
                     },
                }, ...
            ]

=cut
sub parse_table_text_desc
{
    my ($text) = @_;
    die "Can't parse table desc from '$text'"
        unless $text =~ /^ (\w+) \s* \n
                        --+ \s* \n
                        \s* \n
                        (.*) \n
                        \s* \n
                        \s* \#\#\# \s* Столбцы :? \s* \#\#\# \s*
                        (.*)
                        $ /xsi;
    my $desc = { table_name => $1, table_desc => $2, columns => [] };
    my $columns_text = $3;
    if ($desc->{table_desc} =~ s/^\s*tags:\s*(.*)\n\s*//) {
        $desc->{tags} = [split /\s*,\s*/, $1];
    }
    $desc->{table_desc} =~ s/^\s+|\s+$//g;
    for my $column_text (split /\s*\n\s*\n\s*(?=\S+\s*\n:\s+)/, $columns_text) {
        if ($column_text =~ /^\s*(\S+)\s*\n:\s*(.*?)\s*$/s) {
            my $col_desc = {name => $1, text => $2};
            if ($col_desc->{text} =~ s/^FK\s*\(\s*(\w+)\.(\w+)\s*\)\s*//i) {
                $col_desc->{fk} = {table => $1, column => $2};
            }
            if ($col_desc->{text} =~ /^\s*(<\s*>|)\s*$/) {
                $col_desc->{text} = undef;
            }
            push @{$desc->{columns}}, $col_desc;
        } else {
            die "Can't parse column desc for $desc->{table_name}: $column_text";
        }
    }
    return $desc;
}


=head2 normalize_table_desc 

    на входе 
        ссылка на хеш с полями table и (необязательно) db

    результат
        in-place нормализует переданный хеш, т.е. приводит к виду: 
        в поле table -- только имя таблицы, 
        в db -- только имя базы

    возвращаемое значение 
        0 

    в случае неудачи (в table и db противоречивые данные) -- падает

=cut
sub normalize_table_desc 
{
    my ($desc) = @_;

    if( $desc->{table} =~ /\./ ){
        my ($db, $table) = split /\./, $desc->{table}, 2;
        die "can't determine db name" if exists $desc->{db} && $desc->{db} ne $db;
        $desc->{table} = $table;
        $desc->{db} = $db;
    } 
    return 0;
}


=head2 create_table_by_schema

    Параметры позиционные: 
      $dbh
      $table - строка с именем таблицы, которя должна быть создана. 
               С префиксом по БД: "ppc.campaigns", "ppcdict.yandex_offices", 
               или без него: "events". 
               Если БД не указана в префиксе -- берем из $dbh
          Особенность: под DBUnitTest без префикса работать не будет, 
          т.к. в UT имя базы примерно такое: "unit_tests_lena_san_20101126193021", 
          и соответствующего файла в db_schema, конечно, нет. 

    Параметры именованные:
      if_not_exists -- (необязательно) добавить ли в sql-запрос "if not exists", по умолчанию 0
      like  -- (необязательно) строка либо ссылка на хеш с описанием таблицы, 
               по образцу которой надо создать новую. 
               Строка должна иметь вид "table" или "db.table",
               в хеше должно быть поле table и может быть поле db, 
               смысл -- как у одноименных основных параметров
               если не указано db -- будет такой же, как у создаваемой таблицы
      temporary -- надо ли create TEMPORARY table
      engine -- изменить тип создаваемой таблицы (работает только если в схеме есть engine=)

    Результат: 
      создает таблицу в соответствующей базе

    Возвращаемое значение:
      0

    Если не удалось создать (не нашелся файл и т.п.) -- умирает         

    Примеры:
    create_table_by_schema(PPC, "ppc.banners"); # полезно для юнит-тестов
    create_table_by_schema(PPC, "my_table", like => "ppc.banners");
    create_table_by_schema(PPC, "my_table", like => {table => "banners", db => "ppc"});
    create_table_by_schema(PPCLOG, "logcmd_$table_date", like => "ppclog.logcmd_YYYYMMDD", if_not_exists => 1);
    create_table_by_schema(PPCORDSTAT, "direct_stat.bs_order_stat");

=cut
sub create_table_by_schema
{
    my ($dbh, $table, @named_params) = @_;
    validate(@named_params, { like => 0, if_not_exists => 0, temporary => 0, engine => 0 });
    my %opt = @named_params;

    my %dest_table = (
        table => $table
    );
    normalize_table_desc(\%dest_table);
    $dest_table{db} ||= get_one_field_sql($dbh, "select DATABASE()");
    die "dest_table undefined" unless $dest_table{db} && $dest_table{table};

    my %source_table;
    if( exists $opt{like} ){
        if ( ! ref $opt{like} ){
            $opt{like} = {table => $opt{like}}; 
        }
        %source_table = %{$opt{like}};
        normalize_table_desc(\%source_table);
        $source_table{db} ||= $dest_table{db};
        die "source_table undefined" unless $source_table{db} && $source_table{table};
    } else {
        %source_table = %dest_table;
    }

    my $sql = get_create_table_sql(%source_table);
    # имя создаваемой таблицы: "<db>.<table>", или просто "<table>", если взведен флаг $NO_DB_IN_TABLE_NAME (например, полезно для юнит-тестов)
    my $dest_table_str = $NO_DB_IN_TABLE_NAME ? "`$dest_table{table}`" : "`$dest_table{db}`.`$dest_table{table}`";
    $sql =~ s/^(\s*create\s+table\s+)`[^`]+`(\s*\()/$1$dest_table_str$2/i;

    if( $opt{if_not_exists} ){
        $sql =~ s/^\s*create\s+table\s+/CREATE TABLE IF NOT EXISTS /i;
    }

    if( $opt{temporary} ){
        $sql =~ s/^(\s*create\s+table\s+)/CREATE TEMPORARY TABLE /i;
    }

    if ($opt{engine}) {
        $sql =~ s/(\s*ENGINE\s*=\s*)(\w+)/$1$opt{engine}/ig;
    }

    do_sql($dbh, $sql);

    return 0;
}


=head2 my @databases = Yandex::DBSchema::get_databases();

    Получить имена баз

=cut
sub get_databases {
    opendir(my $dh, $DB_SCHEMA_ROOT) || die "Can't open '$DB_SCHEMA_ROOT': $!";
    my @databases = grep {/^[a-z0-9_]+$/} readdir($dh);
    closedir($dh);
    return @databases;
}


=head2 my @tables = Yandex::DBSchema::get_tables('ppc');

    Получить имена всех таблиц, существующих в указаной базе

=cut
sub get_tables {
    my ($db) = @_;
    opendir(my $dh, "$DB_SCHEMA_ROOT/$db") || die "Can't open '$DB_SCHEMA_ROOT/$db': $!";
    my @tables = map {/^(\w+)\.schema\.sql$/} readdir($dh);
    closedir($dh);
    return @tables;
}


1;
