package Yandex::DBShards;

=head1 NAME

    Yandex::DBShards - работа с шардами БД

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use feature qw/state/;

use Carp;
use POSIX qw/strftime/;
use List::MoreUtils qw/uniq any/;
use Time::HiRes qw/time/;
use Yandex::ListUtils qw/chunks nsort/;
use Yandex::Retry;
use Yandex::Log;
use Yandex::Hostname;

use Yandex::DBTools;

use parent qw/Exporter/;
our @EXPORT = qw/get_shard get_shard_multi
                 get_new_id get_new_id_multi
                 save_shard
                 delete_shard
                 sharded_chunks sharded_chunks_iter foreach_shard
                 foreach_shard_parallel
                 update_shard_chain
                 SHARD_IDS
                /;


# $IDS_LOG_PREFIX -- каталог, в который syslog будет складывать логи выданных id
# $IDS_LOG_FILE -- имя файла, в который (через syslog) будут записываться логи выданных id. Если не указан -- логгирования id-шников не будет.
our ($IDS_LOG_PREFIX, $IDS_LOG_FILE);

=head2 $MAX_ID_VALUE

    Максимально допустимое значение id, если начинают генерироваться значения больше указанного - croak
    Мотивация - между Директом и (БК|Модерацией) есть договорённость, что старшие 8 бит отводятся на ServiceID,
    а нижние 56 можно использовать под идентификаторы

=cut
our $MAX_ID_VALUE ||= (1 << 56) - 1;

=head2 %SHARDS_KEYS

    Хэш, определяющий, как по некоторым значениям узнавать номер шарда,
    в котором находятся нужные данные и как генерировать новые неповторяющиеся id.
    
    Ключ - название идентификатора (ClientID, bids_id...)
    Значение - ссылка на хэш с такими данными:
      table - таблица, в которой хранитятся данные о шардах или в которой генерируютя id
      type - тип ключа, возможные варианты:
            'int' - целое число, значение по-умолчанию
            'str' - строка без учёта регистра
            'bin' - бинарные данные (или строка с учётом регистра)
      shard_key - название столбца БД, в котором хранится номер шарда
      chain_key - в таблице хранится не номер шарда, а некий идентификатор,
                по которому нужно узнавать шард (получается цепочка)
      autoinc - допускаем ли мы генерацию id

=cut
our %SHARD_KEYS;

=head2 $SHARD_DB

    Имя базы (в терминах Yandex::DBTools), в которой хранятся все таблички

=cut
our $SHARD_DB;

=head2 $CACHE_EXPIRES ||= 10

    время в секундах, в течении которого мы верим закешированным значениям

=cut
our $CACHE_EXPIRES ||= 10;

=head2 $CACHE_MAX_SIZE

    максимальный размер кеша, измеряется в количестве ключей

=cut
our $CACHE_MAX_SIZE ||= 10_000;

=head2 $STRICT_SHARD_DBNAMES ||= 1

    Умирать ли в get_shard_dbnames, если не указан ключ.
    Если настройка установлена в 0 - молча возвращаем все шарды.

=cut
our $STRICT_SHARD_DBNAMES;
$STRICT_SHARD_DBNAMES = 1 if !defined $STRICT_SHARD_DBNAMES;

=head2 $FIRST_SHARD_ID ||= 1

    Номер умолчального шарда для get_first_shard_dbname

=cut
our $FIRST_SHARD_ID ||= 1;

=head2 $STRICT_SHARD_DBNAMES_KEYS ||= 0

    Умирать ли в get_shard_dbnames, если не удалось по указанному ключу определить шард.
        Если настройка неопределена или ложна (в понимании perl) - значение ключа, для корого шард не определился, будет проигнорировано
        Если истинна - функция умрет с сообщением о невозможности определения шарда "Can't determine shards for $key => $id"

=cut
our $STRICT_SHARD_DBNAMES_KEYS;

=head2 $CHUNK_SIZE ||= 10_000;

    На куски какого размера разбивать запросы в базу 

=cut
our $CHUNK_SIZE ||= 10_000;

=head2 $QUERIES_NUM_FOR_TRX ||= 2

    Количество запросов, начиная с которого get_new_id_multi начинает оборачивать запросы в транзакцию
    (для ускорения, особенно при использовании синхронной репликации типа Galera)

=cut
our $QUERIES_NUM_FOR_TRX = 2;

# кэш, формат такой:
# $CACHE{ClientID}->{$ClientID} = [$shard_id, expires];
# $CACHE{cid}->{$cid} = [$ClientID, $cache_time];
my %CACHE = ();
# суммарное количество ключей
my $CACHE_SIZE = 0;

=head2 get_shard(key, $val, | $chain_key)

    Получает ключ и одно значение, возвращает номер шарда или, если указан chain_key - соответствующее значение.
    Если информации о шарде или мэппинге нет - undef.

    Примеры:
    $shard_number = get_shard(ClientID => 23);
    $ClientID = get_shard(cid => 43, 'ClientID');

=cut
sub get_shard($$;$) {
    my ($key, $val, $chain_key) = @_;
    return undef if !defined $val;
    return get_shard_multi($key, [$val], $chain_key)->{$val};
}

=head2 get_shard_multi(key, [$val1, ...], | $chain_key)

    Получает ключ и массив значений, возвращает ссылку на хеш, 
    где ключи - значения val1, ..., а значения - номера шардов (или значения соотвествующих ключей, если указан chain_key).
    Для непривязанных значений возвращается undef.
    undef-ы в массиве пропускаются

=cut
sub get_shard_multi($$;$);
sub get_shard_multi($$;$) {
    my ($key, $vals, $chain_key) = @_;
    my $info = $SHARD_KEYS{$key};
    my $key_type = $info->{type} || 'int';
    my $field = $info->{shard_key} || $info->{chain_key};
    croak "No mapping for $key" if !$info || !$field;
    croak "Incorrect vals parameter" if ref($vals) ne 'ARRAY';

    my $cache = $CACHE{$key} ||= {};

    my %ret = map {$_ => undef} grep {defined} @$vals;
    my $time = time;

    # получаем закешированные значения
    for my $val (keys %ret) {
        my $norm_val = $key_type eq 'str' ? lc($val) : $val;
        my $c = $cache->{$norm_val};
        if ($c && $c->[1] + $CACHE_EXPIRES > $time) {
            $ret{$val} = $c->[0];
        }
    }

    # получаем недостающие значения из базы
    my @to_select = grep {!defined $ret{$_}} keys %ret;
    if (@to_select) {

        my %str_map;
        my $mod = '';
        if ($key_type eq 'int') {
            $mod = '__int';
        } elsif ($key_type eq 'str') {
            # сохраняем оригиналы
            push @{$str_map{lc($_)}}, $_ for @to_select;
            @to_select = uniq map {lc($_)} @to_select;
        }

        for my $chunk (chunks \@to_select, $CHUNK_SIZE) {
            my $sth = exec_sql($SHARD_DB, ["SELECT $key, $field FROM $info->{table}", WHERE => { "${key}$mod" => $chunk }]);
            while(my ($val, $ret_val) = $sth->fetchrow_array) {
                $val = lc($val) if $key_type eq 'str';
                $CACHE_SIZE += $cache->{$val} ? 0 : 1;
                $cache->{$val} = [$ret_val, $time];
                if ($key_type eq 'str') {
                    $ret{$_} = $ret_val for @{$str_map{$val}};
                } else {
                    $ret{$val} = $ret_val;
                }
            }
        }
    }

    if (defined $chain_key && $chain_key eq $field) {
         # нашли то, что надо
    } elsif ($info->{chain_key}) {
        # уходим в рекурсию
        my @next_vals = uniq grep {defined} values %ret;
        my $next_shards = get_shard_multi($info->{chain_key} => \@next_vals, $chain_key);
        for my $val (keys %ret) {
            $ret{$val} = $next_shards->{$ret{$val}} if defined $ret{$val};
        }
    } elsif (defined $chain_key && $chain_key ne 'shard') {
         # дошли до конца цепочки, но $chain_key не нашли
         croak "Chain key $chain_key is unreachable from $key";
    }
    
    clear_cache() if $CACHE_SIZE > $CACHE_MAX_SIZE;
    return \%ret;
}


=head2 get_sharded_vals($shard_key => $shard_val, $key_name)
    
    Аналог get_sharded_vals_multi для одного ключа
    Возвращает ссылку на массив из найденных значений

    my $uids = get_sharded_vals('ClientID' => 1234, 'uid');
    
=cut
sub get_sharded_vals($$$) {
    my ($shard_key, $shard_val, $key_name) = @_;
    return get_sharded_vals_multi($shard_key => [$shard_val], $key_name)->{$shard_val};
}


=head2 get_sharded_vals_multi($shard_key => $shard_vals, $key_name)
    
    Раскрутка метабазы в "обратную сторону" - по значениям shard_key => shard_vals, найти все 
    идентификаторы типа key_name, относящиеся к ним
    Возвращает ссылку на хэш:
      - ключи - значения shard_vals
      - значения - ссылки на массив в найденными данными

    Предполагается редкое использование этой функции, поэтому кеширования нет.

    my $clientid2uids = get_sharded_vals_multi('ClientID' => [1234,453], 'uid');
    
=cut
sub get_sharded_vals_multi($$$) {
    my ($shard_key, $shard_vals, $key_name) = @_;
    
    croak "Incorrect key name: $key_name" if !$SHARD_KEYS{$key_name};
    croak "Incorrect shard_vals - array expected" if ref $shard_vals ne 'ARRAY';
    croak "Key_name is equal to shard_key: $key_name, $shard_key" if $key_name eq $shard_key;
    croak "Search by shard is not supported" if $shard_key eq 'shard';
    
    # находим цепочку таблиц
    my ($cur, $prev_info) = ($key_name, $SHARD_KEYS{$key_name});
    my @sql_joins;
    while(1) {
        croak "Too long shard chain..." if @sql_joins > 5;
        my $info = $SHARD_KEYS{$cur} || croak "No shard info for $cur";
        if (!@sql_joins) {
            @sql_joins = ($info->{table});
        } else {
            push @sql_joins, "JOIN $info->{table} ON $info->{table}.$cur = $prev_info->{table}.$cur";
        }
        $prev_info = $info;
        if ($info->{shard_key}) {
            croak "Can't complate chain for $key_name, leaf shard info node reached";
        } elsif ($info->{chain_key} && $info->{chain_key} eq $shard_key) {
            last;            
        }
        $cur = $info->{chain_key} || croak "Incorrect shard info for $cur - no chain_key";
    }

    my $sth = exec_sql($SHARD_DB, ["
                           SELECT $prev_info->{table}.$shard_key, $SHARD_KEYS{$key_name}->{table}.$key_name",
                           FROM => @sql_joins,
                           WHERE => {"$prev_info->{table}.$shard_key" => $shard_vals}
                       ]);

    my %ret = map {$_ => []} @$shard_vals;
    while(my ($val, $key) = $sth->fetchrow_array) {
        push @{$ret{$val}}, $key;
    }

    return \%ret;
}


=head2 get_new_id(key, | chain_key, chain_val)

    Получить новый уникальный id по имени идентификатора
    Есть два варианта использования:
    get_new_id('bids_id') - получение нового id, для которого не хранится мэппинг
    get_new_id('cid', ClientID => 123) - получение нового id, сохранение мэппинга
    get_new_id('bid', cid => 125) - получение нового id, сохранение мэппинга, соотвествующего cid

=cut
sub get_new_id($;$$) {  
    my ($key, $chain_key, $chain_val) = @_;
    return get_new_id_multi($key, 1, $chain_key, $chain_val)->[0];
}

=head2 get_new_id_multi($key|type???, $cnt, | chain_key, chain_val)

    Получить нескольно новых идентификаторов за один раз:
    Параметры позиционные:
    - название идентификатора
    - необходимое количество значений
    и два опциональных параметра:
    - название ключа с мэппингом (или 'shard' для указания непосредственного шарда)
    - значение для мэппинга

    Возвращает ссылку на массив с новыми id

=cut
sub get_new_id_multi($$;$$) {  
    my ($key, $cnt, $chain_key, $chain_val) = @_;
    my $info = $SHARD_KEYS{$key};
    croak "Incorrect key '$key'" if !$info;
    croak "No autoinc for key '$key'" if !$info->{autoinc};
    croak "Incorrect count: '$cnt'" if !defined $cnt || $cnt !~ /^\d+$/;
    croak "Chain value is not defined, but chain key is defined" if defined $chain_key && !defined $chain_val;

    my $field;
    my $field_val;
    if ($field = $info->{shard_key}) {
         croak "Chain key is not defined" if !defined $chain_key;
         if ($chain_key eq 'shard') {
             $field_val = $chain_val;
         } else {
             $field_val = get_shard($chain_key => $chain_val);
         }
    } elsif ($field = $info->{chain_key}) {
         croak "Chain key is not defined" if !defined $chain_key;
         if ($chain_key eq $field) {
             $field_val = $chain_val;
         } else {
             $field_val = get_shard($chain_key => $chain_val, $field);
         }
    } elsif ($chain_key) {
        croak "No chain_key for $key, but '$chain_key' provided";
    }

    croak "Undefined value of $field" if $field && !defined $field_val;

    my @ids;
    my $insert_data = $field ? {$key => undef, $field => $field_val} : {$key => undef};
    if ($cnt >= $QUERIES_NUM_FOR_TRX) {
        do_in_transaction {
            for(1..$cnt) {
                push @ids, do_insert_into_table($SHARD_DB, $info->{table}, $insert_data);
                croak "Impossible situation: generated incorrect id '$ids[$#ids]'" unless $ids[$#ids];
                croak "Too big id generated '$ids[$#ids]'" if $ids[$#ids] > $MAX_ID_VALUE;
            }
        };
    } else {
        for(1..$cnt) {
            push @ids, do_insert_into_table($SHARD_DB, $info->{table}, $insert_data);
            croak "Impossible situation: generated incorrect id '$ids[$#ids]'" unless $ids[$#ids];
            croak "Too big id generated '$ids[$#ids]'" if $ids[$#ids] > $MAX_ID_VALUE;
        }        
    }

    _log_new_ids( ids => \@ids, key => $key, cnt => $cnt, insert_data => $insert_data ) if $IDS_LOG_FILE;

    return \@ids;
}


=head2 _log_new_ids

    Пишет (через syslog) лог выданных id.

    Ожидаются именованные параметры:

    _log_new_ids( ids => \@ids, key => $key, cnt => $cnt );

    
    Ожидается, что в переменных $IDS_LOG_PREFIX, $IDS_LOG_FILE указано, 
    как называть лог и в какой каталог его должен положить syslog.

    Если переменная $IDS_LOG_FILE не установлена -- молча ничего не делает.
    
    В лог добавляется reqid текущего запроса, вычисляется как Yandex::Trace::current_span_id()
    Если Yandex::Trace не подгружен/не определен текущий reqid -- запишется пустая строка


    В лог попадут примерно такие записи: 
    <134>1 2014-09-29T19:40:24+04:00 ppcdev1.yandex.ru PPC.dbshards_ids.log 7101 - - {"insert_data":{"vcard_id":null,"ClientID":"485569"},"ids":["8691697"],"reqid":2989659892634964358,"cnt":"1","key":"vcard_id"}
    <134>1 2014-09-29T19:40:24+04:00 ppcdev1.yandex.ru PPC.dbshards_ids.log 7101 - - {"insert_data":{"bid":null,"ClientID":"485569"},"ids":["612303947"],"reqid":2989659892634964358,"cnt":"1","key":"bid"}
    <134>1 2014-09-29T19:40:24+04:00 ppcdev1.yandex.ru PPC.dbshards_ids.log 7101 - - {"insert_data":{"pid":null,"ClientID":"485569"},"ids":["506695707"],"reqid":2989659892634964358,"cnt":"1","key":"pid"}
    <134>1 2014-09-29T19:40:24+04:00 ppcdev1.yandex.ru PPC.dbshards_ids.log 7101 - - {"insert_data":{"phid":null},"ids":["2928686977","2928686982","2928686987","2928686992","2928686997"],"reqid":2989659892634964358,"cnt":"5","key":"phid"}

=cut
sub _log_new_ids
{
    my (%O) = @_;

    return unless $IDS_LOG_FILE;

    $O{reqid} ||= eval{Yandex::Trace::current_span_id()} // '';
    $O{log_time} ||= strftime("%Y-%m-%d:%H:%M:%S", localtime);
    $O{host} ||= _hostname();

    state $log = Yandex::Log->new(
        use_syslog => 1, 
        no_log => 1, 
        no_date => 1,
        syslog_prefix => $IDS_LOG_PREFIX, 
        log_file_name => $IDS_LOG_FILE,
    );

    $log->out(\%O);

    return;
}

sub _hostname
{
    state $hostname;
    $hostname //= Yandex::Hostname::hostfqdn();
    return $hostname;
}

=head2 save_shard

    Сохранить(добавить или перезаписать) для одного или нескольких id мэппинг
    Параметры позиционные:
    - название идентификатора
    - значение идентификатора (или ссылка на массив со значениями)
    - название ключа с мэппингом
    - значение идентификатора мэппинга

    Примеры:
    save_shard(bid => $bid, cid => $cid);
    save_shard(bid => \@bids, ClientID => $ClientID);
    save_shard(ClientID => $ClientID, shard => $shard_number);

=cut
sub save_shard($$$$) {
    my ($key, $vals, $chain_key, $chain_val) = @_;

    my $info = $SHARD_KEYS{$key};
    croak "Incorrect key '$key'" if !$info;
    my $key_type = $info->{type} || 'int';

    $vals = [$vals] if !ref $vals;
    $vals = [map {lc} @$vals] if $key_type eq 'str';
    
    croak "Chain key is not defined" if !defined $chain_key;
    croak "Chain val is not defined" if !defined $chain_val || ref $chain_val;
    croak "$key cannot be zero" if $info->{not_zero} && any { !$_ } @$vals;

    # название поля в таблице мэппинга
    my $field;
    # значение для записи в $field
    my $field_val;
    if ($field = $info->{shard_key}) {
         if ($chain_key eq 'shard') {
             $field_val = $chain_val;
         } else {
             $field_val = get_shard($chain_key, $chain_val);
         }
    } elsif ($field = $info->{chain_key}) {
         if ($chain_key eq $field) {
             $field_val = $chain_val;
         } else {
             $field_val = get_shard($chain_key, $chain_val, $field);
         }
    } else {
        croak "Neither shard_key nor chain_key for $key";
    }
    croak "Undefined $field for $chain_key=$chain_val" if !defined $field_val;
    croak "$field cannot be zero" if $SHARD_KEYS{$field}->{not_zero} && !$field_val;

    do_mass_insert_sql($SHARD_DB, "INSERT INTO $info->{table} ($key, $field)
                                    VALUES %s
                                    ON DUPLICATE KEY UPDATE $field = VALUES($field)
                            ", [map {[$_, $field_val]} @$vals],
                            {max_row_for_insert => $CHUNK_SIZE,},
        );

    # подчищаем кэш
    delete @{$CACHE{$key}}{@$vals};
}

=head2

    Удаление шарда.
    удаляются все записи из шарда, соответствующего ключу (key), с указанными значениями (vals).
    OPTIONS:
        recursive - удалять записи рекурсивно (если применимо)

=cut
sub delete_shard {
    my ($key, $vals, %OPT) = @_;

    my $info = $SHARD_KEYS{$key};
    croak "Incorrect key '$key'" if !$info;

    $vals = [$vals] if !ref $vals;

    if ($info->{shard_key} && $OPT{recursive}) { # то есть  СlientID и флаг, что удалять надо принудительно
        my @fields = grep {defined$_->{chain_key} && $_->{chain_key} eq $key} values(%SHARD_KEYS);
        do_delete_from_table($SHARD_DB, $_->{table}, where => {$key => $vals}) foreach (@fields);
    } elsif ($info->{chain_key} || $info->{shard_key}) {
        # ничего не делать
    } else {
        croak "Neither shard_key nor chain_key for $key";
    }

    for my $chunk (chunks $vals, $CHUNK_SIZE) {
        do_delete_from_table($SHARD_DB, $info->{table}, where => {$key => $chunk});
    }

    # подчищаем кэш
    delete @{$CACHE{$key}}{@$vals};
}

=head2 clear_cache

    безусловная очистка кеша
    
=cut
sub clear_cache {
    $CACHE_SIZE = 0;
    %CACHE = ();
}

=head2 get_shard_dbnames(prefix, suffix, $key => $val|$vals)

    По префиксу и, возможно, суффиксу названия базы, и по ключу / значениям получить 
    структуру, содержащую имена всех подходящих шардов, для передачи в DBTools

    Функция предназначена для использования в Settings, для создания удобных обёрток
    Пример:
    *PPC_HEAVY = sub {Yandex::DBShards::get_shard_dbnames('ppc', 'heavy', @_)};
    do_sql(PPC_HEAVY(OrderID => [1,2]));

    Параметры позиционные:
    - префикс названия БД
    - суффикс, для возможной специализации реплики
    - ключ, по которому определяется набор шардов
    - значение или ссылка на массив значений ключа

    Ключ shard имеет специальное значение, он позволяет указать шарды напрямую.
    Если к ключу shard передать значение 'all' - будут выбраны все шарды
    Важно: результаты запроса с shard => "all" кешируются (%_db_childs_cache); 
    если во время жизни процесса начинаем использовать другой конфиг-файл или в том же конфиг-файле меняется количество шардов 
    -- список всех шардов не поменяется до вызова clear_db_childs_cache.

    Возвращаемая структура - ссылка на хэш, ключи:
    - dbnames - массив из полных названий БД
    - key - переданный ключ
    - val - переданное значение (или ссылка на массив значений)

=cut
{
    my %_db_childs_cache;
    sub _cached_get_db_childs($) {
        return $_db_childs_cache{$_[0]} ||= get_db_childs($_[0]);
    }
    sub clear_db_childs_cache(){
        %_db_childs_cache = ();
    }
}
sub get_shard_dbnames {
    my ($dbname_prefix, $dbname_suffix, $key, $vals_param) = @_;
    my @vals = ref $vals_param ? @$vals_param : ($vals_param);
    my @shards;
    my %BY_SHARD;
    if (!defined $key) {
        if ($STRICT_SHARD_DBNAMES) {
            croak "No shard key defined in strict mode";
        } else {
            # ключ не указан - выбираем все шарды (наверное нужно будет потом умирать в таком случае)
            # если база нешардированная - выбираем основное имя
            # шарды у нас только числовые
            @shards = map {/^\Q$dbname_prefix\E:(\d+)$/ ? $1 : ()} @{_cached_get_db_childs($dbname_prefix)};
            @shards = ('') if !@shards;
        }
    } elsif ($key && $key eq 'shard') {
        # специальный случай - ключ shard
        if (@vals == 1 && $vals[0] eq 'all') {
            @shards = map {/^\Q$dbname_prefix\E:(\d+)$/ ? $1 : ()} @{_cached_get_db_childs($dbname_prefix)};
        } else {
            @shards = uniq @vals;
            my %known_shards = map {/^\Q$dbname_prefix\E:(\d+)$/ ? ($1 => 1) : ()} @{_cached_get_db_childs($dbname_prefix)};
            if (my @invalid_shards = grep {!$known_shards{$_}} @shards) {
                croak "Unknown shards: ".join(',', @invalid_shards);
            }
        }
    } elsif ($key) {
        # указан какой-то ключ для определения шадров
        my $shards_hash = get_shard_multi($key, \@vals);
        while(my ($id, $shard) = each %$shards_hash) {
            if (defined $shard) {
                push @{$BY_SHARD{$shard}}, $id;
            } else {
                if ($STRICT_SHARD_DBNAMES_KEYS) {
                    croak "Can't determine shards for $key => $id";
                } else {
                    # молча игнорируем
                    next;
                }
            }
        }
        @shards = keys %BY_SHARD;
    }
    my $mk_dbname = sub {return join(':', grep {defined $_ && $_ ne ''} $dbname_prefix, @_, $dbname_suffix)};
    # странного вида условие из-за того, что sort делает сравнение даже для массива из одной пустой строки
    if (@shards > 1) {
        # сортировка как минимум нужна для set_slave_db
        @shards = nsort @shards;
    }
    my @dbnames = map {$mk_dbname->($_)} @shards;
    my $ret = {
        dbnames => \@dbnames,
        key => $key, val => $vals_param,
        dbname_vals => {map {$mk_dbname->($_) => $BY_SHARD{$_}} @shards},
    };

    return $ret;
}

=head2 get_first_shard_dbname

    По префиксу и, возможно, суффиксу названия базы получить структуру,
        содержащую имя умолчального шарда, для передачи в DBTools

    Функция предназначена для использования в Settings, в функциях-обёртках для
        обеспечения совместимости со старым, нешардированным кодом.
    
    Пример:
    *PPC = sub {
        if (!$Yandex::DBShards::STRICT_SHARD_DBNAMES && !@_) {
            Yandex::DBShards::get_shard_dbnames('ppc', '', @_);
        } else {
            Yandex::DBShards::get_first_shard_dbname('ppc', '');
        }
    };
    do_sql(PPC, ...);
    do_sql(PPC(shard => 1), ...);

    Параметры позиционные:
    - префикс названия БД
    - суффикс, для возможной специализации реплики

    Возвращаемая структура - аналогично get_shard_dbnames

=cut
sub get_first_shard_dbname {
    my ($dbname_prefix, $dbname_suffix) = @_;
    if ($STRICT_SHARD_DBNAMES) {
        croak "Masking database sharding is not allowed in strict mode";
    } else {
        my @shards = map {/^\Q$dbname_prefix\E:(\d+)$/ ? $1 : ()} @{_cached_get_db_childs($dbname_prefix)};
        my $dbname = join(':', grep {defined $_ && $_ ne ''} $dbname_prefix, (@shards ? $FIRST_SHARD_ID : ''), $dbname_suffix);
        return {
            dbnames => [$dbname],
            dbname_vals => {$dbname => undef},
            key => undef, val => undef,
        };
    }
}

=head2 SHARD_IDS

    Функция для передачи в sql-запросы нового стиля DBTools, возвращает id, соответствующие текущему шарду.
    Например, если cid=1 - в первом шарде, а cid=2 и cid=3 во втором:
    get_all_sql(PPC(cid => [1,2,3]), ["SELECT * FROM campaigns", WHERE => {cid => SHARD_IDS}])
    преобразуется в склейку результатов двух запросов:
    get_all_sql(PPC(shard=>1), ["SELECT * FROM campaigns", WHERE => {cid => [1]}])
    get_all_sql(PPC(shard=>2), ["SELECT * FROM campaigns", WHERE => {cid => [2,3]}])

=cut
sub SHARD_IDS
{
    return sub { $_[0]->{vals} };
}

=head2 sharded_chunks $key => \@data | $id, chunk_size => $n, by => $key2;

    for my $chunk (sharded_chunks cid => $cids, chunk_size => 1_000) {
        my $shard = $chunk->{shard};
        my $_cids = $chunk->{cid};
        do_sql(PPC(shard => $shard), ["insert into table ... $_cids"]);
    }

В качестве key должна быть либо строка из SHARD_KEYS, либо 'shard' - в этом случае
перебираются указанные шарды, может быть полезно в foreach_shard_parallel.

В качестве массива с данными может быть массив скаляров(непосредственно id-шников)
или массив хэшей, содержащих id в одном из ключей

Разбить заданый массив на куски нужной длины, чтобы каждый кусок содержал
в себе данные только из одного шарда
Таким образом, каждый запрос можно выполнить в каждом шарде по очереди

Необязательные опции:
    chunk_size - максимальный размер получаемых массивов, при если 0 - разбивка делается только по шардам, 
        в пределах одного шарда чанк будет максимальной длины
    by - если передаётся массив хэшей - может быть
        - строка - определяет название ключа, содержащего id (по-умолчанию $key)
        - ссылка на функцию, вычисляющую id по $_ (текущему элементу @data) или по первому аргументу $_[0]
        если полученный id undef - будет выброшено исключение
    with_undef_shard - по умолчанию 0 - нужно ли генерировать чанки с данными,
        для которых не нашлось привязок к шардам. По-умолчанию такие данные просто выбрасываются

Deprecated: если указано три аргумента - последний считается опцией chunk_size

=cut

sub sharded_chunks(@)
{
    my ($key, $data) = (shift, shift);
    my %O = @_ == 1 ? (chunk_size => $_[0]) : @_;

    # разбор параметров
    croak "Key is not defined" if !defined $key;
    croak "Key $key is not sharded" if !exists $SHARD_KEYS{$key} && $key ne 'shard';

    $data = [$data] if defined $data && ref($data) ne 'ARRAY';
    croak "Mixed data type" if defined $data && uniq(map {ref} @$data) > 1;

    my $chunk_size = delete($O{chunk_size}) || 0;
    croak "Incorrect chunk size: $chunk_size" if $chunk_size !~ /^\d+$/;
    
    my $data_type = defined $data && @$data ? ref $data->[0] : '';
    croak "Incorrect data type $data_type for key=shard" if $key eq 'shard' && $data_type ne '';

    my $by = delete $O{by};
    croak "Incorrect option 'by' for scalars list" if defined $by && @$data && $data_type eq '';
    croak "Incorrect option 'by' for shards list" if defined $by && $key eq 'shard';
    if (!ref $by) {
        croak "Unsupported data type $data_type" if $data_type ne '' && $data_type ne 'HASH';
    }
    $by ||= $key;

    my $with_undef_shard = delete $O{with_undef_shard};

    croak "Unknown options: ".join(', ', keys %O) if %O;

    return () if !defined $data || !@$data;
    if ($key eq 'shard') {
        return map {+{shard => $_}} uniq @$data;
    }
    
    # получаем массивы данных из $data, сгруппированные по шардам, ключ -1 означает, что шард неопределён
    my %shard2data;
    my $sorter = ($SHARD_KEYS{$key}->{type}||'int') eq 'int' ? \&nsort : sub { sort(@_) };
    if ($data_type eq 'HASH' || ref $by eq 'CODE') {
        # группируем по id { id1 => [d1, d2], id2 => [d3, d4] }
        my %id2data;
        for (@$data) {
            my $id = ref $by eq 'CODE' ? $by->($_) : $_->{$by};
            croak "Undefined $key, by=$by" if !defined $id;
            push @{$id2data{$id}}, $_;
        }
        my $id2shard = get_shard_multi($key => [keys %id2data]);

        # группируем по шардам { 1 => [d1, d2], 2 => [d3, d4] }
        for my $id ($sorter->(keys %$id2shard)) {
            my $shard = $id2shard->{$id} // -1;
            push @{$shard2data{$shard}}, @{$id2data{$id}};
        }
    } else {
        my $id2shard = get_shard_multi($key => $data);
        # группируем по шардам { 1 => [id1, id2], 2 => [id3, id4] }
        for my $id ($sorter->(keys %$id2shard)) {
            my $shard = $id2shard->{$id} // -1;
            push @{$shard2data{$shard}}, $id;
        }
    }

    # режем на чанки
    my @ret;
    for my $shard (nsort keys %shard2data) {
        next if ($shard == -1 || $shard == 0) && !$with_undef_shard;
        my @chunks = $chunk_size ? chunks($shard2data{$shard}, $chunk_size) : $shard2data{$shard};
        for my $chunk (@chunks) {
            push @ret, {
                $key  => $chunk,
                shard => ($shard == -1 ? undef : $shard),
            };
        }
    }
    return @ret;
}


=head2 foreach_shard $key => \@data, %options, sub {};

    foreach_shard cid => $cids, sub {
        my ($shard, $cids_chunk) = @_;
        do_sql(PPC(shard => $shard), ["insert into table ... $cids_chunk"]);
    };

Разбить заданый массив на куски, аналогично sharded_chunks,
для каждого чанка вызвать колбэк с двумя параметрами - shard и ссылка на массив с id

%options содержит опции для sharded_chunks

=cut
sub foreach_shard(@)
{
    croak "Incorrect parameters num" if @_ < 3 || scalar(@_) % 2 != 1;

    my ($key, $data) = (shift, shift);
    my $code = pop;
    my %O = @_;

    for my $c (sharded_chunks($key => $data, %O)) {
        $code->($c->{shard}, $c->{$key});
    }
}


=head2 sharded_chunks_iter($key => $data, %options)

    my $iter = sharded_chunks_iter(cid => $cids, chunk_size => 100)
    while (my ($shard, $c) = $iter->()) {
        get_all_sql(PPC(shard => $shard), ... where => { cid => $c })
    }

Аналог sharded_chunks, но на одну строчку короче

=cut

sub sharded_chunks_iter(@)
{
    my ($key) = @_;
    my @chunks = sharded_chunks(@_);

    my $i = 0;
    return sub {
        my $next = $chunks[$i];
        return unless $next;
        $i++;
        return ($next->{shard}, $next->{$key});
    };
}

=head2 clear_all_autoinc_tables
    
    очистка таблиц, используемых только для генерации id
    именованные параметры:
     - sleep_coef - паузы между уделениями данных

=cut
sub clear_all_autoinc_tables {
    my %O = @_;

    my $sleep_coef = delete($O{sleep_coef}) || 0;
    croak "Unknown options: ".join(', ', keys %O) if %O;

    for my $key (sort keys %SHARD_KEYS) {
        my $info = $SHARD_KEYS{$key};
        next if exists $info->{shard_key} || exists $info->{chain_key};
        clear_autoinc_table($key, sleep_coef => $sleep_coef);
    }
}

=head2 clear_autoinc_table($key, sleep_coef => 0)

    очистка одной таблицы, используемой только для генерации id

=cut
sub clear_autoinc_table {
    my ($key, %O) = @_;

    croak "Undefined key" if !defined $key;
    my $sleep_coef = delete($O{sleep_coef}) || 0;
    croak "Unknown options: ".join(', ', keys %O) if %O;

    my $info = $SHARD_KEYS{$key};
    croak "Unknown key $key" if !defined $info;
    croak "Can't clear table for $key: not just increment table" if  exists $info->{shard_key} || exists $info->{chain_key};
    my @cols = @{get_one_column_sql($SHARD_DB, "SHOW COLUMNS FROM $info->{table}")};
    croak "Can't clear table for $key: more than one column in table" if @cols > 1;
    croak "Can't clear table for $key: incorrect column $cols[0]" if $cols[0] ne $key;
    croak "Incorrect CHUNK_SIZE" if $CHUNK_SIZE < 1;

    my ($min, $max) = get_one_line_array_sql($SHARD_DB, "select min($key), max($key) from $info->{table}");
    return unless defined $min;
    for(my $bound = $min + $CHUNK_SIZE; $bound < $max - $CHUNK_SIZE; $bound += $CHUNK_SIZE) {
        relaxed times => $sleep_coef, sub {
            do_sql($SHARD_DB, "DELETE FROM $info->{table} WHERE $key < ?", $bound);
        };
    }
}

=head2 update_shard_chain ($chain_key, $old_value, $new_value)
    Пробегает все метабазы, и где chain_key равен указанному chain_key - обновляет поле со старого значения на новое.
    Основное использование - объединение клиентов.

    Пример:
        update_shard_chain('ClientID', $client_id, $main_client_id);

=cut
sub update_shard_chain {
    my ($chain_key, $old_value, $new_value) = @_;

    croak "Incorrect key '$chain_key'" if !exists($SHARD_KEYS{$chain_key}); 
    croak "$chain_key=$new_value and $chain_key=$old_value must be in the same shard" if (get_shard($chain_key=>$old_value) != get_shard($chain_key=>$new_value));
    foreach my $info (values(%SHARD_KEYS)) {
        next if !($info->{chain_key} && $info->{chain_key} eq $chain_key);

        do_update_table($SHARD_DB, $info->{table}, {$info->{chain_key} => $new_value}, 
                        where=>{$info->{chain_key} => $old_value});
    }
}

=head2 foreach_shard_parallel($key => \@data | $id, %options, $code);

    foreach_shard_parallel(OrderID => [], sub {

        my ($shard, $order_ids) = @_;

        die if $error;
    });

    Разбить заданый массив на куски нужной длины(chunk_size), чтобы каждый кусок содержал
    в себе id только из одного шарда. И для каждого куска выполнить $code (выполнение будет идти в отдельном процессе).
    
    Параметры:
        $key - ключ для определения номера шарда для данных @data

        %options - (необязательный параметр) - аналогично sharded_chunks
            
        $code - ссылка на функцию для обработки кусков @ids, вызывается с двумя параметрами ($shard, $ids)
            $shard - номер шарда, $ids -список id для обработки
        $code будет выполняться в отдельном дочернем процессе.
        Число одновременно запущенных дочерних процессов будет не больше числа шардов в @ids.
        Если chunk_size == 0 то для одного шарда в один момент времени будет запущен один процесс.
        
    Результат:
        {shard_id => [is_success1, is_success2, is_success3 ....]}
        
        is_successN - определяется для каждого вызова $code.
        Успешное выполнение функции (без установки $@, без die) считается успешным выполнением.
        "undef" шард в хеше с результатом будет представлен ключом с пустой строкой ('')

=cut

sub foreach_shard_parallel {

    my ($key, $data) = (shift, shift);
    require Parallel::ForkManager;

    my $code = pop;
    my %options = @_;
    
    # {shard => [is_sucess_chunk1, is_sucess_chunk2]}
    my %result;
    {  
        my %childs;
        local $SIG{INT} = sub {
            kill 9 => keys %childs; 
            exit 1;        
        };
        local $SIG{TERM} = sub {
            kill 9 => keys %childs; 
            exit 1;        
        };
        
        my @chunks = sharded_chunks($key => $data, %options);
        my %shards;
        $_->{serial} = ++$shards{$_->{shard} // ''} for @chunks;
        
        my $pm = new Parallel::ForkManager(scalar keys %shards);
        $pm->run_on_finish(sub {
            my ($pid, $exit, $shard) = @_;
            push @{$result{$shard // ''}}, defined $exit && $exit == 0 ? 1 : 0;
        });
        
        for my $chunk (sort {$a->{serial} <=> $b->{serial}} @chunks) {
            my $pid = $pm->start($chunk->{shard});
            if ($pid) {
                $childs{$pid} = 1;
                next;
            }
            
            $SIG{INT} = $SIG{TERM} = 'DEFAULT';
            my $shard_name = defined $chunk->{shard} ? $chunk->{shard} : 'unknown';
            $0 .= " SHARD - $shard_name";
            eval {
                $code->($chunk->{shard}, $chunk->{$key});                
            };
            $pm->finish($@ ? 1 : 0);
        }
        
        $pm->wait_all_children;
    }
        
    return \%result;
}

1;


