#######################################################################
#
#  Direct.Yandex.ru
#
#  Yandex::DBTools
#  tools for database
#
#  authors:
#    Elena Bolshakova <lena-san@yandex-team.ru>
#    Georgy Ivanov <gerich@yandex-team.ru>
#    Maxim Kuzmin <kuzmin@yandex-team.ru>
#    Sergey Mudrik <msa@yandex-team.ru>
#    Sergey Zhuravlev <zhur@yandex-team.ru>
#
#  $Id$
#
#  (c) 2004 - 2010 Yandex
#
#######################################################################

=head1 NAME
 
 Yandex::DBTools

=head1 SYNOPSIS

  do_insert_into_table(PPC, 'users_options', {uid => 3882433, statusHidden => 'Yes'}, on_duplicate_key_update => 1, key => 'uid');

  do_update_table(PPC, 'vcards', {geo_id => $geo_id, LastChange__dont_quote => 'NOW()'}
                               , where => {cid => $cid, vcard_id => $vcard_id});

  do_delete_from_table(PPC, 'vcards', where => {vcard_id => $orphan_vcards_id});

  do_mass_insert_sql(PPC, 'insert into tbl (key1, key2) values %s'
                        , [[$val1, $val2], [111, 222], [333, 444]]
                        , { sleep => 2,                  # add sleep between inserts
                            max_row_for_insert => 10000, # redefine max row for one insert
                            dont_quote => 1,             # dont quote values
                          }
                    );

  $user_info_hashref = get_one_line_sql(PPC, "select * from users where uid = ?", 3882433);
  $user_info_hashref = get_one_line_sql(PPC, ["select * from users", where => {uid => 3882433}]);

  ($login, $fio) = get_one_line_array_sql(PPC, ["select login, FIO from users", where => {uid => 3882433}]);

  $camps = get_all_sql(PPC, ["select * from campaigns", where => {cid => sub {return [1..100]}}]);

  $camps = get_all_sql(PPC(cid => \@cids), ["select * from campaigns", where => {cid => SHARD_IDS}]);

  $uid = get_one_field_sql(PPC, "select uid from campaigns where cid = ?", $cid);

  $all_uids_arrayref = get_one_column_sql(PPC, ["select uid from users", where => {statusHidden => 'No'}]);

  $all_user_camps = get_all_sql(PPC, ["select * from campaigns", where => {uid => 3882433, statusEmpty => 'No'}]);

  $camps_hashref = get_hashes_hash_sql($dbh, "select cid, name from camps where name = ?", $name);

  do_sql(PPC, "insert into table (f1, f2) values (?, ?)", $f1, $f2);
  do_sql(PPC, ["update table set field_one = ?", where => {id => 1234}], $field_one);

  $quoted_login = sql_quote($login);
  $quoted_field_name = sql_quote_identifier($fieldname);

  $sql_case = sql_case($fieldname, {$key1 => $value1, ...}, | default => XXX | default__dont_quote => XXX);

  my $lock = sql_lock_guard($dbh, $lock_name, $timeout);

  # better use sql_lock_guard instead
  get_lock_sql(PPC, $Settings::LOCK_NAME, 60);
  release_lock_sql(PPC, $Settings::LOCK_NAME);

=head1 DESCRIPTION

tools for database

В качестве первого параметра($db) может приходить
  - объект($dbh) - потомок DBI::db
  - строка($dbname) - название БД
  - ссылка на хэш из Yandex::DBShards {dbnames => [$dbname1, ...], dbname_vals => {$dbname1 => ..., ...}}

=head1 FUNCTIONS

=cut

package Yandex::DBTools;

require UNIVERSAL;

use parent qw/Exporter/;
our @EXPORT = qw(
    connect_db
    disconnect_db
    disconnect_all
    get_dbh
    get_db_config get_db_childs get_shards_numbers

    set_slave_db
    wait_master

    exec_sql
    do_sql
    do_insert_into_table
    do_replace_into_table
    do_update_table
    do_mass_update_sql
    do_delete_from_table
    do_mass_insert_sql
    do_insert_select_sql
    get_one_line_sql
    get_one_line_array_sql
    get_one_field_sql
    get_one_column_sql
    get_all_sql
    get_hashes_hash_sql
    get_hash_sql

    sql_quote
    sql_quote_identifier
    sql_quote_like_pattern
    sql_fields
    sql_case
    sql_condition

    sql_lock_guard
    get_lock_sql
    release_lock_sql

    is_table_exists
    get_table_partitions
    dbnames

    select_found_rows

    do_in_transaction
);

use warnings;
use strict;

use Carp qw/carp croak/;
use Time::HiRes;
use Data::Dumper;
use List::Util qw/first sum max/;
use List::MoreUtils qw/all any none uniq/;
use Params::Validate qw/:all/;

use DBI;

use Log::Any;

use Yandex::Runtime;
use Yandex::Log;
use Yandex::HashUtils;
use Yandex::Trace;
use Yandex::LiveFile;

# пауза между попытками выполнения запроса при deadlock-е
my @DEADLOCK_PAUSE = (1, 5);

# пауза между попытками переустановки соединения при его потере во время запроса
our $RECONNECT_PAUSE ||= [];


=head2 $TRACE_COMMENT = bool

    Добавлять ли в запрос комментарий с информацией о происхождении запроса

=cut
our $TRACE_COMMENT;

=head2 $Yandex::DBTools::TRACE_COMMENT_VARS

    Параметры, записываемые в _add_trace_comment в формате <KEY1>=<VALUE1>:<KEY2=VALUE2>...

=cut
our $TRACE_COMMENT_VARS //= sub {return {};};


=head2 $Yandex::DBTools::CONFIG_FILE

    Имя файла с конфигом баз данных
    Нужно обязательно указывать либо $CONFIG_FILE, либо %DB_CONFIG

    Для разбора файла используется сериализатор, соответствующий расширению,
    допустимые расширения - .yaml или .json

=cut
our $CONFIG_FILE;

=head2 $Yandex::DBTools::CHECK_HOST_CHANGES

    Проверять ли при каждом soft_reconnect_db что хост БД не изменился с момента подключения

=cut

our $CHECK_HOST_CHANGES;

=head2 %Yandex::DBTools::DB_CONFIG

    Хэш с параметрами соединений с БД.
    Поддерживается иерархическое наследование атрибутов.
    Доступны только листья графа.

    Если для pass указан хэш с ключом 'file' - пароль считываем из указанного файла.
    В файле допускаются строки с комментариями, начинающиеся с '#',
    Начальные и конечные пробельные символы обрезаем.

    Кроме user, можно еще указать хеш extra_users -- его умеет обрабатывать функция get_db_config

    Пример:
    %DB_CONFIG = (
        utf8 => 1,
        port => 3306,
        user => 'root',
        extra_users => {
            my_user => {
                pass => my_pass, 
            },
            my_user_2 =>{
                pass => {file => '/tmp/pass'},
            },
        },
        connect_timeout => 4,
        CHILDS => {
            ppc => {
                host => 'ppcdata1.yandex.ru',
                CHILDS => {
                    '_' => {},
                    slave => {
                        utf8 => 0,
                        ReadOnly => 1,
                        pass => {file => '/tmp/pass'},
                    },
                }
            }
        }
    );
    Создаётся два конфига:
    'ppc'       => {utf8 => 1, port => 3306, user => 'root',
                    connect_timeout => 4, host => 'ppcdata1.yandex.ru'}
    'ppc:slave' => {utf8 => 0, port => 3306, user => 'root',
                    connect_timeout => 4, host => 'ppcdata1.yandex.ru', ReadOnly => 1, pass => read_file('/tmp/pass')}

=cut
our %DB_CONFIG;

=head2 $Yandex::DBTools::QUERIES_LOG

    Если эта переменная определена - все запросы записываются в лог файл с таким именем

=cut
our $QUERIES_LOG;

=head2 $Yandex::DBTools::LOG_WARNINGS_CATEGORY

Log::Any-категория для логирования ворнингов mysql

=cut

our $LOG_WARNINGS_CATEGORY = __PACKAGE__ . '::WARNINGS';


=head2 $Yandex::DBTools::LOG_WARNINGS_COUNT

Ограничение количества ворнингов на оператор

=cut

our $LOG_WARNINGS_COUNT = 1;
our $LOG_WARNINGS_STACKTRACE = 1;

our $MAX_ROW_FOR_MASS_INSERT = 1000;
our $MAX_ROW_FOR_MASS_UPDATE ||= 1000;
our $MAX_ROW_FOR_SELECT = 10_000;
our $MYSQL_MAX_ALLOWED_PACKET = 16_776_192; # mysql var: max_allowed_packet

# настройка sql_case, если ключей меньше чем MIN_IF_SIZE - используем case вместо if
# число 20 выбрано экспериментально
our $MIN_IF_SIZE ||= 20;

our %SLAVE_DB;
our %DBH_INFO;
our %DB_NAME_BY_DBH;
our $DEFAULT_PING_TIMEOUT = 10;    # 10 sec

=head2 $STRICT_LOCK_MODE

    Строгий режим работы с sql-локами:
        не больше одной блокировки на соединение
        умирать при переподключении, если был взят лок
        умирать при попытке освободить лок в отфоркнутом процессе, взятый в родительском

=cut
our $STRICT_LOCK_MODE //= 1;
our %LOCK_DBH;

our $TRANSACTION_LEVEL = 0;
our %TRANSACTION_DBH;
our %TRANSACTION_SAVEPOINT;

=head2 $Yandex::DBTools::DONT_SEND_LETTERS

    Если эта переменная - true, то, в случае, ошибок письма не отсылаются

=cut
our $DONT_SEND_LETTERS;

=head2 $Yandex::DBTools::QUOTE_DB

    База данных, хэндлер которой будет использоваться в sql_quote и sql_quote_identifier
    
=cut
our $QUOTE_DB;

=head2 $Yandex::DBTools::DB_USER

    Если эта переменная определена, get_db_config() будет работать так, как будто ему передан параметр db_user => $Yandex::DBTools::DB_USER
    Если при вызове get_db_config параметр db_user указан явно, значение $Yandex::DBTools::DB_USER игнорируется
    О значениях, которые может принимать эта переменная, см. документацию к get_db_config

=cut
our $DB_USER;

=head2 dbnames($db)
    по $db(в любом формате) получить список dbnames
=cut
sub dbnames
{
    my ($db) = @_;
    if (!defined $db) {
        croak "Db is not defined";
    } elsif (!ref $db) {
        return ($db);
    } elsif (ref $db eq 'HASH') {
        return @{$db->{dbnames}};
    } elsif (UNIVERSAL::isa($db, "DBI::db")) {
        croak "No dbname for dbh $db" if !defined $DB_NAME_BY_DBH{$db};
        return $DB_NAME_BY_DBH{$db};
    } else {
        croak "Unknown db type: ".ref($db);
    }
}

=head2 _dbname_vals($db)

    по $db(в формате хэша) получить список значений, соотвествующих $dbname
    для остальных форматов $db - undef

=cut
sub _dbname_vals
{
    my ($db, $dbname) = @_;
    if (ref $db eq 'HASH') {
        return $db->{dbname_vals}->{$dbname};
    } else {
        return undef;
    }
}

# по $db(в любом формате) проверить, что указана ровно одна база и получить её dbname
sub _one_dbname
{
    my ($db) = @_;
    
    my @dbnames = dbnames($db);
    croak "No dbname" if !@dbnames;
    croak "Too many dbnames: ".join(',', @dbnames) if @dbnames > 1;
    
    return $dbnames[0];
}

# по $db(в любом формате) проверить, что указана хотя бы одна база и получить первый попавшийся dbname
# может использоваться, если нужен хоть какой-то dbh для квотинга
sub _any_dbname
{
    my ($db, $sql) = @_;
    
    my @dbnames = dbnames($db);
    croak "No dbname" if !@dbnames;

    return $dbnames[0];
}

#======================================================================
# DEPRECATED
sub save_dbh2
{
    set_slave_db('ppc' => dbnames(shift));
}

=head2 set_slave_db(DB() => DB_HEAVY())

    Установка соответствия имён БД - имени базы, которое передаётся в функцию,
    и имени базы, в которую перенаправятся читающие запросы.
    Работает и для шардированных баз.

    Например, в приведённом примере, все последующие вызовы get_*_sql(DB, ...) будут 
    перенаправляться в DB_HEAVY(или, соотвествующий шард DB_HEAVY)

    Пример:
    set_slave_db(SHARDED(shard => 'all') => SHARDED_HEAVY(shard => 'all'))

=cut
sub set_slave_db
{
    my ($db, $slave_db) = @_;

    my @dbnames = dbnames($db);
    my @slave_dbnames = dbnames($slave_db);
    croak "Different number of shards in master and slave" if @dbnames != @slave_dbnames;

    # проверка существования баз
    map {get_db_config($_)} uniq @dbnames, @slave_dbnames;

    @SLAVE_DB{@dbnames} = @slave_dbnames;
}

=head2 wait_master( $db | [$db1, $db2], 5, no_warn => 0, no_check_slave => 0)

    Дождаться, пока слэйв указанной базы не догонит мастера.
    Параметры:
      db - либо dbname/dbh, тогда нужных слэйвов необходимо предварительно установить через set_db_slave
           - либо массив из двух элементов - первых элемент dbname мастера, второй - слэйва
      timeout - время, которое мы готовы ждать, в секундах
              если не указано - 5
              если реплика отстаёт больше чем на 5*timeout секунд - ждать даже не пытаемся 
              (это поведение можно отменить опцией no_check_slave)
      опция no_warn отвеняет вывод предупреждений о ошибках получения статусов или ожидания

    возвращаем:
    1     - слэйв догнал мастера
    0     - слейв не догнал мастера

=cut
sub wait_master
{
    my ($db_param, $timeout, %O ) = @_;

    # умолчательный timeout
    $timeout = 5 if !$timeout || $timeout !~ /^\d+$/;

    my $profile = Yandex::Trace::new_profile('db:wait_master');
    
    my ($dbnames, $slave_dbnames);
    if (ref $db_param eq 'ARRAY') {
        croak "Incorrect db parameter in wait_master" if @$db_param != 2;
        ($dbnames, $slave_dbnames) = map { [dbnames($_)] } @$db_param;
    } else {
        $dbnames = [dbnames($db_param)];
        $slave_dbnames = [map {$SLAVE_DB{$_} || $_} @$dbnames];
    }

    croak "Different numbers of dbnames for master and slave" if @$dbnames != @$slave_dbnames;

    my @shards = 
        grep {$_->{dbh}->{Name} ne $_->{slave_dbh}->{Name}}
        map { 
            {
                dbname => $dbnames->[$_],
                dbh => get_dbh($dbnames->[$_]),
                slave_dbname => $slave_dbnames->[$_], 
                slave_dbh => get_dbh($slave_dbnames->[$_]),
            } 
        } 
        grep {$dbnames->[$_] ne $slave_dbnames->[$_]}
        0..$#$dbnames;

    # пока неэффективный вариант - проверяем всё по-очереди
    for my $shard (@shards) {
        # Проверяем, а не слишком ли долго нам его ждать
        if (!$O{no_slave_check}) {
            my $slave = $shard->{slave_dbh}->selectrow_hashref("SHOW SLAVE STATUS");
            if (!$slave) {
                croak "Can't get slave status";
            } elsif (!defined $slave->{Seconds_Behind_Master}) {
                croak "Slave seems to be broken";
            } elsif ($slave->{Seconds_Behind_Master} > $timeout * 5) {
                warn "Slave is too late" if !$O{no_warn};
                return 0;
            }
        }
        # спрашиваем текущую позицию у мастера
        my $master = $shard->{dbh}->selectrow_hashref("SHOW MASTER STATUS");
        if ( !$master ) {
            croak "Can't get master position: $DBI::errstr";
        } elsif ( !$master->{File} || !$master->{Position} ) {
            croak "Can't get master position: No file or position";
        }
        # Ждём...
        my ($res) = $shard->{slave_dbh}->selectrow_array(
                        "SELECT MASTER_POS_WAIT(?, ?, ?)",
                        {},
                        $master->{File}, $master->{Position}, $timeout
            );
        if ( !defined $res ) {
            croak "Can't master_pos_wait: Unknown log name or slave thread is not started";
        } elsif ( $res == -1 ) {
            warn "Can't master_pos_wait: Timeout $timeout exceeded\n" if !$O{no_warn};
            return 0;
        }
    }
    return 1;
}

{
my %live_files;
# внутренняя функция - для dbname получить промердженный(с наследованием) хэш из конфига
sub _get_dbname_config_hash($) {
    my $dbname = shift;
    $dbname = '' if !defined $dbname;
    my $cfg;
    if (%DB_CONFIG) {
        $cfg = \%DB_CONFIG;
    } elsif (defined $CONFIG_FILE) {
        if ($CONFIG_FILE =~ /\.ya?ml$/) {
            require Yandex::LiveFile::YAML;
            $live_files{$CONFIG_FILE} ||= Yandex::LiveFile::YAML->new(filename => $CONFIG_FILE);
        } elsif (defined $CONFIG_FILE && $CONFIG_FILE =~ /\.json$/) {
            require Yandex::LiveFile::JSON;
            $live_files{$CONFIG_FILE} ||= Yandex::LiveFile::JSON->new(filename => $CONFIG_FILE);
        } else {
            die "Unknown CONFIG_FILE format: $CONFIG_FILE";
        }
        $cfg = $live_files{$CONFIG_FILE}->data->{db_config};
    } else {
        die "Neither CONFIG_FILE nor DB_CONFIG defined";
    }

    my $ret = hash_merge {}, $cfg;
    for my $dbname_part (split ':', $dbname) {
        if (exists $cfg->{CHILDS} && $cfg->{CHILDS}->{$dbname_part}) {
            $cfg = $cfg->{CHILDS}->{$dbname_part};
            hash_merge $ret, $cfg;
            delete $ret->{CHILDS} if !exists $cfg->{CHILDS};
        } else {
            die "Can't find db config for $dbname";
        }
    }
    return $ret;
}

=head2 get_db_config($)

    По названию базы получить хэш со свойствами соединения.
    В свойства дописываются:
    dbname - название соединения
    db - вычисляемое название базы, если не указано в конфиге

    CHILDS и extra_users в ответ НЕ включаются

    необязательные именованные параметры: 
    
    db_user => <username>
    если указан, то user и pass возвращаются для указанного пользователя; пароль берется из extra_users
    если такого пользователя нет в extra_users -- умирает
    если не указан и определена переменная $Yandex::DBTools::DB_USER, функция работает так, как будто её был передано db_user => $Yandex::DBTools::DB_USER
    Значение может быть:
        - скаляром с именем пользователя
        - ссылкой на хэш с ключами-именами баз (например "ppcdict", "ppc:1") и значениями -- именами пользователей. Используется значение для переданного имени базы. Если соответствующее значение undef или отсутствует, используется пользователь по умолчанию
        - ссылкой на процедуру, которая принимает имя базы (аналогично ключам хэша) и возвращает имя пользователя или undef. Используется возвращаемое имя пользователя. Если возвращается undef, используется пользователь по умолчанию
    Примеры:
        get_db_config($db, db_user => "direct-ro")
        get_db_config($db, db_user => {"ppc:15" => "direct-ro", "ppcdict" => "direct-ro"})
        get_db_config($db, db_user => sub { $_[0] =~ /^rbac/ ? undef : 'direct-ro' })
    Значения-хэши и значения-процедуры более полезны при определении $Yandex::DBTools::DB_USER, когда заранее неизвестно, для каких баз будет вызываться get_db_config():
        $Yandex::DBTools::DB_USER = sub { $_[0] =~ /^rbac/ ? undef : 'direct-ro' };

    skip_pass => (1|0)
    не пытаться читать пароль, вместо пароля отобразится строка <SKIPPED>
    для случаев, когда нужно прочитать db-config там, где нет файлов с паролями

=cut
sub get_db_config($;@) {
    my $db = shift;
    my %O = @_;
    my $dbname = _one_dbname($db);

    my $ret = _get_dbname_config_hash($dbname);
    
    if ($ret->{CHILDS}) {
        if ($ret->{CHILDS}->{'_'}) {
            hash_merge $ret, $ret->{CHILDS}->{'_'};
        } else {
            die "Can't find db config for $dbname (not leaf)";
        }
    }
    delete $ret->{CHILDS};
    ($ret->{dbname} = $dbname) =~ s/:_$//;
    $ret->{db} ||= (split(':', $dbname))[0];
    
    # если не указан инстанс - он совпадает с первой частью dbname
    $ret->{instance} = (split(':', $dbname))[0] if !defined $ret->{instance};

    if ( ! exists $O{db_user} && defined $DB_USER ) {
        $O{db_user} = $DB_USER;
    }
    if (exists $O{db_user} && ref $O{db_user}) {
        if (ref $O{db_user} eq 'HASH') {
            $O{db_user} = $O{db_user}->{$dbname};
        } elsif (ref $O{db_user} eq 'CODE') {
            $O{db_user} = $O{db_user}->($dbname);
        }
        delete $O{db_user} if !defined $O{db_user};
    }
    if ( exists $O{db_user} ){
        die "no extra user '$O{db_user}'" unless exists $ret->{extra_users} && exists $ret->{extra_users}->{$O{db_user}};
        $ret->{user} = $O{db_user};
        $ret->{pass} = $ret->{extra_users}->{$O{db_user}}->{pass} // '';
    } 
    delete $ret->{extra_users};

    if ($O{skip_pass}) {
        $ret->{pass} = '<SKIPPED>';
    }
    # в поле pass может быть не только строка, но и указание на то,
    # из какого файла нужно пароль прочитать
    if (ref($ret->{pass}) && ref($ret->{pass}) eq 'HASH') {
        if (defined $ret->{pass}->{file}) {
            my $file = $ret->{pass}->{file};
            $live_files{$file} ||= Yandex::LiveFile->new(filename => $file);
            $ret->{pass} = $live_files{$file}->data();
            $ret->{pass} =~ s/^\s*#.*\n?//gm;
            $ret->{pass} =~ s/^\s+|\s+$//g;
        } else {
            die "incorrect pass for $dbname";
        }
    }

    return $ret;
}
}

=head2 get_db_childs

    Получить массив "детей" указанной базы

    Параметры позиционные:
    - dbname - название базы (префикс)
               может быть undef - базы первого уровня (или все базы, в случае recursive=1)

    Именованные параметры:
    - recursive - логический, нужно ли получать детей всех уровней

    Возвращает ссылку на массив из "дочерних" баз (шардов или режимов)
    Если чайлдов нет - возвращает ссылку на пустой массив

=cut
sub get_db_childs($;%) {
    my ($dbname_prefix, %O) = @_;
    $dbname_prefix = '' if !defined $dbname_prefix;

    validate_with(
        params => [%O],
        spec => {recursive => 0},
        );

    my $cfg = _get_dbname_config_hash($dbname_prefix);

    return [_get_db_childs_rec($dbname_prefix, $cfg, $O{recursive})];
}

=head2 get_shards_numbers($dbname)

    Получить номера шардов по имени базы ($dbname)
    
    Результат:
        [id1, id2, id3] - список id шардов 

=cut

sub get_shards_numbers {
    my $dbname = shift;
    return [map {/:(\d+)/ ? $1 : ()} @{get_db_childs($dbname)}];
}

# рекурсивная реализация get_db_childs
sub _get_db_childs_rec {
    my ($pref, $cfg, $rec) = @_;
    my $childs = $cfg->{CHILDS};
    return () if !defined $childs;
    my @ret =
        map {$pref ne '' ? "$pref:$_" : $_}
        grep {!$childs->{$_}->{CHILDS} || $childs->{$_}->{CHILDS}->{'_'}}
        grep {$_ ne '_'}
        keys %$childs;
    if ($rec) {
        push @ret,
            map {_get_db_childs_rec($pref ne '' ? "$pref:$_" : $_, $childs->{$_}, $rec)}
            grep {$_ ne '_'}
            keys %$childs;
    }
    return @ret;
}

#======================================================================

=head2 get_dbh( $db [, $sql] )

    Получить объект-потомок DBI::db (хэндлер базы данных)
    Не работает с несколькими базами (бросает исключение)

    Параметры: 
    $db -- см. описание выше
    $sql -- (необязательный) запрос, который будет выполняться, 

    Результат: 
    хэндлер базы данных (указанной, или её реплики)

    Характерные примеры использования
        get_dbh(PPC)->quote("");

    Особенности реализации
    Реплика возвращается, если выполняются следующие условия:
        - реплика для данного db заранее установлена через set_slave_db
        - запрос - SELECT
        - в запросе не указан хинт /* sendmaster */

=cut
sub get_dbh
{
    my ($db, $sql) = @_;
    
    my $dbname = _one_dbname($db);
    my $slave_dbname = $SLAVE_DB{$dbname};

    my $result_dbh;
    if ($slave_dbname
        && $dbname ne $slave_dbname
        
        && defined $sql
        && $sql =~ m!^\s*select!i 
        && $sql !~ m!/\*\s*send(master|ppc)\s*\*/!i 
    ) {
        # Пытаемся обработать waitmaster
        if ( $sql =~ m!\/\*\s*waitmaster(?:\((.*?)\))?\s*\*\/!i ) {
            my $params = $1 || '';
            my ($timeout, $overwise) = split /\s*,\s*/, $params;
            # проверяем значения, ставим умолчания
            $overwise = 'slave' if !$overwise || ( $overwise ne 'master' && $overwise ne 'die' );
            
            if ( !wait_master([$dbname, $slave_dbname], $timeout) ) {
                # Если мастера не дождались...
                if ($overwise eq 'master') {
                    warn "wait_master failed, use master\n";
                    $result_dbh = soft_reconnect_db($dbname);
                } elsif ($overwise eq 'die') {
                    die "wait_master failed, die\n";
                } else {
                    warn "wait_master failed, use slave\n";
                }
            }
        }
        $result_dbh ||= soft_reconnect_db($slave_dbname);
    } else {
        $result_dbh = soft_reconnect_db($dbname);
    }

    if ( $TRANSACTION_LEVEL ) {
        $result_dbh->{AutoCommit} = 0;
        $TRANSACTION_DBH{''.$result_dbh} = $result_dbh;
        if ( $TRANSACTION_LEVEL > 1 && !$TRANSACTION_SAVEPOINT{''.$result_dbh}->{$TRANSACTION_LEVEL} ) {
            my $sp_name = "level_$TRANSACTION_LEVEL";
            my $sth = $result_dbh->prepare("SAVEPOINT $sp_name");
            _sth_execute($result_dbh, $sth);
            $TRANSACTION_SAVEPOINT{''.$result_dbh}->{$TRANSACTION_LEVEL} = $sp_name;
        }
    }

    return $result_dbh;
}

=head2 get_db_dsn($db_cfg [, $host] )

    По конфигу db получить perl-овый data_source
    Если указан параметр host - он подставляется в dsn, иначе берётся первый хост из конфига

=cut
sub get_db_dsn {
    my ($cfg, $host) = @_;
    $host ||= ref $cfg->{host} ? $cfg->{host}->[0] : $cfg->{host};
    my $dsn = "DBI:".($cfg->{driver} || 'mysql').":$cfg->{db};hostname=$host;port=$cfg->{port}";
    $dsn .= ";mysql_connect_timeout=$cfg->{connect_timeout}" if $cfg->{connect_timeout};
    $dsn .= ";mysql_compression=$cfg->{compression}" if $cfg->{compression};
    $dsn .= ";mysql_enable_utf8=1" if $cfg->{utf8};
    return $dsn;
}

#======================================================================

=head2 connect_db

Connect to database.
takes dbname or db_config hashref
return dbh
dies on error

=cut

sub connect_db
{
    my ($param) = @_;

    my $cfg = ref $param eq 'HASH' && $param->{host} ? $param : get_db_config($param);
    my $profile = Yandex::Trace::new_profile('db:connect_db', tags => "$cfg->{dbname}");

    my $dbh;

    # пытаемся законнектиться в первый доступный host
    my @candidate_hosts = ref $cfg->{host} ? @{$cfg->{host}} : $cfg->{host};
    my $last_host;
    for my $host (@candidate_hosts) {
        my $dsn = get_db_dsn($cfg, $host);
        Apache::DBI->setPingTimeOut($dsn, 0) if $INC{'Apache/DBI.pm'};
        my $auto_commit = exists $cfg->{AutoCommit} ? $cfg->{AutoCommit} : 1;
        my $read_only = exists $cfg->{ReadOnly} ? $cfg->{ReadOnly} : 0;
        if ($dbh = DBI->connect($dsn, $cfg->{user}, $cfg->{pass}, {RaiseError => 0, AutoCommit => $auto_commit, ReadOnly => $read_only})) {
            $dbh->{mysql_enable_utf8} = 1 if $cfg->{utf8};
            $last_host = $host;
            last;
        }
    }

    $dbh || die "Can't connect to $cfg->{dbname}: $DBI::errstr";
    bless $dbh, 'Yandex::DBTools::db';
    
    $dbh->{RaiseError} = 1;
    $dbh->{mysql_auto_reconnect} = 0;
    $dbh->{private_connector_pid} = $$;
    # изначально устанавливаем InactiveDestroy=1, если разрушение объекта будет происходить в том же процессе - 
    # в деструкторе выставим 0
    $dbh->{InactiveDestroy} = 1;

    if ($cfg->{names}) {
        eval { $dbh->do( "SET NAMES $cfg->{names}" ) };
        die "Can't set names '$cfg->{names}' to $cfg->{dbname}: $DBI::errstr" if $@;
    }

    if ($cfg->{dbname}) {
        my $info = $DBH_INFO{$cfg->{dbname}} ||= {};
        if ( $info->{dbh} ) {
            # use the single handler to store all connections for the same data_source
            $info->{dbh}->swap_inner_handle( $dbh );
            $dbh = $info->{dbh};
        } else {
            $info->{dbh} = $dbh;
        }
        delete $info->{need_reconnect};
        $info->{last_ping_time} = time();
        $info->{last_connected_host} = $last_host;
        $info->{ping_timeout} = defined $cfg->{ping_timeout} ? $cfg->{ping_timeout} : $DEFAULT_PING_TIMEOUT;
        $DB_NAME_BY_DBH{''.$dbh} = $cfg->{dbname};
    }

    return $dbh;
}

#======================================================================

#
#   Check if the dbh with given name is too old
#   takes: dbname or db handler
#   returns valid dbh or undef if timeout passed, ping and connect_db failed
#   undef is also returned if invalid dbname passed
#
sub soft_reconnect_db {
    my $db = shift;

    my $dbname = _one_dbname($db);
    my $info = $DBH_INFO{$dbname};
    my $dbh = $info ? $info->{dbh} : undef;

    if (!$dbh || $dbh->{private_connector_pid} != $$ || !$dbh->{Active}) {
        die "Forked connection inside transaction is not supported"  if $dbh && $TRANSACTION_LEVEL && $TRANSACTION_DBH{$dbh};
        die "Reconnect under sql-lock '$LOCK_DBH{$dbh}->{name}' on inactive dbh" if $dbh && $dbh->{private_connector_pid} == $$ && $STRICT_LOCK_MODE && $LOCK_DBH{$dbh};
        return connect_db($dbname);
    }

    if (_reconnection_is_not_necessary($dbh) && defined $info && _ping_if_needed($info)) {
        return $dbh;
    }

    die "Reconnect inside transaction"  if $TRANSACTION_LEVEL && $TRANSACTION_DBH{$dbh};
    die "Reconnect under sql-lock '$LOCK_DBH{$dbh}->{name}'" if $STRICT_LOCK_MODE && $LOCK_DBH{$dbh};

    return connect_db($dbname);
}

=head2 _reconnection_is_not_necessary

    Проверить, что с момента подключения хост не изменился в конфиге.
    Возвращает 1 если проверка отключена или хост совпадает.

=cut

sub _reconnection_is_not_necessary {
    if (!$CHECK_HOST_CHANGES) {
        return 1;
    }

    my ($dbh) = @_;
    my $dbname = $DB_NAME_BY_DBH{$dbh} || 'unknown';
    my $info = $DBH_INFO{$dbname};
    if (!$info) {
        return 1;
    }

    my $last_host = $info->{last_connected_host};

    my $cfg = get_db_config($dbname);
    my %candidate_hosts = map { $_ => undef } (ref $cfg->{host} ? @{$cfg->{host}} : $cfg->{host});
    return exists $candidate_hosts{ $last_host } ? 1 : 0;
}


=head2 has_broken_lock

    По имени базы проверить - нет ли на подключении "испорченного" лока.

    Возвращает 0 если локов нет или они выглядят "живыми".
    Возвращает 1 если есть лок и
        на соединении стоит признак "требуется переподключиться" (это бы привело к потере лока)
        или соединение текущего процесса неактивно

=cut

sub has_broken_lock {
    my $db = shift;

    my $dbname = _one_dbname($db);
    my $info = $DBH_INFO{$dbname};
    my $dbh = $info ? $info->{dbh} : undef;

    return $dbh && $LOCK_DBH{ $dbh } && ( $info->{need_reconnect} || $dbh->{private_connector_pid} == $$ && !$dbh->{Active} ) ? 1 : 0;
}

#======================================================================

=head2 disconnect_db

disconnect db.
disconnect_db($db)

=cut

sub disconnect_db
{
    my $db = shift;

    my @dbnames = dbnames($db);

    for my $dbname (@dbnames) {
        my $info = $DBH_INFO{$dbname};
        next unless $info;
        my $dbh = $info->{dbh};

        if ($TRANSACTION_LEVEL && $TRANSACTION_DBH{"$dbh"}) {
            croak "Can't disconnect_db for $dbname inside do_in_transaction";
        } 
        if ($dbh->{private_connector_pid} == $$) {
            if (! eval { $dbh->disconnect(); 1; }) {
                carp "Can't disconnect from '$dbh->{Name}': $@";
            }
        }
        delete $DBH_INFO{$dbname};
        delete $DB_NAME_BY_DBH{"$dbh"};
        delete $LOCK_DBH{"$dbh"};
    }
}


=head2 disconnect_all

    Обрывает все установленные соединения с БД (созданные в текущем процессе)
    Может быть полезно в демонах для всеобщего реконнекта

=cut

sub disconnect_all
{
    for my $db (keys %DBH_INFO) {
        disconnect_db($db);
    }
}

#======================================================================

=head3 _get_statement_type

    $type = _get_statement_type($sql);
    $type => 'lock'|'read'|'write'|'unknown';

=cut

sub _get_statement_type {
    my ($sql) = @_;

    $sql ||= '';

    # удалим возможные комментарии в начале запроса (вложенные комментарии не поддерживаются в MySQL)
    while ( $sql && $sql =~ m/\s*\w/ && $sql !~ m#^\s*[\w\(]#s ) {
        $sql =~ s#\s*\-\-\s[^\r\n]*##s; # "-- my comment type 1"
        $sql =~ s#/\*.*?\*/##s;         # "/* my comment type 2 */"
        $sql =~ s/\s*#[^\r\n]*//s;      # "# my comment type 3"
    }

    my $query_type = $sql =~ m/^[\(\s]*SELECT\s+(?:get_lock|release_lock|is_free_lock|is_used_lock)\s*\(/si
        ? 'lock'
        : $sql =~ m/^[\(\s]*(?:SET|SELECT)/si
        ? 'read'
        : $sql =~ m/^[\(\s]*(?:DELETE|UPDATE|INSERT|CREATE|ALTER|DROP|REPLACE|LOAD)/si
        ? 'write'
        : 'unknown';
}


=head3 _add_trace_comment

    Добавить в sql-запрос комментарий с информацией для отслеживания запроса
    На данный момент - добавляем reqid из Yandex::Trace

=cut
sub _add_trace_comment {
    my $sql_ref = shift;

    my $trace = Yandex::Trace::current_trace();
    return if !defined $trace;

    my $sanitize = sub {return substr shift=~s/[^0-9a-zA-Z_\.\-]/_/gr, 0, 64; };

    my $trace_comment_vars = $TRACE_COMMENT_VARS->();
    my $trace_info = "reqid:"
        .join(':', 
              map {$sanitize->($_)}
              ($trace->span_id() // '0'), 
              $trace->service(), 
              $trace->method(),
        )
        .join('', 
              map {':'.$sanitize->($_).'='.$sanitize->($trace_comment_vars->{$_})}
              sort(keys %$trace_comment_vars)
        );

    $$sql_ref =~ s!^(\s*\w+)!$1 /* $trace_info */!
        or $$sql_ref =~ s!^!/* $trace_info */ !;
}


sub _sth_execute {
    my ($dbh, $sth) = (shift, shift);

    my $query_type = _get_statement_type($sth->{Statement});
    #print STDERR "unknown statement: ".substr($sth->{Statement}, 0, 40)."\n" if $query_type eq 'unknown';
    my $db_name = $DB_NAME_BY_DBH{$dbh} || 'unknown';
    my $profile = Yandex::Trace::new_profile("db:$query_type", tags => "$db_name");
    my $tries = -1;
    while($tries++ < @DEADLOCK_PAUSE) {
        my $rv  = eval { $sth->execute(@_) };
        if ($rv) {
            if ($query_type eq 'write' && $rv + 0 > 0) {
                $profile->obj_num($rv + 0);
            }
            return $rv;
        }
        if ($dbh->{mysql_errno} eq '1205'
            || $dbh->{mysql_errno} eq '1213' && !$TRANSACTION_LEVEL
        ) {
            sleep($DEADLOCK_PAUSE[$tries]) if $DEADLOCK_PAUSE[$tries];
            next;
        } 
        die_sql("Can't execute the $sth->{Statement} in $db_name: $DBI::err, $DBI::errstr.");
    }
    die_sql("Can't execute the $sth->{Statement} in $db_name after ".(@DEADLOCK_PAUSE+1)." retries: $DBI::err, $DBI::errstr.");
}

=head2 exec_sql

This routine executes a sql statement and returns the sth

  $sth = exec_sql($db, "select cid, uid
                         from camps
                         where name = ? and statusModerate = ?", $name, 'Ready');

  вместо строки sql можно использовать ссылку на массив,
  все элементы которого соединяются в строку запроса,
  причем ссылки на хеш проходят через sql_condition()

  пример:
    do_sql(PPC, ["delete from bids", where => {cid => 1234, bid => [12, 34, 56, 78]}]);
    get_one_field_sql(PPC, ["select * from campaigns where", {cid => 1234, uid => 3882433}]);

=cut

sub exec_sql
{
    my ( $db, $sql_statement, @binds ) = @_;
    my @dbnames = dbnames($db);
    my @sths = ();

    if ( scalar @dbnames ) {
        for my $dbname (@dbnames) {
            my ( $ret, $sth ) = _exec_sql_wrapper( $db, $dbname, $sql_statement, @binds );
            push @sths, $sth;
        }    
    }

    return @sths == 1 ? $sths[0] : Yandex::DBTools::msth->new(\@sths);
}

#======================================================================

=head2 do_sql

This routine executes a sql statement

  do_sql($dbh, "delete from camps where name = ? and uid = ?", $name, $uid);
  return how many rows affected ("0E0" - is 0)

=cut

sub do_sql
{
    my ( $db, $sql_statement, @binds ) = @_;
    my @dbnames = dbnames($db);
    my $ret_val = 0;

    if ( scalar @dbnames ) {
        for my $dbname (@dbnames) {
           my ( $ret, $sth ) = _exec_sql_wrapper( $db, $dbname, $sql_statement, @binds );
           $ret_val += $ret;
        }
    }

    return $ret_val || '0E0';
}

#======================================================================

=item _exec_sql_wrapper

    Выполняет sql-запрос.

    В случае получения ошибок "2006, MySQL server has gone away" или "2013, Lost connection to MySQL server during query"
    пытается сделать реконнект.

    Для модифицирующих запросов пересодинение не делается.

    Параметры:
        $dbh - хендлер базы данных
        $dbname - одно из значений dbnames( $dbh )
        $sql_statement - SQL-стейтмент
        @binds - список bind-параметров

    Возвращает список: результат функции _sth_execute и стейтмент-хендлер ($sth).

    Пример:

    my ( $ret, $sth ) = _exec_sql_wrapper( $dbh, $dbname, $sql_statement );

    my $ret = _exec_sql_wrapper( $dbh, $dbname, $sql_statement );

=cut

sub _exec_sql_wrapper {

    my ( $db, $dbname, $sql_statement, @binds ) = @_;
    my ( $tries, $ret, $sth ) = ( -1, 0, undef );
    my $query_type = _get_statement_type( $sql_statement ) || '';
    my $max_tries = $query_type eq 'read' || $query_type eq 'lock' ? scalar @{ $RECONNECT_PAUSE || [] } : 1;

    while( $tries++ < $max_tries ) {

        my $dbh = get_dbh( $dbname, $sql_statement );
        my $sql = $sql_statement;

        if ( ref( $sql)  eq 'ARRAY' ) {
            $sql = _join_utf8( ' ',
                map { ref($_) eq 'HASH' || ref($_) eq 'ARRAY' ? _make_condition_str($dbh, $_, params => {vals => _dbname_vals($db, $dbname)}) : $_ }
                @$sql
            );
        }

        _check_readonly_dbh( $dbh, $sql );
        _add_trace_comment(\$sql) if $TRACE_COMMENT;
        _log_query( $dbh, $sql, @binds ) if defined $QUERIES_LOG;

        $sth = eval { $dbh->prepare( $sql ) };

        die_sql("Can't prepare $sql: $dbh->errstr.") if $@;

        $ret = eval { _sth_execute( $dbh, $sth, @binds ) };

        my $info = $DBH_INFO{ $dbname };
        if ( $max_tries > 1 && ${ $RECONNECT_PAUSE }[ $tries ] && ( $dbh->{mysql_errno} eq '2006' || $dbh->{mysql_errno} eq '2013' ) ) {
            $info->{need_reconnect} = 1;
            sleep( ${ $RECONNECT_PAUSE }[ $tries ] );
        } elsif ( $@ ) {
            $info->{need_reconnect} = 1;
            die_sql( $@ );
        } else {
            _log_warnings($dbh, $sth)  if $query_type eq 'write' && $sth->{mysql_warning_count};

            # успешно выполнили запрос, значит соединение было живо
            $info->{last_ping_time} = time();
            last;
        }
    }

    return ( $ret, $sth );
}

=head2 _check_readonly_dbh

  обрабатываем флаг $dbh->{ReadOnly}:
  пропускаем только запросы с SELECT|SET|SHOW в начале или все запросы с комментарием /* no_readonly */

=cut

sub _check_readonly_dbh($$)
{
    my ($dbh, $sql_statement) = @_;

    if ($dbh->{ReadOnly}
        && $sql_statement !~ m/^\s*(?:select|explain|set|show|desc)/i
        && $sql_statement !~ m| /\* \s* no_readonly \s* \*/ |x
       )
    {
        die_sql("Can't execute $sql_statement on ReadOnly dbh $DB_NAME_BY_DBH{$dbh}\n");
    }
}

=head2 _compose_sql_parts

  Формирует параметры, которые будут использоваться для формирования запроса на добавление или замены в записи в таблице: INSERT INTO, REPLACE INTO.
  На выходе получаем указатель на хещ:
  {
      ignore => "IGNORE",
      delayed => "DELAYED",
      no_readonly => "/* no_readonly */",
      on_duplicate_str => ...,
      set_str => ...
  }
  Ключи:
      ignore - надо ли добавлять IGNORY
      delayed - надо ли добавлять DELAYED
      no_readonly - надо ли добавлять /* no_readonly */ в запрос
      on_duplicate_str - если есть флаг on_duplicate_key_update, если есть, то формируется список field = value через запятую
      set_str - строка значений, которые будут устанавливаться.

=cut
sub _compose_sql_parts {
    my ($dbh, $values, %PARAM) = @_;
    my $result = {};
    my %KEY_FIELD = ();
    if ($PARAM{key}) {
        map {$KEY_FIELD{$_} = 1} ref($PARAM{key}) eq 'ARRAY' ? @{ $PARAM{key} } : $PARAM{key};
    }

    $result->{set_str} = _make_set_str($dbh, $values); 

    my @on_duplicate_arr = ();
    if ($PARAM{on_duplicate_key_update}) {
        for my $key (keys %$values) {
            my $parsed_field = _parse_options_from_fieldname($key);
            my $field = $parsed_field->{field};
            if ($parsed_field->{op}) {
                die "_make_set_str: operations not allowed in set statement ($key)";
            }
            next if $KEY_FIELD{$field};
            my $field_quoted = $dbh->quote_identifier($field);
            my $rpart;
            if (my $set_func = $parsed_field->{set_func}) {
                $rpart = $set_func->($field, $values->{$key});
            } else {
                $rpart = "VALUES($field_quoted)";
            }
            push @on_duplicate_arr, "$field_quoted = $rpart";
        }
        for my $field (keys %KEY_FIELD) {
            my $field_quoted = $dbh->quote_identifier($field);
            push @on_duplicate_arr, $field_quoted." = last_insert_id(".$field_quoted.")"; 
        }
    }
    $result->{on_duplicate_str} = join ', ', sort @on_duplicate_arr;

    $result->{ignore} = $PARAM{ignore} ? "IGNORE" : "";
    $result->{no_readonly} = $PARAM{no_readonly} ? '/* no_readonly */' : '';
    $result->{delayed} = $PARAM{delayed} ? "DELAYED" : "";

    return $result;
}
#======================================================================

=head2 do_insert_into_table($dbh, $table, $values, %PARAM)

    Вставить одну запись в таблицу 

    Параметры: 
    $db
    $table  -- таблица, куда делаем вставку
    $values -- хеш значений для вставки
    %PARAM -- хеш дополнительных (необязательных) параметров: 
        ignore -- флаг; должны ли игнорироваться дубликаты
        on_duplicate_key_update -- флаг; нужен ли в запросе "on duplicate key update"?
        key -- ключевое поле (или ссылка на массив ключевых полей)
            если указан on_duplicate_key_update, то в запросе добавится "field = VALUES(field)" для всех полей, перечисленных в $values, за исключением %PARAM{key}
            для него будет "$PARAM{key} = last_insert_id($PARAM{key})" -- для того, чтобы правильно возвращался id апдейтнутной строки
        no_readonly -- позволяем insert на ReadOnly $dbh
        delayed -- вставка с параметром DELAYED

    Результат:
    id добавленной/обновленной записи
       при использовании нескольких баз нельзя ожидать id,
       id возвращается только для обновлённых записей в отсутствие autoincrement

    В наборе значений для SET можно отменить квотирование поля, через __dont_quote
      'c.time__dont_quote' => 'NOW()'
    для полей типа SET поддерживается модификатор __sset, следующие записи аналогичны
      'c.flags' => 'f1'
      'c.flags__sset' => {f1 => 1, f2 => 0}
      'c.flags__sset' => ['f1']
    модификатор __smod работает аналогично __sset, но в случае on duplicate key SET модифицируется, а не перезаписывается

    Характерные примеры использования
        $b{bid}   = do_insert_into_table($dbh, 'banners', $banner_values);
        $vcard_id = do_insert_into_table($dbh, 'vcards', $vcard_values, on_duplicate_key_update => 1, key => 'vcard_id');
        do_insert_into_table(PPC, 'stat_table', {ClientID => $ClientID, stat_date => $stat_date, sum => $sum}, on_duplicate_key_update => 1, key => [qw/ClientID stat_date/]);

=cut
sub do_insert_into_table
{
    my ($db, $table, $values, %PARAM) = @_;
    
    my @dbnames = dbnames($db);
    need_void_context() if @dbnames != 1;
    return if !@dbnames;

    my @dbhs = map {get_dbh($_)} @dbnames;
    # dbh для вызовов quote
    my $qdbh = $dbhs[0];

    my $sql_parts = _compose_sql_parts($qdbh, $values, %PARAM);

    $sql_parts->{on_duplicate_str} = ($sql_parts->{on_duplicate_str}) ? "ON DUPLICATE KEY UPDATE $sql_parts->{on_duplicate_str}" : "";

    my $sql = _join_utf8(' ', "INSERT", $sql_parts->{delayed}, $sql_parts->{no_readonly}, $sql_parts->{ignore}, "INTO", $table, 
                              "SET", $sql_parts->{set_str}, $sql_parts->{on_duplicate_str});

    if (@dbnames == 1) {
        do_sql($dbhs[0], $sql);
        # или можно $dbh->last_insert_id(undef, undef, undef, undef); А что лучше -- непонятно. 
        return $dbhs[0]->{'mysql_insertid'};
    } else {
        for my $dbh (@dbhs) {
            do_sql($dbh, $sql);
        }
    }
}

=head2 do_replace_into_table($dbh, $table, $values, %PARAM)

    Вставить одну запись в таблицу, но если есть уже запись с такими же первичными ключами или уникальными индексами, то старую запись удалить, и добавить новую.

    Параметры: 
    $db
    $table  -- таблица, куда делаем вставку
    $values -- хеш значений для вставки
    %PARAM -- хеш дополнительных (необязательных) параметров: 
        no_readonly -- позволяем insert на ReadOnly $dbh
        delayed -- вставка с параметром DELAYED

    Результат:
    id добавленной/обновленной записи
       при использовании нескольких баз нельзя ожидать id,
       id возвращается только для обновлённых записей в отсутствие autoincrement

    В наборе значений для SET можно отменить квотирование поля, через __dont_quote
      'c.time__dont_quote' => 'NOW()'

    Характерные примеры использования
        $b{bid}   = do_replace_into_table(PPC, 'antispam', $antispam_values);

=cut
sub do_replace_into_table
{
    my ($db, $table, $values, %PARAM) = @_;

    my @dbnames = dbnames($db);
    need_void_context() if @dbnames != 1;
    return if !@dbnames;
    my @dbhs = map {get_dbh($_)} @dbnames;
    my $qdbh = $dbhs[0];

    my $sql_parts = _compose_sql_parts($qdbh, $values, %PARAM);

    my $sql = _join_utf8(' ', "REPLACE", $sql_parts->{delayed}, $sql_parts->{no_readonly}, "INTO", $table, "SET", $sql_parts->{set_str});

    if (@dbnames == 1) {
        do_sql($dbhs[0], $sql);
        return $dbhs[0]->{'mysql_insertid'};
    } else {
        for my $dbh (@dbhs) {
            do_sql($dbh, $sql);
        }
    }
}


=head2 _get_token_properties

    Параметр: 
        модификатор имени поля, например: 'not_in', 'dont_quote', 'collate(utf8_bin)'
    
    Результат: 
    {
        op         => 'not like',
        op_arity   => 2,
    }

=cut

{
    my %token_properties = (
        'dont_quote'  => { dont_quote => 1 },
        'int'         => { int => 1 },
        'is_null'     => { op_arity => 1, op => 'IS NULL' },
        'is_not_null' => { op_arity => 1, op => 'IS NOT NULL' },
        'like'        => { op_arity => 2, op => 'LIKE' },
        'not_like'    => { op_arity => 2, op => 'NOT LIKE' },
        'rlike'       => { op_arity => 2, op => 'RLIKE' },
        'not_rlike'   => { op_arity => 2, op => 'NOT RLIKE' },
        'in'          => { op_arity => 2, op => 'IN',      array_required => 1 },
        'not_in'      => { op_arity => 2, op => 'NOT IN',  array_required => 1 },
        'lt'          => { op_arity => 2, op => '<' },
        'gt'          => { op_arity => 2, op => '>' },
        'le'          => { op_arity => 2, op => '<=' },
        'ge'          => { op_arity => 2, op => '>=' },
        'ne'          => { op_arity => 2, op => '<>' },
        'is'          => { op_arity => 2, pattern => "IFNULL(%s, '') = IFNULL(%s, '')"},
        'between'     => { op_arity => 3, op => 'between', array_required => 1 },
        'bit_and'     => { op_arity => 2, op => '&' },
        'bit_or'      => { op_arity => 2, op => '|' },
        'bit_xor'     => { op_arity => 2, op => '^' },
        'starts_with' => { op_arity => 2, op => 'starts_with' },
        'contains'    => { op_arity => 2, op => 'contains'    },
        'contains_any'=> { op_arity => 2, op => 'LIKE', array_required => 1 },
        'collate'     => { params => ['collate'] },

        'scheck'    => { op_arity => 1, check_func => \&sql_set_check, hash_required => 1 },
        'smod'     => { op_arity => 1, set_func => \&sql_set_mod, hash_required => 1 },
        'sset'     => { op_arity => 1, set_func => \&_sql_set_set, hash_required => 1 },
    );

sub _get_token_properties
{
    my ($token) = @_;
    my ($token_id, $params_str) = $token =~ /^ (\w+) (?: \( (.*) \) )? $/xms
        or die "_get_token_properties: failed parsing token <$token>";
    
    my $token_prop = $token_properties{$token_id}
        or die "_get_token_properties: unknown field modifier $token_id";

    my %token_prop = %$token_prop;
    if ($params_str) {
        my $param_names = delete $token_prop{params}
            or die "_get_token_properties: unexpected token parameters $params_str";
        my @params = split /\s*,\s*/, $params_str;
        @token_prop{@$param_names} = @params;
    }

    return \%token_prop;
}
}


=head2 _parse_options_from_fieldname

    Параметр: 
        имя поле с модификаторами, например: 'cid__not_in', 'starttime__ge__dont_quote'
    
    Результат: 
    {
        field      => 'cid', 
        dont_quote => 1, 
        op         => 'not in', # 'not like', 'like', '<', '>',  ...., 'is null', 'is_not_null', 
        op_arity   => 2,        # или 1, арность операции (количество операндов)
    }

=cut

sub _parse_options_from_fieldname
{
    my ($field) = @_;

    my %res;
    ($res{field}, my @tokens_txt) = split '__', $field;

    for my $t (@tokens_txt){
        my $props = _get_token_properties($t);
        if (exists $res{dont_quote} && exists $props->{dont_quote}) {
            die "_parse_options_from_fieldname: multiple 'dont_quote' in '$field'";
        } elsif (exists $res{op} && exists $props->{op}) {
            die "_parse_options_from_fieldname: multiple operations in '$field'";
        }
        @res{keys %$props} = values %$props;
    }

    return \%res;
}


=head2 sql_condition($where, %params)

    Вход:
      ссылка на хеш или массив с where-условиями (поле => требуемое_значение)
      параметры:
       operation => AND или OR
       params => ссылку на структуру, которая будет передаваться в вызовы функций
    
    Результат:
      строка с sql-условием (БЕЗ слова 'where')

    Все условия соединяются через params{operation}, (по умолчанию AND).
    Ссылка на массив превращается в **корректное** условие IN (если массив пустой, то условие "0").
    Если хеш пуст, то условие '0'.

    Если в хеше встречается значение неподходящего типа -- умираем.

      {cid => 123456, statusActive => 'Yes'} => " cid = 123456 AND statusActive = 'Yes' "
      {bid => [$bid1, $bid2, $bid3]} => " bid IN ($bid1, $bid2, $bid3) "

    Если в качестве значения встречаем ссылку на функцию - выполняем её с аргументом, переданным в params,
    после этого рассматриваем результат как обычное значение 

      sql_condition({cid => sub {shift->{ids}}}, params => {ids => [123]}) =>  "cid IN (123)"

    Можно использовать и другие опрераци, через два подчеркивания: 
      cid__not_in => [1,2,3]
      login__like => '%-super'
      vcard_id__is_not_null => 1
    Для полей типа SET есть свой модификатор проверки
      flags__scheck => {geoglag => 1, show => 0} # записи, где geoglag включен, show выключен (на очтальные флаги не смотри)

    Список доступных операций см. в _parse_options_from_fieldname

    Можно отменить квотирование поля, через __dont_quote
      Опасная опция, но полезная в случае, если очень хочется поле сравнить с другим полем или чтем-нибудь типа NOW()
      'c.cid__dont_quote' => 'co.cid'

    Если указать модификатор __int - значит поле числовое, можно проверить значение и не квотить его
    Это позполяет оптимизировать in(...) с последующим group-by или in(...) с числами больше 2**32 (mysql не хочет использовать индекс)

    Можно добавить сложные условия:
      _OR => { uid => 1, cid => 2 }
      _AND => { ... }
      _NOT => { ... }
      _TEXT => "count(bid)>count(cid)"


    Примеры:
    $Yandex::DBTools::QUOTE_DB = 'ppcdict';
    my $cond1 = sql_condition({cid => 12345 }) => "`cid` = '12345'";
    my $cond2 = sql_condition({bid => [$bid1, $bid2], cid => $cid }) => "bid IN ($bid1, $bid2) AND cid = $cid";
    my $cond3 = sql_condition({bid => [$bid1, $bid2], _OR=>{cid => $cid, uid=>$uid }}) => "(cid = $cid OR uid = $uid) AND bid IN ($bid1, $bid2)"
    my $cond4 = sql_condition({_NOT => { _OR => { uid => 10, bid => 20 } } }) => "NOT ((`uid` ='10' OR `bid` ='20'))"
    my $cond5 = sql_condition({ tries__le => 3, _NOT => { _OR => [_TEXT => "length(text) > 1000", _TEXT => "TIMESTAMPDIFF(day, date_from, date_to) >= 30"] } })
        => "NOT ((( length(text) > 1000 ) OR ( TIMESTAMPDIFF(day, date_from, date_to) >= 30 ))) AND `tries` <='3'"

    DEPRECATED (с использованием указателя на dbh или имени БД):
    sql_condition($dbh, $where, %params):
    sql_condition($dbh, {cid => 12345 }) => "`cid` = '12345'"
    sql_condition($dbh, {bid => [$bid1, $bid2], cid => $cid }) => "bid IN ($bid1, $bid2) AND cid = $cid"

=cut
sub sql_condition {
    my $dbh = @_ % 2 ? $QUOTE_DB : shift;

    return _make_condition_str(get_dbh($dbh), @_);
}

# построение условия WHERE
sub _make_condition_str
{
    my ($dbh, $where, %params) = @_;

    my $operation = delete($params{operation}) // 'AND';

    my $where_arr = [];

    my $conditions = ref $where eq 'ARRAY' ? [ @$where ] : [ map { $_ => $where->{$_} } sort keys %$where ];
    
    while ( @$conditions ) {
        my ($f,$v) = splice @$conditions, 0, 2;
        if (ref($v) eq 'CODE') {
            $v = $v->($params{params});
        }

        my $field_hash = _parse_options_from_fieldname($f);
        my $fieldname_quoted = join('.', map {$dbh->quote_identifier($_)} split(/\./, $field_hash->{field}));

        die "_make_condition_str: operation $field_hash->{op} requres an array"
            if $field_hash->{array_required} && ref $v ne 'ARRAY';
        die "_make_condition_str: operation $field_hash->{op} requres an hash"
            if $field_hash->{hash_required} && ref $v ne 'HASH';

        if ( $field_hash->{check_func} ) {
            push @$where_arr, '(' . $field_hash->{check_func}->($field_hash->{field}, $v) . ')';
        } elsif ( $field_hash->{field} eq '_AND' ) {
            push @$where_arr, '(' . _make_condition_str( $dbh, $v, %params ) . ')';
        } elsif ( $field_hash->{field} eq '_OR' ) {
            push @$where_arr, '(' . _make_condition_str( $dbh, $v, operation => 'OR', %params ) . ')';
        } elsif ($field_hash->{field} eq '_NOT') {
            push @$where_arr, 'NOT (' . _make_condition_str( $dbh, $v, %params ) . ')';
        } elsif ( $field_hash->{field} eq '_TEXT' ) {
            push @$where_arr, "( $v )";
        } elsif (ref $v eq ''){
            $field_hash->{op} ||= '=';
            $field_hash->{op_arity} ||= 2;
            if ($field_hash->{int} && defined $v) {
                if ($v =~ /^-?\d+$/) {
                    $field_hash->{dont_quote} = 1;
                } else {
                    croak "Incorrect value of $f: $v";
                }
            }
            die "_make_condition_str: value for field with operation $field_hash->{op} should be 1 ($f)"
                if $field_hash->{op_arity} == 1 && $v ne '1';
           
            my $rpart = $field_hash->{op_arity} == 2 ? $v : '';
            if ( $field_hash->{op} eq 'starts_with' ) {
                $rpart = sql_quote_like_pattern($rpart) . '%';
            }
            if ( $field_hash->{op} eq 'contains' ) {
                $rpart = '%' . sql_quote_like_pattern($rpart) . '%';
            }
            $rpart = $dbh->quote($rpart) if $field_hash->{op_arity} == 2 && ! $field_hash->{dont_quote};
            
            if ( $field_hash->{pattern} ) {
                die "_make_condition_str: pattern printed conditions available only for 2 operand operators"
                    if $field_hash->{op_arity} != 2;
                push @$where_arr, sprintf(" ".$field_hash->{pattern}." ", $fieldname_quoted, $rpart);
            } elsif ( $field_hash->{op} eq 'starts_with' || $field_hash->{op} eq 'contains' ) {
                push @$where_arr, "$fieldname_quoted ". 'LIKE ' . $rpart;
            } else {
                push @$where_arr, "$fieldname_quoted ". ($field_hash->{op}) . $rpart;
            }

        } elsif (ref $v eq 'ARRAY'){
            $field_hash->{op} ||= 'IN';
            $field_hash->{op_arity} ||= 2;
            if ($field_hash->{int}) { 
                $v = [grep {defined $_} @$v];
                if (grep {defined $_ && $_ !~ /^-?\d+$/} @$v) {
                    croak "Incorrect value of $f: ".join(', ', @$v);
                } else {
                    $field_hash->{dont_quote} = 1;
                }
            }
            if ($field_hash->{pattern}) {
                die "_pattern condition requires scalar values";
            } elsif ($field_hash->{op} =~ /^(IN|NOT\s+IN)$/i){
                my $cond_on_empty_arr = $field_hash->{op} =~ /^NOT\s+IN$/i ? 1 : 0;
                my $s = join ',', map { $field_hash->{dont_quote} ? $_ : $dbh->quote($_)} @$v;
                push @$where_arr, ($s ? "$fieldname_quoted $field_hash->{op} ($s)" : $cond_on_empty_arr)
            } elsif ($field_hash->{op} =~ /^(between)$/i){
                die "_make_condition_str: BETWEEN requires array of two values" if @$v != 2;
                my ($l, $r) = map { $field_hash->{dont_quote} ? $_ : $dbh->quote($_)} @$v;
                push @$where_arr, " $fieldname_quoted BETWEEN $l AND $r ";
            } elsif ($field_hash->{op} =~ /^(like)$/i) {
                my @arr_like =  map { $fieldname_quoted." ".$field_hash->{op}." ".$_}
                                map { $field_hash->{dont_quote} ? $_ : $dbh->quote($_) }
                                map { '%' . sql_quote_like_pattern($_) . '%' }
                                    @$v;
                push @$where_arr, "(".join (" OR ", @arr_like).")";
            } elsif ($field_hash->{op} =~ /^(rlike)$/i) {
                my @arr_rlike = map { $fieldname_quoted." ".$field_hash->{op}." ".$_ } 
                                map { $field_hash->{dont_quote} ? $_ : $dbh->quote($_) } 
                                    @$v;
                push @$where_arr, "(".join (" OR ", @arr_rlike).")";
            } else {
                die "_make_condition_str: operation with array as right part should be IN, NOT IN or BETWEEN ($f)";
            }
        } else { 
            die "incorrect WHERE condition";
        }

        if ($field_hash->{collate}) {
            $where_arr->[-1] .= " COLLATE " . $dbh->quote($field_hash->{collate});
        }
    }

    my $condition = join(" $operation ", @$where_arr) || '0';

    return $condition;
}


=head2 _make_set_str

=cut

sub _make_set_str
{
    my ($dbh, $values, %PARAM) = @_;
    $PARAM{dont_quote} ||= {};

    my @set_parts;
    for my $f (sort keys %$values){
        my $field_hash = _parse_options_from_fieldname($f);
        die "_make_set_str: operations not allowed in set statement ($f)" if $field_hash->{op};

        my $lpart = join '.', map {$dbh->quote_identifier($_)} split /\./, $field_hash->{field};
        my $rpart;
        if ($field_hash->{set_func}) {
            $rpart = $field_hash->{set_func}->($field_hash->{field}, $values->{$f});
        } elsif ( !($field_hash->{dont_quote} || $PARAM{dont_quote}->{$field_hash->{field}}) ) {
            $rpart = $dbh->quote($values->{$f});
        } else {
            $rpart = $values->{$f};
        }
        push @set_parts, _join_utf8(' ', $lpart, "=", $rpart);
    }

    my $set_str = join(",", @set_parts);

    return $set_str; 
}

{
    # кешируем, не беспокоясь о дисконнектах - quote продолжит работать
    my $_quote_dbh;

=head2 sql_quote($str)

    Квотинг строки (для подстановки в sql запрос)
    Для использования необходима предварительная установка переменной $QUOTE_DB
    Быстрый аналог get_dbh($QUOTE_DB)->quote($str)

=cut
sub sql_quote {
    if (!$_quote_dbh) {
        croak "QUOTE_DB not defined" if !defined $QUOTE_DB;
        $_quote_dbh = get_dbh($QUOTE_DB);
    }
    return $_quote_dbh->quote($_[0]);
}

=head2 sql_quote_identifier($str)

    Квотинг идентификатора (для подстановки в sql запрос)
    Для использования необходима предварительная установка переменной $QUOTE_DB
    Быстрый аналог get_dbh($QUOTE_DB)->quote_identifier($str)

=cut
sub sql_quote_identifier {
    if (!$_quote_dbh) {
        croak "QUOTE_DB not defined" if !defined $QUOTE_DB;
        $_quote_dbh = get_dbh($QUOTE_DB);
    }
    return $_quote_dbh->quote_identifier($_[0]) if !$_[0] || $_[0] !~ /\./;
    return join '.', map {$_quote_dbh->quote_identifier($_)} split /\./, $_[0];
}
}

=head2 sql_quote_like_pattern

    Квотинг шаблона для оператора LIKE
    Для подстановки в SQL полученную строку надо дополнительно квотить через sql_quote

    $where{reverse_domain__like} = sql_quote(sql_quote_like_pattern('UR_ur/ur.xednay') . '%');
    $where{reverse_domain__like} => 'UR\_ur/ur.xednay%'

=cut

sub sql_quote_like_pattern {
    my ($pattern) = @_;

    $pattern =~ s/([_%\\])/\\$1/g;
    return $pattern;
}

=head2 sql_fields

    Готовит список полей в базе для вставки в SQL-запрос: берёт их в кавычки и объединяет
    в список через запятую.

=cut

sub sql_fields {
    my (@fields) = @_;

    croak 'Missing parameter: fields' unless @fields;

    my @quoted_fields;
    foreach my $field (@fields) {
        croak "Can't pass an empty field" if ! defined $field || $field eq '';
        push @quoted_fields, join '.', map { sql_quote_identifier($_) } split m{\.}, $field;
    }

    return join ',', @quoted_fields;
}

=head2 sql_case(field, {key1 => value1, ...}, | default => XXX)

    формирование куска SQL вида
    CASE field WHEN key1 THEN value1 ... ELSE default_value END

    на настояций момент - field безусловно квотится как identifier
    key, value, default - просто квотятся

    если опция default не указана - подставляется NULL

    опция default__dont_quote - значение для default, которое не нужно квотить

    если указан флаг dont_quote_value - не квотятся ни значения из кеша, ни default
    
    если ключей больше, чем $MIN_IF_SIZE=20 и все ключи - положительные целые числа,
    то строится бинарное дерево сравнений IF, где в листьях небольшие CASE:
    IF(f < 50, IF(f < 25, CASE..., CASE...), IF(f<75, CASE..., CASE...)))
    MySQL выполняет case полным перебором, поэтому мы оптимизируем число сравнений

    для нечисловых ключей дерево не строится из-за возможных отличий сравнения строк в perl и mysql

    см. также do_mass_update_sql для выполнения массовых update

    DEPRECATED
    Первым параметром может указываться db, которая будет использоваться для квотинга

=cut
{
    # чтобы не таскать все эти параметры по рекурсивным вызовам - сохраняем значения в локализованных переменных
    our ($_dbh, $_hash, $_quoted_field, $_dont_quote_value, $_quoted_default, $_keys);

sub sql_case {
    my $db = @_%2 ? shift : $QUOTE_DB;
    my ($field, $hash, %O) = @_;
    local $_dbh = get_dbh(_any_dbname($db));
    local $_hash = $hash;
    local $_quoted_field = join '.', map {$_dbh->quote_identifier($_)} split(/\./, $field);
    local $_dont_quote_value = delete $O{dont_quote_value};
    local $_quoted_default = 
        exists $O{default__dont_quote} 
        ? delete $O{default__dont_quote}
        : exists $O{default} 
        ? ( $_dont_quote_value
            ? delete $O{default}
            : $_dbh->quote(delete $O{default})
            )
        : 'NULL';

    croak "Incorrect options: ".join(',', keys %O) if keys %O;

    if (!keys %$hash) {
        return $_quoted_default;
    } elsif (keys %$hash > $MIN_IF_SIZE && all {/^\d+$/} keys %$hash) {
        local $_keys  = [sort {$a <=> $b} keys %$hash];
        return _sql_if_btree(0, $#{$_keys});
    } else {
        return _sql_case();
    }
}

# построение CASE
sub _sql_case {
    my @keys = $_[0] ? @{$_[0]} : keys %$_hash;
    return "CASE $_quoted_field " . join(' ', map {
        'WHEN ' . $_dbh->quote($_) . ' THEN ' . ($_dont_quote_value ? $_hash->{$_} : $_dbh->quote($_hash->{$_}))
        } @keys) . " ELSE $_quoted_default END"
}

# рекурсивная функция построения IF
sub _sql_if_btree {
    my ($start, $end) = @_;
    if ($end - $start < $MIN_IF_SIZE || $end == $start) {
        return _sql_case([map {$_keys->[$_]} $start .. $end]);
    }
    my $mid = int($start + ($end - $start) / 2);
    return "IF($_quoted_field <= $_keys->[$mid]"
        .",\n"._sql_if_btree($start, $mid)
        .",\n"._sql_if_btree($mid+1, $end)
        .")";
}
}


=head2 sql_set_check(field, {opt1 => bool1, opt2 => bool2, ...}) 

    формируем sql выражение, проверяющее, действительно ли в поле field типа SET 
    установлены флаги, которым во втором аргументе соответствуют true и 
    сняты флаги, которым во втором аргументе соответствуют false
    флаги, не переданные во втором аргументе не проверяются

    примеры:
    field='opt1,opt3', {opt1 => 1, opt2 => 0} -> true
    field='opt1,opt3', {opt1 => 1, opt3 => 0} -> false
    
=cut
sub sql_set_check($$) {
    my ($field, $opts) = @_;
    return "1" if !%$opts;
    my @conds = map {($opts->{$_} ? "" : "not ")."find_in_set(".sql_quote($_).", ".sql_quote_identifier($field).")"} keys %$opts;
    return "(".join(" and ", @conds).")";
}


=head2 sql_set_mod(field, {opt1 => bool1, opt2 => bool2})

    формируем sql выражение, устанавливающее или снимающее флаги в поле field типа SET 
    согласно переданному второму аргументу

    field='opt1,opt3', {opt3 => 0, opt2 => 1} -> 'opt1,opt2'

=cut
sub sql_set_mod($$) {
    my ($field, $opts) = @_;
    my $sql = sql_quote_identifier($field);
    return $sql if !defined $opts || !%$opts;
    $sql = "concat(',', ifnull($sql, ''), ',')";
    for my $key (sort keys %$opts) {
        if ($opts->{$key}) {
            $sql = "concat($sql, ".sql_quote($key).", ',')";
        } else {
            $sql = "replace($sql, ".sql_quote(",$key,").", ',')";
        }
    }
    return $sql;
}


# обработка разных способоз задания set-ов для модификатора __sset
sub _sql_set_set {
    my (undef, $vals) = @_;
    if (!defined $vals) {
        return sql_quote('');
    } elsif (ref $vals eq 'HASH') {
        return sql_quote(join ',', sort grep {$vals->{$_}} keys %$vals);
    } elsif (ref $vals eq 'ARRAY') {
        return sql_quote(join ',', sort @$vals);
    }
}

=head2 do_update_table($dbh, $table, $values, named_param_list)

    Выполняет update в таблице

    Параметры 
      $db
      $table   -- таблица для апдейта
      $values  -- ссылка на хеш со значениями для апдейта
      named_param_list 
        where -- ссылка на хеш с where-условиями {поле => требуемое_значение}
                 как интерпретируются условия -- см. в sql_condition
                 в частности, если where не указано, то ничего не обновляем
        dont_quote -- ссылка на массив имен полей, значения которых НЕ надо квотировать при подстановке в sql-запрос. + см. ниже про __dont_quote
                 Опасная опция, но полезная в случае, если очень хочется полю присвоить его же старое значение или что-нибудь типа NOW():
                 "UPDATE banners SET vcard_id = 156, LastChange = LastChange".
                 "UPDATE advq_testcase SET sent_time = NOW() WHERE case_id = 123456"
        no_readonly -- позволяем update на ReadOnly $dbh
        ignore -- 1/0, UPDATE IGNORE: не падать на дублирующихся ключах и невалидных значениях для полей


    В where-условии можно использовать и другие опрераци, через два подчеркивания: 
      cid__not_in => [1,2,3]
      login__like => '%-super'
      vcard_id__is_not_null => 1
    Для полей типа SET есть модификатор __smod
      flags__smod => {flag1 => 0, flag2 =>1} # выключить flag1, включить flag2, остальные не трогать

    Список доступных операций см. в _parse_options_from_fieldname

    В where-условии и в наборе значений для SET можно отменить квотирование поля, через __dont_quote
      'c.cid__dont_quote' => 'co.cid'
                 

    Характерные примеры использования: 
        do_update_table($dbh, 'banners', \%b, where => {bid => $bid});
        do_update_table($dbh, 'banners', {vcard_id => $vcard_id, LastChange => 'LastChange'}, where => {bid => \@bid}, dont_quote => ['LastChange']);
        do_update_table($dbh, 'banners', {vcard_id => $vcard_id, LastChange__dont_quote => 'LastChange'}, where => {bid => \@bid});
        do_update_table($dbh, 'banners b join phrases p using (pid)', {'b.opts__smod' => {x=>1}}, where => {'p.cid' => $cid});

=cut

# а еще лучше так: LastChange__dont_quote => 'LastChange', id__not_in => $ids, ...
# Если окажется ощутимая потребность -- можно добавить limit.
sub do_update_table
{
    my ($db, $table, $values, %PARAM) = @_;

    if (my @unknown_params = grep {!/^(where|dont_quote|no_readonly|ignore)$/} keys %PARAM) {
        croak "Unknown parameters was sent to do_update_table: ".join(',', @unknown_params);
    }
    
    my @dbnames = dbnames($db);
    return '0E0' if !@dbnames;
    my $qdbh = get_dbh($dbnames[0]);

    my $where = $PARAM{where} || {};
    my %DONT_QUOTE = map { $_ => 1 } @{$PARAM{dont_quote} || []};

    my $set_str = _make_set_str($qdbh, $values, dont_quote => \%DONT_QUOTE); 

    my $no_readonly = $PARAM{no_readonly} ? '/* no_readonly */' : '';
    my $ignore = ($PARAM{ignore}) ? 'IGNORE' : '';

    return do_sql($db, ["UPDATE",$no_readonly, $ignore, $table, "SET", $set_str, "WHERE", $where]);
}

=head2 do_mass_update_sql

    do_mass_update_sql($db, 'table', 'table_id',
        { $id1 => { $field => $value}, $id2 => { $field2 => $value, $field3 => $value }},
        where => { ... },
        byfield_options => {
            $field => {
                default__dont_quote => $field,
                dont_quote_value => 1,
                ..
            },
            ...
        }
    )

    Функция выполняет массовый update с помощью sql_case
    Порядок установки значений полей произвольный, т.е. ссылаться на изменяемые в этом же запросе поля нельзя

    Параметры:
        $db
        $table -- имя таблицы
        $id_field -- имя поля таблицы, по которому выбирается значения из $update_values
        $update_values -- хеш хешей. Ключи первого уровня - значения $id_field; значения -- хеши: имя поля => новое значение в поле
        + опциональные именованные:
            where -- дополнительные условия. Сюда будет дописаны все id из $update_values
            byfield_options -- ссылка на хеш. ключ -- имя поля, значение -- ссылка на хеш с опциями для sql_case
                               если для поля не указано значение по умолчанию, то поле без значения сохранит своё значение
                               если нужно всем всюду проставить одинаковое значение (константу или sql-expression) - можно не указывать поле в update_values,
                               и указать default/default__dont_quote в byfield_options

    Возвращает количество измененных записей

=cut

sub do_mass_update_sql
{
    my ($db, $table, $id_field, $update_values, %PARAM) = @_;

    my $where = delete $PARAM{where} // {};
    my $byfield_options = delete $PARAM{byfield_options} // {};

    if (my @unknown_params = keys %PARAM) {
        croak 'Unknown parameters was sent to do_mass_update_sql: ' . join(', ', @unknown_params);
    }

    my $affected_rows = 0;
    my $is_num_ids = none {/\D/} keys %$update_values;
    my @ids = sort {$is_num_ids ? $a <=> $b : $a cmp $b} keys %$update_values;
    while(my @ids_chunk = splice @ids, 0, $MAX_ROW_FOR_MASS_UPDATE) {
        my %field2values_map;
        for my $id (@ids_chunk) {
            my $field2value = $update_values->{$id};
            for my $field (keys %$field2value) {
                my $value = $field2value->{$field};
                $field2values_map{$field}->{$id} = $value;
            }
        }

        my %updates;
        for my $field (keys %field2values_map) {
            my $values_map = $field2values_map{$field};
            my $sql_case_options = $byfield_options->{$field} // {};
            if (!exists $sql_case_options->{default} && !exists $sql_case_options->{default__dont_quote}) {
                $sql_case_options = { %$sql_case_options, 'default__dont_quote' => sql_quote_identifier($field) };
            }
            $updates{"${field}__dont_quote"} = sql_case($id_field, $values_map, %$sql_case_options);
        }

        # прописываем default/default__dont_quote для столбцов, для которые не было никаких содержательных изменений
        for my $field (grep {!$field2values_map{$_}} keys %$byfield_options) {
            my $sql_case_options = $byfield_options->{$field};
            if (exists $sql_case_options->{default}) {
                $updates{$field} = $sql_case_options->{default};
            } elsif (exists $sql_case_options->{default__dont_quote}) {
                $updates{"${field}__dont_quote"} = $sql_case_options->{default__dont_quote};
            }
        }
        
        if (%updates) {
            $affected_rows += do_update_table($db, $table, \%updates,
                where => {
                    %$where,
                    $id_field => \@ids_chunk,
                }
            );
        }
    }
    return $affected_rows;
}

#
# Склеивает строки, снимая utf8 бит
#
sub _join_utf8
{
    my $d=shift;
    return join ($d, map { utf8::is_utf8($_) ? utf8::encode($_) : (); $_ } @_ );
}

=head2 do_delete_from_table($dbh, $table, named_param_list)

    удалить записи из таблицы

    Параметры
      $db
      $table   -- таблица для удалений
      named_param_list 
        where -- ссылка на хеш с where-условиями {поле => требуемое_значение}
                 как интерпретируются условия -- см. в sql_condition
                 в частности, если where не указано, то ничего не удаляем
        no_readonly -- позволяем delete на ReadOnly $dbh

    Характерный пример использования: 
        do_delete_from_table($dbh, 'vcards', where => {vcard_id => $orphan_vcards});

=cut
sub do_delete_from_table
{
    my ($db, $table, %PARAM) = @_;

    if (my @unknown_params = grep {!/^(where|no_readonly)$/} keys %PARAM) {
        croak "Unknown parameters sended to do_update_table: ".join(',', @unknown_params);
    }
    
    my $no_readonly = $PARAM{no_readonly} ? '/* no_readonly */' : '';
    return do_sql($db, ["DELETE $no_readonly FROM $table", WHERE => $PARAM{where} || {}]);
}


=head2 my $sql_inserts_arr = sql_mass_inserts($sql_template, \@rows, $options)

    генерация набора sql запросов на массовую вставку данных
    возвращает ссылку на массив строк
    
    возможные опции:
      max_row_for_insert => 10000,                    # redefine max row for one insert
      desired_sql_length => MYSQL_MAX_ALLOWED_PACKET  # желаемый размер результирующих запросов
      dont_quote => 1,                                # dont quote values
      no_readonly => 1,                               # позволяем insert на ReadOnly $dbh    
      quote_db => $QUOTE_DB,                          # каким dbh-ем квотить значения

=cut
my %_sql_mass_inserts_spec = (
    max_row_for_insert => 0,
    desired_sql_length => 0,
    dont_quote => 0,
    no_readonly => 0,
    quote_db => 0,
    );
sub sql_mass_inserts {
    my ($sql, $values, $options) = @_;
    $options //= {};

    validate_with(params => [%$options], spec => \%_sql_mass_inserts_spec);

    croak 'need arrayref data'  if ref $values ne 'ARRAY';
    return []  if !@$values;
    croak 'need arrayref data row'  if any { ref $_ ne 'ARRAY' } @$values;

    utf8::encode($sql) if utf8::is_utf8($sql);
    $sql =~ s! ! /* no_readonly */ ! if $options->{no_readonly};

    my $max_row_for_insert = $options->{max_row_for_insert} || $MAX_ROW_FOR_MASS_INSERT;
    my $desired_sql_length = $options->{desired_sql_length} || $MYSQL_MAX_ALLOWED_PACKET;

    my $quote_dbh = get_dbh(_any_dbname($options->{quote_db} // $QUOTE_DB));

    my @sql_inserts;
    for(my $i = 0; $i <= $#$values; ) {
        my @rows;
        my $cur_len = length($sql);
        while($i <= $#$values && @rows < $max_row_for_insert) {
            my $row_sql = '('
                . _join_utf8(",",  map { $options->{dont_quote} ? $_ : $quote_dbh->quote($_) } @{ $values->[$i] } )
                . ')';
            last if @rows && $cur_len + length($row_sql) + 1 >= $desired_sql_length;
            $cur_len += length($row_sql) + 1;
            push @rows, $row_sql;
            $i++;
        }
	
        my $sql_complete = sprintf $sql, _join_utf8(",", @rows);
        die "length of sql in sql_mass_inserts() greater than $MYSQL_MAX_ALLOWED_PACKET bytes\n" if length($sql_complete) > $MYSQL_MAX_ALLOWED_PACKET; # mysql var: max_allowed_packet

        push @sql_inserts, $sql_complete;
    }

    return \@sql_inserts;
}


=head2 do_mass_insert_sql

  $sql = 'insert into tbl (key1, key2) values %s';
  $values = [[$val1, $val2], [111, 222], [333, 444], ...];
  $options = {
      sleep => 2,                  # add sleep 2.0 sec between inserts
      sleep_coef => 0.5,           # add sleep exec_time * sleep_coef
      опции sql_mass_inserts
  };

  $count = do_mass_insert_sql($db, $sql, $values, $options);

=cut
my %_mass_insert_options_spec = (
    sleep => 0,
    sleep_coef => 0,
    %_sql_mass_inserts_spec,
    );
sub do_mass_insert_sql
{
    my ($db, $sql, $values, $options) = @_;
    $options //= {};
    validate_with(params => [%$options], spec => \%_mass_insert_options_spec);

    my @dbhs = map {get_dbh($_)} dbnames($db);
    return '0E0' if !@dbhs;

    my $sql_inserts = sql_mass_inserts(
        $sql, $values, 
        hash_merge {quote_db => $dbhs[0]}, hash_cut($options, keys %_sql_mass_inserts_spec)
        );

    my $affected_rows = 0;
    while(my $insert_sql = shift @$sql_inserts) {
        my $t1 = Time::HiRes::time;
        $affected_rows += sum(map {do_sql($_, $insert_sql)} @dbhs);
        my $exec_time = Time::HiRes::time - $t1;
        
        if (@$sql_inserts) {
            my $sleep = max($options->{sleep} || 0, ($options->{sleep_coef} || 0) * $exec_time);
            Time::HiRes::sleep($sleep) if $sleep > 0;
        }
    }

    return $affected_rows || '0E0';
}


=head2 do_insert_select_sql

    Делает указанный select в базу, записывает результат посредством указанного insert или replace(возможно в другую базу).

    Позиционные параметры:
    - db - имя БД или готовый DBI::db
    - insert_sql - sql запрос (insert, insert on duplicate key или replace), формат аналогичен do_mass_insert_sql,
                   место, куда подставляется массив значений обозначено %s
    - select_sql - sql запрос - либо просто строка, либо массив в новом стиле
    
    Именованные параметры:
    - binds - массив "связанных" значения, которые будут подставлены в SELECT вместо знаков "?"
    - dbw - имя базы, в которую будет делаться запись, если не указано, запись идёт в базу db
    - max_row_for_select - какими кусками(по сколько строк) идёт чтение
    Также принимаются все опции, поддерживаемые do_mass_insert_sql

    Возвращает число изменённых строк, аналогично DBI::db::do

    Примеры:
    do_insert_select_sql(PPC, "INSERT INTO tbl (name) VALUES %s", 
                              "SELECT name FROM tbl2 where id = ?",
                              binds => [$id],
                              );
    do_insert_select_sql(PPC, "INSERT INTO tbl (name) VALUES %s", 
                              ["SELECT name FROM tbl2", WHERE => {id => $id}],
                              dbw => MONITOR,
                              sleep => 1, dont_quote => 1,
                              );

=cut
sub do_insert_select_sql {
    my ($db, $insert_sql, $select_sql, %O) = @_;
    %O = validate_with(params => [%O], spec => {
        binds => {type => ARRAYREF, optional => 1, default => []},
        dbw => {optional => 1, default => $db},
        max_row_for_select => {optional => 1, default => $MAX_ROW_FOR_SELECT},
        %_mass_insert_options_spec,
    });
    
    my $sth = exec_sql($db, $select_sql, @{$O{binds}});
    my $cnt = 0;
    while(my $chunk = $sth->fetchall_arrayref([], $O{max_row_for_select})) {
        $cnt += do_mass_insert_sql($O{dbw}, $insert_sql, $chunk,
                                   hash_cut(\%O, keys %_mass_insert_options_spec)
                                  );
        last if @$chunk < $O{max_row_for_select};
    }
    return $cnt;
}

#======================================================================

=head2 get_one_line_sql

This routine executes a sql statement and returns the one line result as hash ref

  $hash_ref = get_one_line_sql($dbh, "select cid, uid
                                      from camps
                                      where name = ? and statusModerate = ?", $name, 'Ready');
  print "cid: $hash_ref->{cid}; uid: $hash_ref->{uid}\n";

=cut

sub get_one_line_sql
{
    my ($db, $sql_statement) = (shift, shift);

    my $sth = exec_sql($db, $sql_statement, @_);
    my $hash_ref_result = $sth->fetchrow_hashref();
    $sth->finish();

    return $hash_ref_result;
}

#======================================================================

=head2 get_one_line_array_sql

This routine executes a sql statement and returns the one line result as array

  ($cid, $uid) = get_one_line_array_sql($dbh, "select cid, uid
                                      from camps
                                      where name = ? and statusModerate = ?", $name, 'Ready');

=cut

sub get_one_line_array_sql
{
    my ($db, $sql_statement) = (shift, shift);

    my $sth = exec_sql($db, $sql_statement, @_);
    my @ret = $sth->fetchrow_array();
    $sth->finish();

    return @ret;
}

#======================================================================

=head2 get_one_field_sql

This routine executes a sql statement and returns the one field result

  $count = get_one_field_sql($dbh, "select count(*) from camps where uid = ?", $uid);

=cut

sub get_one_field_sql
{
    my ($db, $sql_statement) = (shift, shift);

    my $sth = exec_sql($db, $sql_statement, @_);
    my $result = ($sth->fetchrow_array())[0];
    $sth->finish();

    return $result;
}

#======================================================================

=head2 get_one_column_sql

  This routine executes a sql statement and returns the reference to array with all values of first column

  $arr = get_one_column_sql($dbh, "select cid from camps where name = ?", $name);
  # => [1, 2, 4]

  $arr = get_one_column_sql($dbh, "select cid from camps where 0")
  # => []

=cut

sub get_one_column_sql
{
    my ($db, $sql_statement) = (shift, shift);

    my $sth = exec_sql($db, $sql_statement, @_);

    my @ret;
    while(my ($val) = $sth->fetchrow_array()) {
        push @ret, $val;
    }
    $sth->finish();

    return \@ret;
}

#======================================================================

=head2 get_all_sql

This routine executes a sql statement and returns the all result

  $ref_to_array_hashref = get_all_sql($dbh, "select cid, name from camps where name = ?", $name);

  $ref_to_array_hashref => [{cid => 1, name => 'user 1'}, {cid => 2, name => 'user 2'}, {cid => 3, name => 'user 3'}]

=cut

sub get_all_sql
{
    my ($db, $sql_statement) = (shift, shift);
    my ($result, $row) = ([], undef);

    my $sth = exec_sql($db, $sql_statement, @_);

    while ($row = $sth->fetchrow_hashref()) {
        push @{$result}, $row;
    }
    $sth->finish();

    return $result;
}

=head2 get_hashes_hash_sql

  получить хеш: ключ - первый столбец, значение - хэш строки

  $hashref = get_hashes_hash_sql($dbh, "select cid, name from camps where name = ?", $name);

  $hashref: {1 => {cid => 1, name => 'user 1'}, 2 => {cid => 2, name => 'user 2'}, 3 => {cid => 3, name => 'user 3'}}

=cut

sub get_hashes_hash_sql {
    my ($db, $SQL, @binds) = @_;

    return {} if !dbnames($db);
    my $sth = exec_sql($db, $SQL, @binds);
    my $first_col_name = $sth->{NAME}->[0];
    my $hash = {};
    while( my $row = $sth->fetchrow_hashref() ) {
        my $key = defined $row->{$first_col_name} ? $row->{$first_col_name} : '';
        $hash->{ $key } = $row;
    }
    return $hash;
}

=head2 get_hash_sql

  получить хеш: ключ - первый столбец, значение - второй столбец

  $hashref = get_hash_sql($dbh, "select cid, name from camps where name = ?", $name);

  $hashref: {1 => 'user 1', 2 => 'user 2', 3 => 'user 3'}

=cut

sub get_hash_sql {
    my ($dbh, $SQL, @binds) = @_;

    my $sth = exec_sql($dbh, $SQL, @binds);
    my $hash = {};
    while (my ($key, $value) = $sth->fetchrow_array()) {
        $key = '' unless defined $key;
        $hash->{$key} = $value;
    }
    return $hash;
}

=head2 parse_data_source

    Парсинг перлового data_source

=cut
sub parse_data_source {
    my $data_source = shift;
    # удаляем dbd:mysql:
    $data_source =~ s/^ [^:]+ : [^:]+ : //x;
    sub _param {
        return $_[0] =~ m/;\Q$_[1]\E=([^;]+)/ ? $1 : undef;
    }
    my @parts = split /;/, $data_source;
    if ($parts[0] =~ /:/) {
        # ppcdict:ppcmaster01b.yandex.ru:3311;mysql_enable_utf8=1;mysql_connect_timeout=60
        my ($db, $host, $port) = split ':', $parts[0];
        return {db => $db, host => $host, port => $port};
    } elsif (my $host = _param($data_source, 'hostname')) {
        # ppclog;hostname=test-mysql.yandex.ru;port=3309;mysql_enable_utf8=1
        return {db => $parts[0], host => $host, port => _param($data_source, 'port')};
    } elsif (my $socket = _param($data_source, 'mysql_socket')) {
        # ppc;mysql_socket=/var/run/mysqld.ppc/mysqld.sock;mysql_enable_utf8=1
        return {db => $parts[0], mysql_socket => $socket};
    } else {
        die "data_source $data_source not parsed\n"
    }
}

=head2 my $lock = sql_lock_guard(PPC, 'super_lock', 3)

    Получить именованный sql lock.

    Позиционные параметры:
    db - база данных
    lock_name - имя блокировки
    timeout - сколько времени пытаемся получить блокировку, по умолчанию 1 час

    Возвращает объект, который при выходе из зоны видимости сам снимет блокировку.
    Если блокировку получить не удалось (нет коннекта или таймаут) - умирает.

=cut
sub sql_lock_guard {
    my ( $db, $lock_name, $timeout ) = @_;

    my $dbh = get_dbh($db);
    my $lock_result = get_lock_sql($dbh, $lock_name, $timeout);
    if (!$lock_result) {
        die "Can't get sql lock '$lock_name'";
    }
    return Yandex::DBTools::LockGuard->new(dbh => $dbh, lock_name => $lock_name);
}

=head2 get_lock_sql

    Получить именованный sql lock.
    Умирает, если в текущем соединении с бд уже был взят какой-либо лок (даже с тем же именем)

    Возвращает:
        1 -- лок получен
        0 -- таймаут получения лока
        undef -- ошибка

    $res = get_lock_sql(PPC, 'super_lock', 3);

=cut

sub get_lock_sql {
    my ( $db, $lock_name, $timeout ) = @_;

    $timeout = defined $timeout ? $timeout : 3600;
    my $dbh = get_dbh($db);
    if ($STRICT_LOCK_MODE && exists $LOCK_DBH{'' . $dbh}) {
        my $old_lock_name = $LOCK_DBH{'' . $dbh}->{name} // 'unknown';
        die "Allowed only one lock per connection. Old lock name is $old_lock_name";
    }
    my $res = get_one_field_sql($dbh, 'SELECT get_lock(?, ?)', $lock_name, $timeout);
    if ($res) {
        $LOCK_DBH{'' . $dbh} = {
            name => $lock_name,
            pid => $$,
        };
    }
    return $res;
}

=head2 release_lock_sql

    Отпустить именованный sql lock
    Умирает при попытке в отфоркнутом процессе освободить лок, взятый в родительском

    Возвращает:
        1 -- лок успешно отпущен
        0 -- лок существует, но захвачен кем-то другим (при этом лок НЕ отпускается)
        undef -- лок не существует

    $res = release_lock_sql(PPC, 'super_lock');

=cut

sub release_lock_sql {
    my ( $db, $lock_name ) = @_;

    my $dbh = get_dbh($db);
    my $lock_info = $LOCK_DBH{'' . $dbh};
    if ($STRICT_LOCK_MODE
        && $lock_info
        && $lock_name eq $lock_info->{name}
        && $lock_info->{pid} != $$
    ) {
        die "Can't release sql-lock '$lock_name' in forked connection!";
    }
    my $res;
    if ($dbh->{Active}) {
        $res = get_one_field_sql($dbh, 'SELECT release_lock(?)', $lock_name);
    }
    if (($res || !defined $res)
        && $lock_info && $lock_info->{name} eq $lock_name
    ) {
        delete $LOCK_DBH{'' . $dbh};
    }
    return $res;
}

# Отослать сообщение об ошибке и умереть
sub die_sql {
    my $msg = shift || '';
    if (!$DONT_SEND_LETTERS && $msg !~ /Table 'ppc(?:price)?log\..*' doesn't exist/) {
        require Yandex::SendMail;
        Yandex::SendMail::send_alert(Carp::longmess($msg), "SQL error");
    }
    die $msg;
}

{
my $log;
# внутренняя функция для логгирования запросов
sub _log_query {
    my ($db, $sql, @params) = @_;
    $log ||= new Yandex::Log(
        log_file_name => $QUERIES_LOG,
        date_suf => "%Y%m%d",
        msg_prefix => "[$$]",
        auto_rotate => 1,
        );

    my (@trace, @stack);
    my $i = 1;
    push @trace, "$stack[3]($stack[0]:$stack[2])" while ((@stack = caller $i++));
    
    $log->out({db => [dbnames($db)],
               sql => $sql,
               params => \@params,
               trace => \@trace,
              }
        );
}
}


{
my $wlog;

sub _log_warnings {
    my ($dbh, $sth) = @_;

    my $num_warn = $sth->{mysql_warning_count};
    return  if !$num_warn;

    $wlog ||= Log::Any->get_logger(category => $LOG_WARNINGS_CATEGORY);

    if ($wlog->is_warn) {
        my $warns = $dbh->selectall_arrayref(
            "SHOW WARNINGS LIMIT ?", {Slice => {}}, $LOG_WARNINGS_COUNT
        );
        return  if !$warns || !@$warns;

        my $reqid = Yandex::Trace::current_span_id() || 0;
        for my $i (0 .. $#$warns) {
            my $warn = $warns->[$i];
            my $msg ="$warn->{Level} $warn->{Code}: $warn->{Message} ($reqid)";
            $msg .= sprintf(" / %d more skipped", $num_warn-$LOG_WARNINGS_COUNT)  if $i == $#$warns && $num_warn > $LOG_WARNINGS_COUNT;
            if ($LOG_WARNINGS_STACKTRACE) {
                # skip _exec_sql_wrapper
                local $Carp::CarpLevel = $Carp::CarpLevel + 1;
                $msg .= "\n" . Carp::longmess();
            }
            $wlog->warn($msg);
        }
    };

    return;
}
}


=head2 is_table_exists(db, table_name)

    Проверить, существует ли таблица.
    Имя таблицы можно передавать с указанием схемы: db.table_name

=cut
sub is_table_exists {
    my ($db, $table_name) = @_;
    croak if !defined $table_name;
    _one_dbname($db);
    my $COND;
    if ($table_name =~ /^(.*)\.(.*)$/) {
        $COND = {table_schema => $1, table_name => $2};
    } else {
        $COND = {table_schema__dont_quote => 'database()', table_name => $table_name};
    }
    my $count = get_one_field_sql($db, ["SELECT COUNT(*) FROM information_schema.tables", WHERE => $COND]);
    return $count;
}


=head2 get_table_partitions($db, $table_name)

    Возвращаем названия партиций указанной таблицы
    Ходим в information_schema.partitions

=cut

sub get_table_partitions
{
    my ($db, $table_name) = @_;
    _one_dbname($db);

    return get_one_column_sql($db, '
        SELECT partition_name
        FROM information_schema.partitions
        WHERE table_schema = database()
          AND table_name   = ?
        ORDER BY partition_ordinal_position
    ', $table_name);
}


=head2 do_in_transaction( \&code );

    Выполнить в транзакции
    Если была ошибка - rollback и падаем
    Вложенные транзакции реализованы через savepoint

=cut

sub do_in_transaction (&)
{
    my ($code) = @_;

    $TRANSACTION_LEVEL ++;

    my $return;
    my $result = eval { $return = $code->(); 1 };
    my $error_msg = $@;

    $TRANSACTION_LEVEL --;

    if ( !$TRANSACTION_LEVEL ) {
        my $commit_error;

        # перед коммитом пингуем все базы, чтобы уменьшить вероятность того,
        # что закоммитив в часть баз, наткнёмся на нерабочую
        if ( $result ) {
            for my $dbh ( values %TRANSACTION_DBH ) {
                my $dbname = $DB_NAME_BY_DBH{''.$dbh} || 'unknown';
                my $info = $DBH_INFO{$dbname};
                if ($info && !eval { _ping_if_needed($info) }) {
                    $commit_error = "Error pinging $dbname: error = $@, DBI error = " . $dbh->errstr();
                }
            }
        }

        for my $dbh ( values %TRANSACTION_DBH ) {
            if ( $result && !$commit_error ) {
                eval { $dbh->commit() }  or do { $commit_error = $dbh->errstr() || 'Unknown error' };
            } else {
                eval { $dbh->rollback() };
            }

            my $dbname = $DB_NAME_BY_DBH{''.$dbh};
            my $config = get_db_config($dbname);
            $dbh->{AutoCommit} = exists $config->{AutoCommit} ? $config->{AutoCommit} : 1;
        }
        %TRANSACTION_DBH = ();
        %TRANSACTION_SAVEPOINT = ();
        die "Commit error: $commit_error" if defined $commit_error;
    } else {
        my $sp_level = $TRANSACTION_LEVEL+1;
        for my $dbh_ref ( keys %TRANSACTION_SAVEPOINT ) {
            my $savepoint = delete $TRANSACTION_SAVEPOINT{$dbh_ref}->{$sp_level};
            next if !$savepoint;

            my $dbh = $TRANSACTION_DBH{$dbh_ref};
            my $sql = ($result ? 'RELEASE SAVEPOINT' : 'ROLLBACK TO') . " $savepoint";
            my $sth = $dbh->prepare($sql);
            _sth_execute($dbh, $sth);
        }
    }

    die "Transaction failed: $error_msg"  if !$result;

    return $return;
}

=head2 select_found_rows

    Выполняет подсчет найденных строк "SELECT found_rows()", адаптированно к шардированию.
    Обязательно предварительно в SELECT запросе указывать SELECT SQL_CALC_FOUND_ROWS

    exec_sql(PPC(shard=>...), "SELECT SQL_CALC_FOUND_ROWS ..."; 
    my $count = select_found_rows(PPC(shard=>...));    

=cut
sub select_found_rows {
    my $db = shift;
    return sum (@{get_one_column_sql( $db, "SELECT found_rows()")}) || 0;
}

=head2 _ping_if_needed

    Если соединение неактивно более чем $ping_timeout секунд, делает проверочный запрос в базу
    Возвращает 1, если пинг отключен, недавно был или успешно сделался.

    $connect_alive = _ping_if_needed($dbname);
    $connect_alive => 1|0

=cut

sub _ping_if_needed {
    my ($info) = @_;

    my $ping_timeout = $info->{ping_timeout};
    if (!$ping_timeout) {
        # пинг отключен
        return 1;
    }

    my $last_ping_time = $info->{last_ping_time};
    my $dbh = $info->{dbh};

    if ($info->{need_reconnect}) {
        return 0;
    } elsif (defined $last_ping_time && time() <= $last_ping_time + $ping_timeout) {
        # пинг недавно был и не требуется
        return 1;
    } elsif ($dbh->ping()) {
        # успешно пинганули
        $info->{last_ping_time} = time();
        return 1;
    } else {
        # неуспешно пинганули
        return 0;
    }
}


package Yandex::DBTools::db;

# мини обёртка вокруг DBI::db для корректной установки InactiveDestroy при разрушении объектов

use strict;
use warnings;
use parent qw/-norequire DBI::db/;


my $GLOBAL_DESTRUCTION = 0;

sub DESTROY {
    return if $GLOBAL_DESTRUCTION;
    my $self = shift;
    if ($self->{private_connector_pid} == $$) {
        $self->{InactiveDestroy} = 0;
    }
}

# во время global destruction порядок разрушения объектов не определён
# и внутренности DBI::db могут разрушиться раньше вызова Yandex::DBTools::db::DESTROY
# END срабатывает перед global destruction
sub END {
    for my $dbh (grep {$_} map {$_->{dbh}} values %Yandex::DBTools::DBH_INFO) {
        if ($dbh->{private_connector_pid} == $$) {
            $dbh->{InactiveDestroy} = 0;
        }
    }
    $GLOBAL_DESTRUCTION = 1;
}

sub ping {
    my ($dbh) = @_;

    my $db_name = $DB_NAME_BY_DBH{$dbh} || 'unknown';
    my $profile = Yandex::Trace::new_profile('db:ping', tags => "$db_name");
    my $rv = eval {
        my $sth = $dbh->prepare_cached('SELECT /* PING */ 1');
        die unless $sth;
        $sth->execute() or die;
        $sth->finish();
        return 1;
    };
    return ($rv) ? 1 : 0;
}

package Yandex::DBTools::LockGuard;

# только для использования из sql_lock_guard

sub new {
    shift;
    bless {@_, pid => $$};
}

{
my $GLOBAL_DESTRUCTION = 0;

sub DESTROY {
    my $self = shift;
    return if $GLOBAL_DESTRUCTION || $self->{pid} != $$;
    local $STRICT_LOCK_MODE = 0;
    Yandex::DBTools::release_lock_sql($self->{dbh}, $self->{lock_name});
}

sub END {
    $GLOBAL_DESTRUCTION = 1;
}
}

package Yandex::DBTools::msth;

use Carp qw/croak/;
use Hash::Util qw/lock_keys/;

sub new
{
    my (undef, $sths) = @_;
    croak "sth's count must not be 1" unless @$sths != 1;
    my $self = {sths => $sths, idx => 0};
    if (@$sths) {
        $self->{$_} = $sths->[0]->{$_} for (qw/NAME NUM_OF_FIELDS/);
    }
    bless $self;
    lock_keys(%$self);
    return $self;
}

sub fetchrow_array
{
    my ($self) = (shift);
    while($self->{idx} <= $#{$self->{sths}}) {
        my @ret = $self->{sths}->[$self->{idx}]->fetchrow_array(@_);
        return @ret if @ret;
        $self->{idx}++;
    }
    return ();
}

sub fetchrow_hashref {
    my ($self) = (shift);
    while($self->{idx} <= $#{$self->{sths}}) {
        my $ret = $self->{sths}->[$self->{idx}]->fetchrow_hashref(@_);
        return $ret if $ret;
        $self->{idx}++;
    }
    return undef;
}

sub fetchall_arrayref {
    my ($self, $slice, $max_rows) = @_;
    croak unless !defined $max_rows || $max_rows > 0;
    my @buf;
    while($self->{idx} <= $#{$self->{sths}}) { 
        my $rows = $max_rows ? $max_rows - @buf : undef;
        my $res = $self->{sths}->[$self->{idx}]->fetchall_arrayref($slice, $rows);
        push @buf, @$res;
        if ($max_rows && @buf == $max_rows) {
            return \@buf;
        }        
        $self->{idx}++;
    }
    return \@buf;
}

sub finish {
    my $self = shift;
    for my $sth (@{$self->{sths}}) {
        $sth->finish;
    }
}

1;

=head1 AUTHORS

in alphabetical order:

  Elena Bolshakova C<lena-san@yandex-team.ru>
  Georgy Ivanov C<gerich@yandex-team.ru>
  Maxim Kuzmin C<kuzmin@yandex-team.ru>
  Sergey Mudrik C<msa@yandex-team.ru>
  Sergey Zhuravlev C<zhur@yandex-team.ru>

=head1 COPYRIGHT

Copyright (c) 2004 - 2010 Yandex. All rights reserved.

=cut
