package Yandex::DBUnitTest;

=head1 NAME
    
    DBUnitTest - юнит-тесты с доступом к mysql

=head1 SYNOPSIS

    use Settings;
    use Yandex::DBTools;
    use DBUnitTest qw/:all/;

    my $dataset = {
        bids => {
            original_db => PPC,
        },
    };
    init_test_dataset($dataset);
    check_test_dataset($dataset);
    do_insert_into_table(UT, 'bids', {});
    push @{$dataset->{bids}->{rows}}, {};
    check_test_dataset($dataset);

    copy_table(PPC, 'bids');
    ok(get_one_field_sql(UT, "select count(bid) from bids") == 0);
    do_insert_into_table(UT, 'bids', {});
    ok(get_one_field_sql(UT, "select count(bid) from bids") == 1);

=head1 DESCRIPTION

    Удобное написание юнит-тестов, использующих БД
    Модуль позволяет создавать временную БД, копировать части таблиц из 
    основных баз, автоматически удалает созданную БД.
    Если установлена переменная окружения UNIT_OPT_DB=0, все тесты пропускаются.
    Подменяет функцию Yandex::DBTools::get_dbh, чтобы все операции проводились над базой UNIT_TESTS

    Есть 3 варианта(уровня) использования:
    1. использовать init_test_dataset для создания таблиц и заполнения их тестовыми данными
       вызвать проверяемую функцию, проверить её возвращаемое значение
       сделать изменения в тестовых наборах данных (обычный доступ к хэшам/массивам)
       проверить данные в БД на соответствие перловым структурам с помощью check_test_dataset
    2. использовать create_table / copy_table для создания таблиц
       использовать replace_test_data/check_test_data для заполнения проверки данных
       отличие от 1-го способа - здесь идёт работа по таблицам, а не сразу со всеми таблицами
    3. создавать таблички и прверять результаты руками

=cut

use strict;
use warnings;

use feature 'state';

use POSIX qw/strftime/;
use List::MoreUtils qw/uniq/;
use Data::Dumper;
use YAML;
use Fcntl ':flock';
require Test::More;
require Test::Deep;

use Yandex::DBTools;
use Yandex::DBShards;
$Yandex::DBTools::CONFIG_FILE = '/etc/dbtools/db-config.unit-tests.yaml';
use Yandex::DBSchema;
$Yandex::DBSchema::NO_DB_IN_TABLE_NAME = 1;

=head2 $NO_CHANGELING

    Ещё один способ запрещать подмену функций

=cut
our $NO_CHANGELING;


=head2 $SHARDS_NUM

    количество шардов в тестовой БД

=cut
our $SHARDS_NUM = 4;

#use MailService; # иначе Yandex::DBTools::die_sql не работает... # кажется, уже не требуется

use base qw/Exporter/;
our @EXPORT_OK = qw/
    copy_table create_table
    replace_test_data check_test_data
    init_test_dataset check_test_dataset
    UT UNIT_TESTS
    SHUT SHARDED_UNIT_TESTS SHUT_1 SHUT_2
    SHUT_3 SHUT_4
    /;
our @EXPORT = qw/UT UNIT_TESTS SHUT SHUT_1 SHUT_2 
    SHUT_3
    SHUT_4
/;

my ($orig_dbtools_get_dbh, $orig_dbtools_connect_db, $orig_get_shard_dbnames, $orig_get_first_shard_dbname);
{no strict 'refs'; $orig_get_shard_dbnames = *{"Yandex::DBShards::get_shard_dbnames"}{CODE};}
{no strict 'refs'; $orig_get_first_shard_dbname = *{"Yandex::DBShards::get_first_shard_dbname"}{CODE};}

use constant UT => 'unit_tests';
use constant UNIT_TESTS => 'unit_tests';
*SHUT = sub {
    my $ret = $orig_get_shard_dbnames->('sharded_unit_tests', '', @_);
    $ret->{database} = 'sharded_unit_tests';
    return $ret;
};
*SHARDED_UNIT_TESTS = *SHUT;
use constant SHUT_1 => 'sharded_unit_tests:1';
use constant SHUT_2 => 'sharded_unit_tests:2';
use constant SHUT_3 => 'sharded_unit_tests:3';
use constant SHUT_4 => 'sharded_unit_tests:4';

our $ENV_DB_NAME = 'UNIT_TESTS_DATABASE';
my $DATABASE;
# созданные базы: имя -> pid
my %CREATED_DB = ();
our %CREATED_TABLES;
my $LOCK_FILE_LOCATION;

our ($_lock_file, $_lock_pid, $_lock_fh);

=head2 $MAX_ROWS_CNT_FOR_DELETE

    максимальное количество строк, которое можно удалить  при выполнении
    replace_test_data, по-умолчанию - 1000

=cut
our $MAX_ROWS_CNT_FOR_DELETE ||= 1000;

=head2 $DEFAULT_ENGINE

    Тип создаваемых таблиц, по-умолчанию - MyISAM    

=cut
our $DEFAULT_ENGINE ||= 'MyISAM';

=head2 $SHARDED_DB_RE
    
    Регулярное выражение, определеющее какие базы данных считать шардированными. Например:
        $SHARDED_DB_RE = qr/(?:ppc|ppcstat)/
    Технически, запросы к базам из SHARDED_DB_RE выполняются на SHUT, при этом нужно помнить,
        что в ней может оказаться меньше шардов, чем ожидается от оригинальной базы.

=cut
our $SHARDED_DB_RE;

# В юнит-тестах используем для квотирования базу unit-tests
$Yandex::DBTools::QUOTE_DB = UT;

=head2 import

    При импорте, помимо функций, можно указывать имена функций, или тэги
    :create_db - создание новой БД
    :all / :subs - импортировать все функции
    :no_init - не инициализировать базу
    :no_skip - не пропускать тесты, даже если указана переменная окружения $ENV{UNIT_OPT_DB}=0
    :no_changeling - не подменять все базы данных на UT

=cut
sub import {
    my $pkg = shift;

    my %opts = map {(my $flag=$_) =~ s/^://; $flag => 1} grep {/^:/} @_;

    my @unknown_opts = grep {!/^(?:create_db|no_init|no_skip|no_changeling|all|subs)$/} keys %opts;
    die "Unknown import options: ".join(', ', @unknown_opts) if @unknown_opts;

    # если установрена переменная окружения - скипаем все тесты и выходим
    if (!$opts{no_skip} && defined $ENV{UNIT_OPT_DB} && !$ENV{UNIT_OPT_DB}) {
        require Test::More;
        if (Test::More->builder->has_plan) {
          SKIP: {
              Test::More::skip("--no-db option used", Test::More->builder->expected_tests);
            }
        } else {
            Test::More->builder->skip_all("--no-db option used");
        }
        exit(0);
    }

    # подменяем Yandex::DBTools::get_dbh, чтобы вся работа шла с базой UT
    if (!$NO_CHANGELING && !$opts{no_changeling} && !defined $orig_dbtools_get_dbh) {
        no strict 'refs';
        no warnings;
        $orig_dbtools_get_dbh = *{"Yandex::DBTools::get_dbh"}{CODE};
        *{"Yandex::DBTools::get_dbh"} = sub {
            my ($db, $sql) = @_;
            return $orig_dbtools_get_dbh->(
                (ref $db ? $db : $db =~ /unit_tests/ ? $db : UNIT_TESTS),
                $sql
                );
        };
        $orig_dbtools_connect_db = *{"Yandex::DBTools::connect_db"}{CODE};
        *{"Yandex::DBTools::connect_db"} = sub {
            my $db = shift;
            return $orig_dbtools_connect_db->(
                ($db =~ /unit_tests/ ? $db : UNIT_TESTS),
                @_
                );
        };
        *{"Yandex::DBShards::get_shard_dbnames"} = sub {
            my ($prefix, $suffix, @params) = @_;
            my $ret = $orig_get_shard_dbnames->(
                (_is_sharded($prefix) ? 'sharded_unit_tests' : 'unit_tests'),
                undef,
                @params
                );
            $ret->{database} = $prefix;
            return $ret;
        };
        *{"Yandex::DBShards::get_first_shard_dbname"} = sub {
            my ($prefix, $suffix, @params) = @_;
            my $ret = $orig_get_first_shard_dbname->(
                (_is_sharded($prefix) ? 'sharded_unit_tests' : 'unit_tests'),
            );
            $ret->{database} = $prefix;
            return $ret;
        };
    }

    $LOCK_FILE_LOCATION = $ENV{UNITTESTS_TMP_DIRECTORY} // '/tmp';
    $DATABASE = $ENV{$ENV_DB_NAME};
    state $db_idx = 0;
    if ($opts{create_db} || !$opts{no_init} && !$DATABASE) {
        my $login = [getpwuid($<)]->[0];
        $login =~ s/-/_/g;
        $DATABASE = join '_', 'unit_tests', $login, $$, $db_idx++, strftime("%Y%m%d%H%M%S", localtime);
        for my $db ($DATABASE, map {$DATABASE."_".$_} 1..$SHARDS_NUM) {
            do_sql(UNIT_TESTS, "CREATE DATABASE $db DEFAULT CHARSET utf8;");
            $CREATED_DB{$db} = $$;
        }
        $ENV{$ENV_DB_NAME} = $DATABASE;
        $_lock_file = "$LOCK_FILE_LOCATION/dbunittest_$DATABASE.lock";
    } elsif (! $opts{no_init}) {
        $_lock_file = "$LOCK_FILE_LOCATION/dbunittest_$DATABASE.lock";
        $_lock_pid = $$;
        open($_lock_fh, ">>", $_lock_file) || die "Can't open $_lock_file: $!";
        flock($_lock_fh, LOCK_EX) || die "Can't lock $_lock_file: $!";
    }

    if (!$opts{no_init}) {
        die "No created unit_tests database" if !$DATABASE;
        die "Incorrect unit_tests database" if $DATABASE !~ /^[a-z0-9_]+$/;
        if (!%Yandex::DBTools::DB_CONFIG) {
            %Yandex::DBTools::DB_CONFIG = %{YAML::LoadFile($Yandex::DBTools::CONFIG_FILE)->{db_config}};
        }
        $Yandex::DBTools::DB_CONFIG{CHILDS}{unit_tests}{db} = $DATABASE;
        $Yandex::DBTools::DB_CONFIG{CHILDS}{sharded_unit_tests}{CHILDS}{1}{db} = $DATABASE.'_1';
        $Yandex::DBTools::DB_CONFIG{CHILDS}{sharded_unit_tests}{CHILDS}{2}{db} = $DATABASE.'_2';
        $Yandex::DBTools::DB_CONFIG{CHILDS}{sharded_unit_tests}{CHILDS}{3}{db} = $DATABASE.'_3';
        $Yandex::DBTools::DB_CONFIG{CHILDS}{sharded_unit_tests}{CHILDS}{4}{db} = $DATABASE.'_4';
        do_sql(UNIT_TESTS, "USE $DATABASE");
    }

    if ($opts{subs} || $opts{all}) {
        @_ = @EXPORT_OK;
    }
    local $Exporter::ExportLevel = $Exporter::ExportLevel+1;
    $pkg->SUPER::import(grep {!/^:/} @_);
}

sub cleanup {
    while (my($db, $tables) = each %{$CREATED_TABLES{$$}}) {
        foreach my $tbl (keys %$tables) {
            do_sql($db, "DROP TABLE $tbl");
        }
    }
    for my $db (keys %CREATED_DB) {
        next unless $CREATED_DB{$db} == $$;
        do_sql(UNIT_TESTS, "DROP DATABASE $db");
        my $lock_file = defined $LOCK_FILE_LOCATION ? "$LOCK_FILE_LOCATION/dbunittest_$db.lock" : "/tmp/dbunittest_$db.lock";
        unlink($lock_file) if -f $_lock_file;
    }
    if ($_lock_fh && $_lock_pid == $$) {
        flock($_lock_fh, LOCK_UN);
        close($_lock_fh);
        ($_lock_fh, $_lock_pid) = ();
    }
}

END {
    cleanup();
}

=head2 UT

    Константа, сокращение от UNIT_TESTS, экспортируется по-умолчанию.
    
=cut

=head2 copy_table(PPC, 'bids')

    Копировать структуру таблицы из существующей базы данных

    Особенность: 
    первым параметром ($db) надо передавать строку ("ppc", "ppclog", MONITOR...), или имя таблицы указывать с префиксом по БД ("ppc.banners"), или и то, и другое вместе. 
    Иначе работать не будет, 
    "read_file '/opt/www/beta.lena-san.8080/db_schema/unit_tests_lena_san_20101126193021/campaigns.schema.sql' - sysopen: No such file or directory" 
    Причина: create_table_by_schema должна откуда-нибудь понять, в каком каталоге искать файл со схемой.

    Принимает два позиционных параметра и именованные:
        temporary -- создавать ли таблицы как временные (boolean)
        engine -- движок для создаваемых таблиц (по умолчанию MyISAM)
        like -- строка либо ссылка на хеш с описанием таблицы, по образцу которой надо создать новую
                Строка должна иметь вид "table" или "db.table", в хеше должно быть поле table и может быть поле db
        with_data -- заполнить таблицу данными из соответствуюзего .data файла из db_schema

=cut
sub copy_table {
    my ($db, $table, %opts) = @_;
    my $orig_table = $table;
    
    if( $table !~ /\./) {
        if (!ref $db && $db !~ /^unit_tests/) {
            $table = "$db.$table";
        } elsif (ref $db eq 'HASH' && $db->{database}) {
            $table = "$db->{database}.$table";
        }
    }

    my %create_options = (
        temporary => $opts{temporary},
        engine => $opts{engine} || $DEFAULT_ENGINE,
    );
    $create_options{like} = $opts{like} if $opts{like};
    create_table_by_schema($db, $table, %create_options);

    if ($opts{with_data}) {
        my ($database, $tbl) = split /\./, $table;
        for my $sql (@{Yandex::DBSchema::get_table_data_sqls(db => $database, table => $tbl)}) {
            do_sql($db, $sql);
        }
    }

    _save_table_info($db, $orig_table);
    return $orig_table; 
}

=head2 create_table($db, $table_name => qw/id:auto:pk name price:float/)

    Быстрое создание таблиц.
    По-умолчанию, создаются TEMPORARY таблицы, есть нужна регулярная, нужно указать
    флажок -not_temporary

    Примеры:
    create_table(UT, $tbl => qw/id:auto:pk name price:float/)
    create_table(UT, $tbl => qw/-not_temporary name/)
    create_table(UT, $tbl => 'id:auto', 'name', 'index:name,id', ':price float not null, unique(id, price)')
    create_table(UT, $tbl => qw/-not_temporary name/)
    create_table(SHUT, $tbl => qw/-not_temporary name/)

=cut
sub create_table {
    my ($db, $table, @data) = @_;
    my @sql;
    my $TEMPORARY = 0;
    for my $d (@data) {
        if ($d =~ /^(index|unique|primary):(\w+(?:,\w+)*)$/) {
            push @sql, "$1 ($2)";
        } elsif ($d =~ /^-(.*)$/) {
            if ($1 eq 'temporary') {
                $TEMPORARY = 1;
            } else {
                die "Unknown flag '$d'";
            }
        } elsif ($d =~ /^:(.*)/) {
            push @sql, $1;
        } elsif ($d =~ /^([\w\(\)]+)/) {
            my ($colname, @opts) = split /:/, $d;
            my $type = '';
            my $add_type = '';
            my $flag_not_null = 0;
            my $flag_primary = 0;
            for my $o (@opts) {
                if ($o =~ /^((?:varchar|char|varbinary|binary|(?:tiny||medium|large)(text|blob)|date|datetime|timestamp|decimal|float|double|(?:tiny|small|big)?int)[\(\)\d\,]*)$/) {
                    $type = $1;
                } elsif ($o eq 'auto') {
                    if (!$type && $type !~ /int/) {
                        $type = 'int unsigned';
                    }
                    $flag_not_null = 1;
                    $flag_primary = 1;
                    $add_type .= ' auto_increment';
                } elsif ($o =~ /^not_?null$/) {
                    $flag_not_null = 1;
                } elsif ($o =~ /^(pk|primary)$/) {
                    $flag_not_null = 1;
                    $flag_primary = 1;
                } else {
                    die "Incorrect column specifier: $o";
                }
            }
            push @sql, join(" ",
                            sql_quote_identifier($colname),
                            $type || 'varchar(100)',
                            ($flag_not_null ? 'not null' : ''),
                            $add_type,
                            ($flag_primary ? 'primary key' : '')
                );
        } else {
            die "Incorrect column definition: '$d'";
        }
    }
    my $CREATE_SQL = "CREATE ".($TEMPORARY ? 'TEMPORARY' : '')." TABLE $table (\n".join("\n, ", @sql)."\n)";
    if ($DEFAULT_ENGINE) {
        $CREATE_SQL .= " ENGINE=$DEFAULT_ENGINE";
    }

    do_sql($db, $CREATE_SQL);
    _save_table_info($db, $table);
    return $table;
}

# тешим свою паранойю - если строк в таблице слишком много - умираем.
sub _check_table_size {
    my ($db, $table) = @_;
    my $rows_cnt = get_one_field_sql($db, "SELECT count(*) from $table");
    if ($rows_cnt > $MAX_ROWS_CNT_FOR_DELETE) {
        die "Too many rows for delete: $rows_cnt";
    }
}

=head2 replace_test_data($db, $table_name, $test_data)

    заменить данные в таблице на переданный набор строк.
    параметры позиционные:
      - $db - для совместимости может не передаваться, expires: 2013-12-01
      - table_name - имя таблицы
      - test_data - тестовые данные - ссылка на:
          * массив хешей - для нешардированных баз будет записан в UT, для шардированных - в первый шард SHUT
          * хеш - ключ - номер шарда для записи, значение - массив хешей (собственно данные для вставки).
                на нешардированных базах - умирает. Номер шарда 'all' допустим, если нет других ключей

=cut
sub replace_test_data {
    my $db = @_==3 ? shift : UT;
    my ($table, $test_data) = @_;

    my $sharded = _is_sharded($db);
    
    _check_table_size($db, $table);
    # удаляем старые данные
    do_sql($db, "DELETE FROM $table LIMIT ?", $MAX_ROWS_CNT_FOR_DELETE);

    if (ref $test_data eq 'HASH') {
        die q/can't replace data in not sharded database/ unless $sharded;
        die q/usage shard key 'all' isn't allowed with another keys/ if exists $test_data->{all} && scalar keys %{ $test_data } > 1;
        # Пишем в тот шард, который указан
        while (my ($shard, $rows) = each %$test_data) {
            _insert_rows(SHUT(shard => $shard), $table, $rows);
        }
    } else {
        # если база шардированая, а шард не указан - пишем в первый ## expires: 01.01.2014
        _insert_rows($sharded ? SHUT_1 : UT, $table, $test_data);
    }
}

# Записывает в базу переданные строчки (вынесенро из replace_test_data)
sub _insert_rows {
    my ($db, $table, $rows_data) = @_;
    # вставляе переданные строки
    for my $row (@$rows_data) {
        do_insert_into_table($db, $table, $row);
    }
}

=head2 check_test_data($table_name, $test_data, $message)

    проверить, что содержащиеся в таблице данные соответствуют переданному набору
    параметры позиционные:
      - table_name - имя таблицы
      - test_data - массив хешей
        проверка будет осуществляться с помощью Test::Deep, массив превратится в bag,
        чтобы не зависить от порядка строк, а хеши - в superhashof, для проверки только
        тех ключей, что есть в переданном тестовои наборе
      - message - сообщение, передаваемое в ok
    результат выводится в STDOUT в формате TAP

=cut
sub check_test_data {
    my ($table, $test_data, $message) = @_;
    my ($ok, $details) = _cmp_details_test_data(UT, $table, $test_data);
    return Test::More::ok($ok, (defined $message ? $message : '').(defined $details ? "($details)" : ''));
}

# внутренняя проверка соответствия данных таблице
sub _cmp_details_test_data {
    my ($db, $table, $test_data) = @_;
    $test_data ||= [];

    # определяем, какие ключи нам нужно запрашивать
    my @keys = map { sql_quote_identifier($_) } uniq sort map {keys %$_} @$test_data;
    @keys = (1) if !@keys;
    my $real_rows = get_all_sql($db, "SELECT ".join(",", @keys)." FROM $table");
    my $test_bag = Test::Deep::bag(map {Test::Deep::superhashof($_)} @$test_data);
    my ($ok, $stack) = Test::Deep::cmp_details($real_rows, $test_bag);
    return $ok, ($ok ? undef : Test::Deep::deep_diag($stack));
}

=head2 init_test_dataset(\%db)

    Создать в тестовой базе определенные таблички и заполнить их тестовыми данными.
    На вход принимает один параметр - ссылку на хэш, где ключи это названия таблиц,
    а значения это ссылки на хеши с описанием таблиц и наборов данных/
    В хеше с описанием таблицы могут быть такие ключи:
      - original_db - база данных, из которой нужно импортировать структуру таблицы
      - create_string - ссылка на массив в формате второго аргумента create_table, описывающий структуру таблицы
      - no_check - булевское значение, нужно ли проверять строки таблицы в момент check_test_dataset
      - rows - тестовый набор данных:
          * массив хешей - для нешардированных баз будет записан в UT, для шардированных - в первый шард SHUT
          * хеш - ключ - номер шарда для записи, значение - массив хешей (собственно данные для вставки).
                на нешардированных базах - умирает. Номер шарда 'all' допустим, если нет других ключей
      - like - строка либо ссылка на хеш с описанием таблицы, по образцу которой надо создать новую. 
               Строка должна иметь вид "table" или "db.table", в хеше должно быть поле table и может быть поле db,

=cut
sub init_test_dataset {
    my $data = shift;
    for my $tbl (sort keys %$data) {
        my $tdata = $data->{$tbl};
        my $tname = $tdata->{name} || $tbl;

        # Если база выглядит и объявлена как шардированная
        my $sharded = _is_sharded($tdata->{original_db});

        foreach my $db (dbnames($tdata->{original_db} || UT)) {
            if (is_table_exists($db, $tname)) {
                _check_table_size($db, $tname);
                do_sql($db, "DROP TABLE $tname");
            }
        }

        # Не создаем таблицы под шардированные данные в нешардированных базах
        if (!$sharded && ref $tdata->{rows} eq 'HASH') {
            die q/Creating table for sharded data in not sharded database isn't allowed!/;
        }

        # создаём таблицу
        if ($tdata->{create_string}) {
            create_table($tdata->{original_db}, $tname, @{$tdata->{create_string}});
        } elsif ($tdata->{original_db}) {
            my $database;
            if ($sharded) {
                # таблички создатутся во всех шардах
                $database = $tdata->{original_db};
            } else {
                # таблички создатуться только в одной базе/шарде
                $database = ref $tdata->{original_db} eq 'HASH' && $tdata->{original_db}->{database} ? $tdata->{original_db}->{database} : $tdata->{original_db};
            }
            copy_table($database, $tname, like => $tdata->{like}, engine => $tdata->{engine});
        } else {
            die "I dont know how to create $tbl - neither create_string nor original_db is defined";
        }

        # вносим данные
        if ($tdata->{rows}) {
            replace_test_data($tdata->{original_db} || UT, $tname, $tdata->{rows});
        }
    }
    Test::More::note('database initialized with test dataset');
}

=head2 check_test_dataset(\%db, $message)

    Проверка таблиц тестовой БД на соответствие переданным наборам данных
    На входе два параметра:
      - ссылка на хэш, тот же, что и в init_test_dataset
      - сообщение для ok (опционально)
    Результат выполнения выводит в STDOUT в формате TAP

=cut
sub check_test_dataset {
    my ($db, $message) = @_;

    my ($global_ok, @global_details) = _check_test_dataset($db);

    return ( Test::More::ok($global_ok, $message || undef) or Test::More::note(map { "\n$_" } @global_details) );
}

sub _check_test_dataset {
    my $db = shift;

    my $global_ok = 1;
    my @global_details;

    for my $tbl (sort keys %$db) {
        my $tdata = $db->{$tbl};
        my $tname = $tdata->{name} || $tbl;
        if ($tdata->{no_check}) {
            next;
        }
        my $sharded = _is_sharded($tdata->{original_db} || UT);
        if (ref $tdata->{rows} eq 'HASH') {
            unless ($sharded) {
                # Если база нешардирована, а в dataset данные как для шардированной - то это точно не порядок
                $global_ok = 0;
                push @global_details, "Table $tbl: can't compare sharded data in not sharded database";
                next;
            }
            
            while (my ($shard, $rows) = each %{ $tdata->{rows} }) {
                if ($shard eq 'all') {
                    # Проверяем данные во всех шардах
                    foreach my $dbname (dbnames($tdata->{original_db})) {
                        $dbname =~ m/:(\d+)$/;
                        my $num = $1;
                        my ($ok, $details) = _cmp_details_test_data($dbname, $tname, $rows);
                        $global_ok &&= $ok;
                        push @global_details, "Table $tbl (shard $num): $details" if !$ok && defined $details;
                    }
                } else {
                    # Проверяем тот шард, что указан
                    my ($ok, $details) = _cmp_details_test_data(SHUT(shard => $shard), $tname, $rows);
                    $global_ok &&= $ok;
                    push @global_details, "Table $tbl (shard $shard): $details" if !$ok && defined $details;
                }
            }
        } else {
            # если база шардированая, а шард не указан - читаем из первого шарда ## expires: 01.01.2014
            my ($ok, $details) = _cmp_details_test_data($sharded ? SHUT_1 : UT, $tname, $tdata->{rows});
            $global_ok &&= $ok;
            push @global_details, "Table $tbl: $details" if !$ok && defined $details;
        }
    }

    return ($global_ok, @global_details);
}

# Проверяет, можно ли считать указанную $db шардированной
sub _is_sharded {
    my $db = shift;

    return 0 unless $db;

    return 1 if $db =~ m/^sharded_unit_tests(:\d+)?$/;
    return 1 if ref $db eq 'HASH' && $db->{database} =~ m/^sharded_unit_tests$/;

    if (defined $SHARDED_DB_RE) {
        die 'Marking database UT as sharded is not allowed' if $SHARDED_DB_RE =~ m/unit_tests/;

        return 1 if $db =~ $SHARDED_DB_RE;
        return 1 if ref $db eq 'HASH' && $db->{database} =~ $SHARDED_DB_RE;
    }

    return 0;
}

# Сохраняет информацию о созданных таблицах, для удаления при завершении работы
sub _save_table_info {
    my ($db, $table) = @_;

    foreach my $dbname (dbnames($db)) {
        $CREATED_TABLES{$$}->{$dbname}->{$table} = 1;
    }
}

1;
