package Yandex::Overshard;

=head1 NAME

    Yandex::Overshard - сахар для шардинга

=head1 DESCRIPTION

Группировка, аггрегирующие функции, сортировка, offset и limit на стороне perl

=head1 SYNOPSYS

 use Yandex::Overshard;

 my $camps = overshard group => ['cid'], count => 'bid_count', order => '-LastChange', 
    get_all_sql(PPC(cid => \@cids), [ 'select cid, 0 as bid_count, LastChange from banners', where => {
        cid => \@cids,
    } ]);

=cut

use strict;
use warnings;

use Carp;
use Params::Validate qw/validate/;
use JSON;
use List::Util qw/min/;
use List::MoreUtils qw/any uniq/;

use Yandex::ListUtils qw/xsort/;
use Yandex::Validate qw/is_valid_int/;
use Yandex::HashUtils qw/hash_copy hash_merge hash_map/;

use base qw/Exporter/;
our @EXPORT = qw/
    overshard
    enrich_data
/;

=head2 $GROUP_CONCAT_SEPARATOR

    Разделитель для функций group_concat, group_array, group_concat_distinct и group_array_distinct
    По-умолчанию - определен как запятая ","

    Если определен, то:
        group_concat - объединяет значения через $GROUP_CONCAT_SEPARATOR
        group_array, group_concat_distinct и group_array_distinct - дополнительно разделяют значения ключей по $GROUP_CONCAT_SEPARATOR
        group_concat_distinct - объединяет получившиеся уникальные значения через $GROUP_CONCAT_SEPARATOR
    Если не определен (undef), то:
        group_concat - объединяет значения через запятую ","
        group_array, group_concat_distinct и group_array_distinct - не производят дополнительных действий над получаемыми значениями
        group_concat_distinct - объединяет получившиеся уникальные значения через запятую ","

    Полезен, если нужен overshard над GROUP_CONCAT(DISTINCT key) для уменьшения числа строк/количества данных в результах sql-запроса.

    local $Yandex::Overshard::GROUP_CONCAT_SEPARATOR = undef;
    Чтобы отменить разделение значений в group_array, group_concat_distinct и group_array_distinct

=cut
our $GROUP_CONCAT_SEPARATOR //= ',';

=head2 %AGGREGATE

аггрегирующие функции
Вызываются для каждой записи с параметрами
  $acc - накопленное значение в группе
  $row - значение из текущей записи из селекта

=cut

my %AGGREGATE = (
    sum => sub { my ($acc, $row) = @_; return ($acc||0) + ($row||0) },
    min => sub { 
        my ($acc, $row) = @_; 
        return undef if (!defined $acc && !defined $row);
        return $acc if defined $acc && !defined $row;
        return $row if defined $row && !defined $acc;
        return $acc < $row ? $acc : $row 
    },
    max => sub { 
        my ($acc, $row) = @_; 
        return undef if (!defined $acc && !defined $row);
        return $acc if defined $acc && !defined $row;
        return $row if defined $row && !defined $acc;
        return $acc > $row ? $acc : $row 
    },
    minstr => sub {
        my ($acc, $row) = @_; 
        return undef if (!defined $acc && !defined $row);
        return $acc if defined $acc && !defined $row;
        return $row if defined $row && !defined $acc;
        return lc($acc) lt lc($row) ? $acc : $row 
    
    },
    maxstr => sub {
        my ($acc, $row) = @_; 
        return undef if (!defined $acc && !defined $row);
        return $acc if defined $acc && !defined $row;
        return $row if defined $row && !defined $acc;
        return lc($acc) gt lc($row) ? $acc : $row 
    },
    minstr_cs => sub {
        my ($acc, $row) = @_; 
        return undef if (!defined $acc && !defined $row);
        return $acc if defined $acc && !defined $row;
        return $row if defined $row && !defined $acc;
        return $acc lt $row ? $acc : $row 
    
    },
    maxstr_cs => sub {
        my ($acc, $row) = @_; 
        return undef if (!defined $acc && !defined $row);
        return $acc if defined $acc && !defined $row;
        return $row if defined $row && !defined $acc;
        return $acc gt $row ? $acc : $row 
    },
    count => sub { my ($acc, $row) = @_; return ( ($acc||0) + 1 ); },
    group_concat => sub {
        my ($acc, $row) = @_;
        return $row unless defined $acc;
        return join ((defined $GROUP_CONCAT_SEPARATOR ? $GROUP_CONCAT_SEPARATOR : ','), $acc, $row);
    },
    group_array => sub {
        my ($acc, $row) = @_;
        $acc ||= [];
        for (defined $GROUP_CONCAT_SEPARATOR ? (split $GROUP_CONCAT_SEPARATOR, $row) : $row) {
            push @$acc, $_;
        }
        return $acc;
    },
    # в первый проход - уникализируем значения через хеш
    group_concat_distinct => \&group_distinct,
    group_array_distinct => \&group_distinct,
    count_distinct => \&group_distinct,

    # для среднего значения - в первый проход считам сумму и количество, затем делим одно на другое (в group_by)
    avg => sub {
        my ($acc, $row) = @_;
        $acc->{sum} += $row;
        $acc->{cnt}++;
        return $acc;
    },
);

# типы, допустимые в order_by
our %ORDER_BY_TYPES = (
    'num' => sub {
        return defined $_[0] ? 0+$_[0] : 0;
    },
    'str_ci' => sub {
        return defined $_[0] ? lc($_[0]) : "";
    },
    'str_cs' => sub {
        return defined $_[0] ? $_[0] : "";
    },
);
$ORDER_BY_TYPES{str} = $ORDER_BY_TYPES{str_ci};

=head2 overshard

    $rows = overshard group => [qw/key1 key2'], sum => 'key3', order => 'key4,-key5', offset => '0', limit => '1', $rows;

Группировку (а также sum,min,max,order) можно указывать:
    group => 'field_name'
    group => [ qw/field1 field2/ ]
    group => 'field1, field2'

    distinct => 1

    having => sub {
        my $row = shift;
        return $row->{sum} > 0;
    }

Для сортировки дополнительно можно указать '-' перед именем, чтобы сортировать в обратном порядке:
    order => '-field'
    order => [ qw/field1, -field2/ ] # field asc, field2 desc
    order => 'field1, -field2'

=cut

sub overshard
{
    my $data = pop @_;
    my %opt = validate(@_, {
        group => 0, order => 0, offset => 0, limit => 0, distinct => 0,
        having => 0,
        map { $_ => 0 } keys %AGGREGATE,
    });
    for my $o (qw/group order/, keys %AGGREGATE) {
        $opt{$o} = _as_array($opt{$o}) if defined $opt{$o};
    }
    # distinct реализуем как частный случай группировки по всем полям
    # если группировка уже задана, то distinct игнорируем: записи уже уникальные
    if ($opt{distinct} && !$opt{group} && @$data ) {
        $opt{group} = [ keys %{$data->[0]} ];
    }
    if ($opt{group}) {
        $data = group_by($data, %opt);
    }
    elsif ($opt{having}) {
        croak "having without group does not makes sense";
    }
    if ($opt{order}) {
        $data = order_by($data, %opt);
    }
    if (defined $opt{limit}) {
        $data = limit_offset($data, %opt);
    }
    return $data;
}

=head2 group_by

    $rows = group_by($rows, group => 'fields', sum => '', min => '', max => '', ...)

Группировка и аггрегирующие функции

=cut

sub group_by
{
    my ($rows, %opt) = @_;
    my $group_by = $opt{group};
    if ($opt{having} && ref $opt{having} ne 'CODE') {
        croak 'having must be a coderef';
    }
    my @copy_keys;
    my $r0 = $rows->[0];
    if ($r0) {
        my %agg_keys = map { $_ => 1 } map { $opt{$_} ? @{$opt{$_}} : () } keys %AGGREGATE;
        # собираем все поля, по которым не делаем аггрегацию, для копирования в группу
        @copy_keys = grep { !$agg_keys{$_} } keys %$r0;
    }
    my $gdata = {};
    my $json = JSON->new;
    for my $row (@$rows) {
        my @ids = @{$row}{@$group_by};
        my $item = $gdata->{ $json->encode(\@ids) } ||= hash_copy {}, $row, @copy_keys;
        for my $agg (keys %AGGREGATE) {
            if ($opt{$agg}) {
                _group_aggregate($item, $row, $opt{$agg}, $AGGREGATE{$agg});
            }
        }
    }

    # второй проход при вычислении среднего значения или уникализации
    if (any { defined $opt{$_} } qw/avg group_concat_distinct group_array_distinct count_distinct/) {
        for my $row (values %$gdata) {
            for my $avg (@{$opt{avg}}) {
                # 0 здесь получиться не может, но на всякий случай проверяем
                $row->{$avg} = $row->{$avg}{cnt} == 0 ? undef : $row->{$avg}{sum} / $row->{$avg}{cnt};
            }
            for my $key (@{$opt{group_concat_distinct}}) {
                $row->{$key} = join ((defined $GROUP_CONCAT_SEPARATOR ? $GROUP_CONCAT_SEPARATOR : ','), keys %{ $row->{$key} });
            }
            for my $key (@{$opt{group_array_distinct}}) {
                $row->{$key} = [ keys %{ $row->{$key} } ];
            }
            for my $key (@{$opt{count_distinct}}) {
                $row->{$key} = scalar keys %{ $row->{$key} };
            }
        }
    }

    my $result = $opt{having} 
        ? [ grep { $opt{having}->($_) } values %$gdata ]
        : [ values %$gdata ]
        ;

    return $result;
}

=head2 _group_aggregate

Выполнить аггрегирующие функции над текущей записью из селекта и накопленным в группе значением

=cut

sub _group_aggregate
{
    my ($item, $row, $fields, $sub) = @_;
    for my $f (@$fields) {
        $item->{$f} = $sub->($item->{$f}, $row->{$f});
    }
}

=head2 order_by

Сортировка по указаным ключам
$rows - array of hashes
Именованые парамеры:
    order => 'key' | 'key1,key2' | [ 'key1', 'key2' ] - поля для сортировки
    Если перед именем поля идет знак '-' - сортировка производится в обратном порядке
    После имени поля может идти :<тип>, допустимые значения:
        :str - (по-умолчанию) алиас str_ci
        :num - сортировка принудительно производится в числовом порядке
        :str_ci - строки, без учёта регистра (case-insensitive)
        :str_cs - строки, с учётом регистра

Возвращает отсортированный массив

=cut

sub order_by
{
    my ($rows, %opt) = @_;

    my @order_by;
    for my $item (my @tmp = @{$opt{order}}) {
        my $desc = $item =~ s/^([-+])// && $1 eq '-' ? 1 : 0;
        my $type = $item =~ s/:(\w+)$// ? $1 : 'str';
        croak "Incorrect order_by type: $type" if !exists $ORDER_BY_TYPES{$type};
        push @order_by, {id => $item, desc => $desc, type => $type};
    }

    my $xsort_sub = sub {
        my $row = shift;
        my @ret;
        for my $ord (@order_by) {
            my $v = $row->{$ord->{id}};
            $v = $ORDER_BY_TYPES{$ord->{type}}->($v);
            push @ret, $ord->{desc} ? \$v : $v;
        }
        return @ret;
    };

    return [ xsort {$xsort_sub->($_)} @$rows ];
}

=head2 limit_offset

LIMIT и OFFSET
$rows - array of hashes
Именованые парамеры
    limit => Int
    offset => Int (необязательный)

=cut

sub limit_offset
{
    my ($rows, %opt) = @_;
    my $limit = $opt{limit};
    my $offset = $opt{offset};
    if (defined $limit) {
        croak "invalid limit value '$limit'" unless is_valid_int($limit, 0);
    }
    if (defined $offset) {
        croak "invalid offset value '$offset'" unless is_valid_int($offset, 0);
    }
    if ( defined $offset && ! defined $limit ) {
        croak "limit is required with offset";
    }

    $offset ||= 0;
    my $upper_bound = min($limit + $offset - 1, $#$rows);

    return [ @$rows[$offset .. $upper_bound] ];
}

=head2 enrich_data($rows, %opt, $code)

    my $rows = get_all_sql(PPC(...), [ "select c.cid, c.ManagerUID from campaigns c", where => { ... } ]);
    enrich_data $rows, 
        using => 'ManagerUID',
        map_fields => { fio => ManagerFIO, login => ManagerLogin },
        sub {
            my $manager_uids = shift;
            #get_user_data($manager_uids)
            get_hashes_hash_sql(PPC(uid => $manager_uids), ["SELECT uid as ManagerUID, fio as ManagerFIO ", where => { uid => SHARD_IDS }])
    };

Дополнить значения в $rows значениями, предоставленными $code, по ключу using.
По сути, эта функция выполняет left join между разными шардами

    select c.cid, c.ManagerUID, m.fio as ManagerFIO from campaigns c left join users m on c.ManagerUID = m.uid

Функция $code вызывается один раз, принимет в качестве параметра ссылку на массив, сформированный из $rows->[...]->{$using}, 
и должна вернуть ссылку на хеш, где ключи - это $rows->[...]->{$using}, а значения - хеши, которыми нужно дополнить 
записи в $rows.

Например:
$rows = [ { ManagerUID => 1 }, { ManagerUID => 2 } ]
$code([1,2]) => { 1 => { ManagerFIO => 'xxx' }, 2 => { ManagerFIO => 'yyy' } }
# результат:
$rows = [ { ManagerUID => 1, ManagerFIO => 'xxx' }, { ManagerUID => 2, ManagerFIO => 'yyy' } ]

map_fields указывает правило преобразования полей: если $code возвращает { 1 => { login => '...', uid => '...', other_fields => ... } }
а в map_fields написано { login => ManagerLogin, fio => ManagerFIO }, то login будет переименован в ManagerLogin (аналогично с uid),
а все прочие поля будут отброшены.

Функция модифицирует записи в $rows

Инвариант: $code может возвращать массив хешей, в таком случае в хешах должен быть ключ, указаный using

В случае, если $code (с учетом преобразований map_fields) возвращает конфликтующие записи (одинаково названное поле), будет
выброшено исключение.

=cut
sub enrich_data
{
    my $rows = shift;
    my $code = pop @_;
    my %opt = @_;
    my $key = delete $opt{using};
    my $map_fields = delete $opt{map_fields};

    if (keys %opt) {
        croak "unknown parameters: ".(join ",", keys %opt);
    }
    unless ($rows && ref $rows eq 'ARRAY') {
        croak "first param must be array of hashes";
    }
    unless (defined $key) {
        croak "'using' must be defined";
    }
    unless (defined $code && ref $code eq 'CODE') {
        croak "coderef not defined";
    }

    my $keys = [ uniq grep { defined } map { $_->{$key} } @$rows ];
    my $enrich = $code->($keys);
    my $map_fields_sub;
    if ($map_fields) {
        $map_fields_sub = sub {
            my $hash = shift;
            my %new_hash;
            while (my ($k, $v) = each %$map_fields) {
                $new_hash{$v} = $hash->{$k};
            }
            return \%new_hash;
        };
    }
    if (ref $enrich eq 'ARRAY') {
        $enrich = { map { $_->{$key} => ( $map_fields ? $map_fields_sub->($_) : $_) } @$enrich };
    }
    else {
        if ($map_fields) {
            $enrich = hash_map { $map_fields_sub->($_) } $enrich;
        }
    }
    my $null_hash = ( $map_fields ? { map { $_ => undef } values %$map_fields } : {} );
    for my $r (@$rows) {
        my $hash = $r->{$key} ? $enrich->{ $r->{$key} } : $null_hash;
        for my $k (keys %$hash) {
            croak "conflict on key '$k'" if exists $r->{$k} && $k ne $key;
            $r->{$k} = $hash->{$k};
        }
    }
    return $rows;
}

=head2 _as_array

Преобразовать строку, разделенную запятыми, в ссылку на массив (если параметр не является массивом)

=cut

sub _as_array
{
    my $val = shift;
    return $val if ref $val eq 'ARRAY';
    croak "only array ref or string allowed" if ref $val ne '';
    return [ split /\s*,\s*/, $val ];
}

=head3 group_distinct
    
    Набирает значения во внутренний хеш. Используется для организации GROUP_CONCAT(DISTINCT ...) с результатом в качестве строки или массива

=cut
sub group_distinct {
    my ($acc, $row) = @_;
    $acc ||= {};
    for (defined $GROUP_CONCAT_SEPARATOR ? (split $GROUP_CONCAT_SEPARATOR, ($row//'')) : $row) {
        $acc->{$_} = undef;
    }
    return $acc;
}

1;


